/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors.jmx.internal;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.JMException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;

import org.ow2.jasmine.probe.JasmineTarget;
import org.ow2.jasmine.probe.collectors.jmx.Metric;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionException;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionFactory;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;
import org.ow2.jasmine.probe.util.MetaData;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * An instance is associated to a Test JmxCollector. Its to role is to
 * collect values from MBean attributes.
 * @author danesa
 *
 */
public class JmxProxy {

    private Log logger = LogFactory.getLog(JmxProxy.class);

    private JmxConnectionService jmxConnectionService = null;

    private JmxConnectionFactory cfactory = null;

    private MBeanServerConnection cnx = null;

    private JasmineTarget target = null;

    private Map<String, String> targetProps = new HashMap<String, String>();

    /**
     * The target's JMX URL.
     */
    private String targetUrl;

    /**
     * The name of the managed server represented by the target.
     */
    private String serverName = null;

    /**
     * The name of the management domain to which belongs the target server.
     */
    private String domainName = null;

    /**
     * Pass to true if arrived to set server and domain names.
     * Avoids useless iterations.
     */
    private boolean serverNameSet = false;

    /**
     * Keeps last error message when trying to set server and domain name.
     * Avoids useless warnings.
     */
    private String error;

    public JmxProxy() {
    }

    public JmxConnectionService getJmxConnectionService() {
        return jmxConnectionService;
    }

    public void setJmxConnectionService(JmxConnectionService jmxConnectionService) {
        this.jmxConnectionService = jmxConnectionService;
    }

    public JasmineTarget getTarget() {
        return target;
    }

    public void setTarget(JasmineTarget target) {
        this.target = target;
        this.targetProps = target.getProperties();
        this.targetUrl = targetProps.get("url");
    }

    /**
     * Collect Metrics corresponding to the mbeans and a given names list to poll.
     * @param objname pattern for MBean ObjectNames
     * @param indicatorName the corresponding indicator name
     * @param names the list of names to poll
     * @return a list of Metrics, one Metric per MBean
     * @throws JmxConnectionException
     */
    protected List<Metric> collectMetrics(final ObjectName objname, final String indicatorName, final List<String> names) throws JmxConnectionException {

        LinkedList<Metric> metrics = new LinkedList<Metric>();
        try {
            MBeanServerConnection cnx = getMBeanServerConnection();
            // determine the set of mbeans to poll
            Set<ObjectName> onset = null;
            if (objname.isPattern()) {
                logger.debug("collect pattern:" + objname.toString());
                // Must loop on all the mbeans matching the pattern
                try {
                    onset = JmxUtil.getMBeans(cnx, objname);
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            } else {
                logger.debug("collect single mbean:" + objname.toString());
                try {
                    if (cnx.isRegistered(objname)) {
                        onset = new HashSet<ObjectName>();
                        onset.add(objname);
                    }
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            }
            for (ObjectName on : onset) {
                Metric m = collectMBeanMetric(cnx, on, indicatorName, names);
                if (m != null) {
                    setMultiValue(names, m);
                    metrics.add(m);
                }
            }
        } catch (JmxConnectionException e) {
            logger.warn("Cannot get Jmx access to target {0} with url {1} - {2}", target.getName(), targetUrl, e.getMessage());
            throw e;
        }

        return metrics;
    }

    /**
     * Collect Metric corresponding to an mbean and a list of names to poll.
     */
    private Metric collectMBeanMetric(final MBeanServerConnection cnx, final ObjectName objname, final String indicatorName, final List<String> names) throws JmxConnectionException {

        logger.debug("mbean:" + objname.toString());

        // The attributes to be used to construct the Metric
        AttributeList attl =  new AttributeList();

        Metric metric = new Metric();

        // get the time just before calling the appropriate getAttributes method
        long realTime = System.currentTimeMillis();

        if (!JmxUtil.hasAttributes(names)) {
            // no names provided by the indicator, poll all the attributes in the MBean
            try {
                attl = JmxUtil.getAttributes(cnx, objname);
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (JMException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        } else {
            try {
                attl = JmxUtil.getAttributesForProvidedNames(cnx, objname, names, indicatorName);
            } catch (IOException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            } catch (JMException e) {
                // TODO Auto-generated catch block
                e.printStackTrace();
            }
        }

        if (attl.isEmpty()) {
            logger.warn("No metrics found for indicator: " + indicatorName  + ", for mbean: " + objname.toString() + ", and target: " + target.getName());
            return null;
        } else {
            metric.setMBean(objname);
            metric.setAttributeList(attl);
            metric.setTarget(target.getName());
            metric.setTimestamp(realTime);
            // add metadata
            metric.setProperty(MetaData.SERVER, serverName);
            metric.setProperty(MetaData.DOMAIN, domainName);
            metric.setProperty(MetaData.URL, targetUrl);
            return metric;
        }
    }


    /**
     * Set multiValue to false if attr in collector specifies one attribute (or fragment) having a simple type
     * @param col the collector
     * @param m metric to check
     */
    private void setMultiValue(List<String> attrlist, Metric m) {
        if (JmxUtil.hasAttributes(attrlist) && (attrlist.size() == 1)) {// check the unique attr element's type
            Attribute att = (Attribute) m.getAttributeList().get(0);
            if (ComplexValuesUtil.hasSimpleType(att.getValue())) {
                m.setMultiValue(false);
            }
        }
    }

    /**
     * Get a MBeanServerConnection
     * @return MBeanServerConnection
     */
    private MBeanServerConnection getMBeanServerConnection() throws JmxConnectionException {
        if (cfactory == null) {
            cfactory = jmxConnectionService.getJmxConnectionFactory(targetUrl, targetProps);
        }
        MBeanServerConnection cnx = cfactory.getMBeanServerConnection();
        setServerAndDomain(cnx);
        return cnx;
    }

    protected void resetMBeanServerConnection() {
        jmxConnectionService.removeJmxConnectionFactory(targetUrl);
        cfactory = null;
    }

    private void setServerAndDomain(final MBeanServerConnection cnx) {
        if (serverNameSet) {
            return;
        }
        try {
            this.domainName = JmxCollectorService.DEFAULT_DOMAIN;
            this.serverName = JmxCollectorService.DEFAULT_SERVER;
            /**
             * Consider particular target cases:
             */
            /**
             * Java EE Server
             * --------------
             */
            ObjectName on = ObjectName.getInstance("*:j2eeType=J2EEServer,*");
            Set ons = cnx.queryNames(on, null);
            if (!ons.isEmpty()) {
                Iterator<?> onames = ons.iterator();
                ObjectName serverOn = (ObjectName) onames.next();
                this.domainName = (String) serverOn.getDomain();
                String name = serverOn.getKeyProperty("name");
                if (name != null) {
                    this.serverNameSet = true;
                    this.serverName = name;
                }
                return;
            }
            /**
             * Tomcat WEB Server
             */
            on = ObjectName.getInstance("Catalina:*");
            ons = cnx.queryNames(on, null);
            if (!ons.isEmpty()) {
                this.domainName = "Catalina";
                this.serverName = target.getName();
                this.serverNameSet = true;
                return;
            }
            // Jetty WEB Server
            on = ObjectName.getInstance("org.mortbay.jetty:type=server,*");
            ons = cnx.queryNames(on, null);
            if (!ons.isEmpty()) {
                this.domainName = "org.mortbay.jetty";
                Iterator<?> onames = ons.iterator();
                ObjectName serverOn = (ObjectName) onames.next();
                String serverId = serverOn.getKeyProperty("id");
                if (serverId != null) {
                    serverName = serverId;
                } else {
                    serverName = target.getName();
                }
                this.serverNameSet = true;
                return;
            }
        } catch (JMException e) {
            if (error == null) {
                error = e.toString();
                logger.warn("Cannot get server and domain names for target {0} ({1})", target.getName(), error);
            } else {
                if (!error.equals(e.toString())) {
                    // different error
                    error = e.toString();
                    logger.warn("Cannot get server and domain names for target {0} ({1})", target.getName(), error);
                }
            }
        } catch (IOException e) {
            if (error == null) {
                error = e.toString();
                logger.error("A communication problem occurred with target {0} ({1})", target.getName(), error);
            } else {
                if (!error.equals(e.toString())) {
                    // different error
                    error = e.toString();
                    logger.error("A communication problem occurred with target {0} ({1})", target.getName(), error);
                }
            }
        }

    }
}
