/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: LewysCollectorService.java 8543 2011-07-13 14:59:36Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.lewys.internal;

import org.apache.felix.ipojo.annotations.*;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;

/**
 * This CollectorService implements the "lewys" collector.
 * This is just a wrapper on a LeWYS probe.
 * It should be able to treat any lewys command, and any resource.
 * @author durieuxp
 */
@Component(name="LewysCollectorService")
@Provides
public class LewysCollectorService implements JasmineCollectorService {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * List of Collector
     */
    private HashMap<String, LewysCollector> collectorList = new HashMap<String, LewysCollector>();

    private Collection<JasminePropertyInfo> properties;

    /**
     * Supported OS
     */
    protected int osType;
    public final static int LINUX_2_4 = 10;
    public final static int LINUX_2_6 = 11;
    public final static int WINDOWS = 20;
    public final static int MAC_OS_X = 30;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="lewys")
    private String indicatorType;

    @Validate
    public void start() {
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo("cmd", "lewys command (disk, memory, network, ...)", true));
        properties.add(new JasminePropertyInfo("resources", "comma separated list of resources", false));

        // Check System version
        if (System.getProperty("os.name").equalsIgnoreCase("linux")) {
            if (System.getProperty("os.version").startsWith("2.6")) {
                osType = LINUX_2_6;
            } else {
                // Assumes that other versions will be as 2.4
                osType = LINUX_2_4;
            }
        }
        else if (System.getProperty("os.name").startsWith("Windows")) {
            osType = WINDOWS;
        }
        else if (System.getProperty("os.name").equals("Mac OS X")) {
            osType = MAC_OS_X;
        }
    }

    @Invalidate
    public void stop() {
        // TODO
    }

    // --------------------------------------------------------------------------------
    // JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Create a Collector for this JasmineIndicator
     * @param indicator
     * @param period in second
     * @param probeid
     * @return the Collector
     * @throws JasmineCollectorException operation failed
     */
    public synchronized JasmineCollector getCollector(JasmineIndicator indicator, int period, String probeid) throws JasmineCollectorException {
        // Build a key with the name and the probeId.
        String key = indicator.getName() + probeid;
        LewysCollector collector = collectorList.get(key);
        if (collector == null) {
            // sanity check, not really needed
            if (! indicator.getType().equals(indicatorType)) {
                logger.error("Bad type: " + indicator.getType());
                throw new JasmineCollectorException("Bad type: " + indicator.getType());
            }
            collector = new LewysCollector(key, indicator, period, osType);
            collectorList.put(key, collector);
        }
        return collector;
    }

    /**
     * Retrieve the description of all specific properties hidden in JasmineCollector
     * Return a Collection of all properties with their description
     * @return  a Collection of property descriptions
     */
    public Collection<JasminePropertyInfo> getPropertiesInfo() {
        return properties;
    }
}
