/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: LewysCollector.java 9019 2011-09-28 10:12:26Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.lewys.internal;

import org.objectweb.lewys.common.exceptions.NoResourceToProbeException;
import org.objectweb.lewys.common.exceptions.NoSuchResourceException;
import org.objectweb.lewys.common.exceptions.ProbeException;
import org.objectweb.lewys.probe.Probe;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.util.CsvConverter;

import java.util.*;

/**
 */
public class LewysCollector extends JCollector {

    /**
     * Lewys cmd: network, disk, memory, ...
     */
    protected String cmd;

    /**
     * Lewys probe used to collect results
     */
    protected Probe probe = null;

    /**
     * String Array of Resource names
     */
    protected String[] resources = null;

    /**
     * resourceIds
     */
    protected int[] rids = null;

    /**
     * Constructor
     * @param name Collector name
     * @param indicator definition
     * @param period in seconds
     * @param ostype Type of the underlying operating system
     */
    public LewysCollector(String name, JasmineIndicator indicator, int period, int ostype) {
        super(name, indicator, period);
        // specific part of JasmineIndicator (property list)
        Map<String, String> props = indicator.getProperties();
        this.cmd = props.get("cmd");

        // Create the lewys probe, depending on cmd and ostype
        if (cmd.equals("disk")) {
            try {
                switch (ostype) {
                    case LewysCollectorService.LINUX_2_4:
                        probe = new org.objectweb.lewys.probe.linux.DiskProbe();
                        break;
                    case LewysCollectorService.LINUX_2_6:
                        probe = new org.objectweb.lewys.probe.linux.Linux2_6DiskProbe();
                        break;
                    case LewysCollectorService.WINDOWS:
                        probe = new org.objectweb.lewys.probe.windows.DiskProbe();
                        break;
                    case LewysCollectorService.MAC_OS_X:
                        probe = new org.objectweb.lewys.probe.macosx.DiskProbe();
                        break;
                    default:
                        throw new RuntimeException("No DiskProbe available for " + System.getProperty("os.name"));
                }
            } catch (NoResourceToProbeException ex) {
                throw new RuntimeException("Can't set Disk probe", ex);
            }
        } else if (cmd.equals("memory")) {
            try {
                switch (ostype) {
                    case LewysCollectorService.LINUX_2_4:
                        probe = new org.objectweb.lewys.probe.linux.MemoryProbe();
                        break;
                    case LewysCollectorService.LINUX_2_6:
                        probe = new org.objectweb.lewys.probe.linux.Linux2_6MemoryProbe();
                        break;
                    case LewysCollectorService.WINDOWS:
                        probe = new org.objectweb.lewys.probe.windows.MemoryProbe();
                        break;
                    case LewysCollectorService.MAC_OS_X:
                        probe = new org.objectweb.lewys.probe.macosx.MemoryProbe();
                        break;
                    default:
                        throw new RuntimeException("No Memory probe available for " + System.getProperty("os.name"));
                }
            } catch (NoResourceToProbeException ex) {
                throw new RuntimeException("Can't set Memory probe", ex);
            }
        } else if (cmd.equals("network")) {
            try {
                switch (ostype) {
                    case LewysCollectorService.LINUX_2_4:
                    case LewysCollectorService.LINUX_2_6:
                        probe = new org.objectweb.lewys.probe.linux.NetworkProbe();
                        break;
                    case LewysCollectorService.WINDOWS:
                        probe = new org.objectweb.lewys.probe.windows.NetworkProbe();
                        break;
                    case LewysCollectorService.MAC_OS_X:
                        probe = new org.objectweb.lewys.probe.macosx.NetworkProbe();
                        break;
                    default:
                        throw new RuntimeException("No Network probe available for " + System.getProperty("os.name"));
                }
            } catch (NoResourceToProbeException ex) {
                throw new RuntimeException("Can't set Network probe", ex);
            }
        } else if (cmd.equals("cpu")) {
            try {
                switch (ostype) {
                    case LewysCollectorService.LINUX_2_4:
                    case LewysCollectorService.LINUX_2_6:
                        probe = new org.objectweb.lewys.probe.linux.CpuProbe();
                        break;
                    case LewysCollectorService.WINDOWS:
                        probe = new org.objectweb.lewys.probe.windows.CpuProbe();
                        break;
                    case LewysCollectorService.MAC_OS_X:
                        probe = new org.objectweb.lewys.probe.macosx.CpuProbe();
                        break;
                    default:
                        throw new RuntimeException("No Cpu probe available for " + System.getProperty("os.name"));
                }
            } catch (NoResourceToProbeException ex) {
                throw new RuntimeException("Can't set Cpu probe", ex);
            }
        } else if (cmd.equals("kernel")) {
            try {
                switch (ostype) {
                    case LewysCollectorService.LINUX_2_4:
                    case LewysCollectorService.LINUX_2_6:
                        // This doesn't work. Pb in Lewys code ?
                        probe = new org.objectweb.lewys.probe.linux.KernelProbe();
                        break;
                    case LewysCollectorService.WINDOWS:
                        probe = new org.objectweb.lewys.probe.windows.KernelProbe();
                        break;
                    default:
                        throw new RuntimeException("No Kernel probe available for " + System.getProperty("os.name"));
                }
            } catch (NoResourceToProbeException ex) {
                throw new RuntimeException("Can't set Kernel probe", ex);
            }
        } else {
            logger.warn("LeWYS command not supported: " + cmd);
        }

        // Get lewys resource ids
        // Assumes that each "underline" will be replaced by a "space" (See csv2array)
        resources = CsvConverter.csv2array(props.get("resources"));
        rids = new int[resources.length];
        for (int i = 0; i < resources.length; i++) {
            try {
                int id = probe.getResourceId(resources[i]);
                rids[i++] = id;
            } catch (NoSuchResourceException e) {
                logger.warn("This resource is not supported by LeWYS: " + resources[i]);
            }
        }
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * Retrieve the last results for this indicator
     * @return JasmineIndicatorValue
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("");

        // Get values from the LeWYS probe
        long[] values = null;
        try {
            values = probe.getValue(rids);
        } catch (ProbeException e) {
            logger.error("Cannot get Probe values: " + e);
            throw new JasmineCollectorException("Cannot get Probe values");
        } catch (NoSuchResourceException e) {
            logger.error("Cannot get Probe values: " + e);
            throw new JasmineCollectorException("Cannot get Probe values: Unknown Resource");
        }
        if (values == null) {
            return null;
        }

        // keep same timestamp for all pollings
        long time = System.currentTimeMillis();

        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(indicator.getName());
        jiv.addMetadata("cmd", cmd);

        // Set domain and server names in metadata
        // server is JmxCollectorService.META_SERVER;
        jiv.addMetadata("server", getServerName());
        // domain is JmxCollectorService.META_DOMAIN;
        jiv.addMetadata("domain", getDomainName());
        // Set target in JIV
        if (hostName != null) {
            jiv.setTarget(hostName);
        } else {
            jiv.setTarget(getServerName());
        }

        for (int i = 0; i < values.length; i++) {
            if (indicator.getScale() != 1) {
                values[i] = values[i] / indicator.getScale();
            }
            // Build a result
            JasmineSingleResult jsr = new JasmineSingleNumberResult();
            jsr.setName(resources[i]);
            jsr.setValue(values[i]);
            jsr.setTimestamp(time);
            jsr.addProperty("resource",  resources[i]);
            // Add this result to the JasmineIndicatorValue
            jiv.addValue(jsr);
        }
        return jiv;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("");
        // nothing to do
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("");
        // nothing to do
    }

    // ----------------------------------------------------------
    // Interface used by the CollectorService
    // ----------------------------------------------------------

    // ----------------------------------------------------------
    // Private methods
    // ----------------------------------------------------------

    /**
     * Build an indicator name with the original indicator name and the resource name
     * @param name
     * @param resource
     * @return
     */
    private String buildCompositeName(String name, String resource) {
        String ret = name + "." + resource.trim().replace(' ', '_');
        return ret;
    }
}