/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: LewysCollectorService.java 9792 2012-02-08 10:44:45Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.lewys.internal;

import java.util.ArrayList;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

/**
 * This CollectorService implements the "lewys" collector.
 * This is just a wrapper on a LeWYS probe.
 * It should be able to treat any lewys command, and any resource.
 * @author durieuxp
 */
@Component(name="LewysCollectorService")
@Provides
public class LewysCollectorService extends JCollectorService {
    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Get system resource parameters using the LeWYS probe system.";
    public final static String LONG_DESC =
            "Resource names should be exactly those defined in LeWYS.";

    /**
     * Sources
     */
    public final static String SOURCE_DESC =
            "No source indicator used.";
    public final static int SOURCE_MIN = 0;
    public final static int SOURCE_MAX = 0;

    /**
     * Properties
     */
    public final static String PROP_CMD = "cmd";
    public final static String PROP_CMD_DESC =
            "lewys command. One of (disk|memory|network|cpu|kernel)";
    public final static String PROP_RESOURCES = "resources";
    public final static String PROP_RESOURCES_DESC =
            "Comma separated list of resource names. When space characters are used in resource names, they can be replaced by '_'";


    /**
     * Supported OS
     */
    protected int osType;
    public final static int LINUX_2_4 = 10;
    public final static int LINUX_2_6 = 11;
    public final static int WINDOWS = 20;
    public final static int MAC_OS_X = 30;

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="lewys")
    private String indicatorType;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(0);
        collectorInfo.setSourceMax(0);
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_CMD, PROP_CMD_DESC, true));
        properties.add(new JasminePropertyInfo(PROP_RESOURCES, PROP_RESOURCES_DESC, false));
        collectorInfo.setPropertyInfos(properties);

        // Check System version
        if (System.getProperty("os.name").equalsIgnoreCase("linux")) {
            if (System.getProperty("os.version").startsWith("2.6")) {
                osType = LINUX_2_6;
            } else {
                // Assumes that other versions will be as 2.4
                osType = LINUX_2_4;
            }
        }
        else if (System.getProperty("os.name").startsWith("Windows")) {
            osType = WINDOWS;
        }
        else if (System.getProperty("os.name").equals("Mac OS X")) {
            osType = MAC_OS_X;
        }
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    // JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Create a Collector for this JasmineIndicator
     * @param indicator
     * @param probe
     * @return the Collector
     * @throws JasmineCollectorException operation failed
     */
    public synchronized JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe) throws JasmineCollectorException {
        JCollector collector = new LewysCollector(probe.getId(), indicator, probe.getPeriod(), osType);
        collector.setProbeManager(probeManager);
        addCollector(indicator.getName(), collector);
        return collector;
    }

}
