/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MergeCollectorService.java 8973 2011-09-23 12:31:17Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.merge.internal;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Manages a list of JCollector objects (in fact JMergeCollector instances).
 * @author durieuxp
 */
@Component(name="MergeCollectorService")
@Provides
public class MergeCollectorService extends JCollectorService {

    /**
     * Property names
     */
    public final static String PROP_OP = "op";
    public final static String PROP_SOURCE = "source";
    public final static String PROP_FIELD = "field";


    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="merge")
    private String indicatorType;

    @Validate
    public void start() {
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_OP, "merge operation (sum|average|max|min)", true));
        properties.add(new JasminePropertyInfo(PROP_SOURCE, "source indicator name", true));
        properties.add(new JasminePropertyInfo(PROP_FIELD, "field to be merged", true));
    }

    @Invalidate
    public void stop() {
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    //  JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Get a Collector for this indicator and this probe
     * @param indicator
     * @param period
     * @param probeid
     * @return the JasmineCollector
     * @throws org.ow2.jasmine.probe.collector.JasmineCollectorException
     */
    public JasmineCollector getCollector(JasmineIndicator indicator, int period, String probeid)
            throws JasmineCollectorException {
        String name = indicator.getName();
        JCollector collector = collectorList.get(name);
        if (collector == null) {

            // Get the collector corresponding to the source indicator
            Map<String, String> props = indicator.getProperties();
            String sourceIndicatorName = props.get(PROP_SOURCE);
            JasmineCollector sourceCollector = null;
            try {
                sourceCollector = probeManager.getCollector(sourceIndicatorName, period, probeid);
            } catch (JasmineProbeException e) {
                logger.error("Cannot get value: " + e);
                throw new JasmineCollectorException(e.getMessage());
            }

            // Check the operation
            String operation = props.get(PROP_OP);
            if (operation == null) {
                String err = "Operation not defined in MergeCollectorService";
                logger.error(err);
                throw new JasmineCollectorException(err);
            }


            // Create the MergeCollector
            String tomerge = props.get(PROP_FIELD);
            collector = new JMergeCollector(name, indicator, period, operation, sourceCollector, tomerge);
            collector.setProbeManager(probeManager);
            collectorList.put(name, collector);
        }
        return collector;
    }

    /**
     * return the List of indicators used by this indicator.
     * @param indic JasmineIndicator
     * @return
     */
    @Override
    public List<String> getDependantIndicators(JasmineIndicator indic) {
        List<String> ret = new ArrayList<String>();
        Map<String, String> props = indic.getProperties();
        String sourceIndicatorName = props.get(PROP_SOURCE);
        ret.add(sourceIndicatorName);
        return ret;
    }
}
