/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MergeCollectorService.java 9792 2012-02-08 10:44:45Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.merge.internal;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Manages a list of JCollector objects (in fact JMergeCollector instances).
 * @author durieuxp
 */
@Component(name="MergeCollectorService")
@Provides
public class MergeCollectorService extends JCollectorService {

    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Merge source indicator's values";
    public final static String LONG_DESC =
            "The definition requires the merge operation and a source indicator";

    /**
     * Sources
     */
    public final static String SOURCE_DESC =
            "A source indicator must be provided. It should be multi-valued";
    public final static int SOURCE_MIN = 1;
    public final static int SOURCE_MAX = 1;

    /**
     * Properties
     */
    public final static String PROP_OP = "op";
    public final static String PROP_OP_DESC =
            "merge operation. Must be one of : min|max|average|sum";
    public final static String PROP_FIELD = "field";
    public final static String PROP_FIELD_DESC =
        "The field to be merged. It aims to filter out some non relevant values from the merge operation.";



    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="merge")
    private String indicatorType;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(SOURCE_MIN);
        collectorInfo.setSourceMax(SOURCE_MAX);
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_OP, PROP_OP_DESC, true));
        // TODO add support for merge on field
        //properties.add(new JasminePropertyInfo(PROP_FIELD, PROP_FIELD_DESC, false));
        collectorInfo.setPropertyInfos(properties);
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    //  JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Get a Collector for this indicator and this probe.
     * First parse indicator properties, and get a collector for each source.
     * The goal is to 'notify' to sources that an additional 'consumer' collector exists for them.
     * This allows for collectors that manage a cache of values to update cache size.
     * @param indicator the indicator for which values are collected by the collector
     * @param probe the probe using the indicator
     * @return the JasmineCollector instance created to collect values for the given indicator
     * within the given probe.
     * @throws JasmineCollectorException The collector could not be created.
     */
    public JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe)
            throws JasmineCollectorException {

        // Get the collector corresponding to the source indicator
        Map<String, String> props = indicator.getProperties();

        // Check the operation
        String operation = props.get(PROP_OP);
        if (operation == null) {
            String err = "Operation not defined in MergeCollectorService";
            logger.error(err);
            throw new JasmineCollectorException(err);
        }

        if (indicator.getSources().isEmpty()) {
            logger.error("Merge indicator without source defined: " + indicator.getName());
            throw new JasmineCollectorException("Merge indicator without source defined: " + indicator.getName());
        }
        String sourceIndicatorName = indicator.getSources().get(0);
        JasmineCollector sourceCollector = null;
        try {
            sourceCollector = probeManager.getCollector(sourceIndicatorName, probe.getId());
        } catch (JasmineProbeException e) {
            logger.error("Cannot get value: " + e);
            throw new JasmineCollectorException(e.getMessage());
        }

        // Create the MergeCollector
        String tomerge = props.get(PROP_FIELD);
        JCollector collector = new JMergeCollector(probe.getId(), indicator, probe.getPeriod(), operation, sourceCollector, tomerge);
        collector.setProbeManager(probeManager);
        addCollector(indicator.getName(), collector);
        return collector;
    }

    /**
     * return the List of indicators used by this indicator.
     * @param indic JasmineIndicator
     * @return
     */
    @Override
    public List<String> getDependantIndicators(JasmineIndicator indic) {
        return indic.getSources();
    }

}
