/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SlopeCollector.java 8859 2011-09-13 14:31:16Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.slope.internal;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;

/**
 * @author durieuxp
 */
public class SlopeCollector extends JCollector {

    private JasmineCollector numerator;
    private JasmineCollector denominator;

    private Number numprevious = null;
    private Number denprevious = null;

    /**
     * Constructor
     * @param key
     * @param indic
     * @param period
     * @param n
     * @param d
     */
    public SlopeCollector(String key, JasmineIndicator indic, int period, JasmineCollector n, JasmineCollector d) {
        super(key, indic, period);
        this.numerator = n;
        this.denominator = d;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue
     * @throws JasmineCollectorException
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("");
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        JasmineSingleResult jsr = new JasmineSingleNumberResult();
        long timestamp = System.currentTimeMillis();
        Number result = 0;
        Number numval = null;
        Number denval = null;

        // get source values
        JasmineIndicatorValue jivnum = numerator.getLastResult();
        if (jivnum == null) {
            logger.warn("No result available on source indicator for numerator");
            return null;
        }
        JasmineIndicatorValue jivden = denominator.getLastResult();
        if (jivden == null) {
            logger.warn("No result available on source indicator for denominator");
            return null;
        }
        if (jivnum.getValues().size() > 1 || jivden.getValues().size() > 1) {
            logger.warn("Slope on multivalued indicator not supported");
            throw new JasmineCollectorException("Slope on multivalued indicator not supported");
        }

        for (JasmineSingleResult jsr1 : jivnum.getValues()) {
            // only 1 value per JasmineIndicatorValue
            numval = (Number) jsr1.getValue();
            break;
        }
        for (JasmineSingleResult jsr1 : jivden.getValues()) {
            // only 1 value per JasmineIndicatorValue
            denval = (Number) jsr1.getValue();
            break;
        }

        // Check that we already have previous values
        if (numprevious  == null || denprevious == null) {
            logger.debug("First call: No previous value yet");
            numprevious = numval;
            denprevious = denval;
            return null;
        }

        // compute slope
        Number diff1 = diffValues(numval, numprevious);
        Number diff2 = diffValues(denval, denprevious);
        result = divideValues(diff1, diff2);
        if (indicator.getScale() != 1) {
            result =  divideValues(result, indicator.getScale());
        }
        jsr.setTimestamp(timestamp);
        jsr.setValue(result);
        jsr.setName(indicator.getName());
        jiv.setName(indicator.getName());
        jiv.addValue(jsr);
        return jiv;
    }

    /**
     * Stop polling
     */
    public void stopPolling() {
        logger.debug("");
        // Nothing to do
    }

    /**
     * Start polling
     */
    @Override
    public void startPolling() {
        logger.debug("");
        // nothing to do
    }

    // ------------------------------------------------------------------------------------
    // private methods
    // ------------------------------------------------------------------------------------

}