/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SlopeCollectorService.java 8859 2011-09-13 14:31:16Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.slope.internal;

import org.apache.felix.ipojo.annotations.*;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.*;

/**
 * @author durieuxp
 */
@Component(name="SlopeCollectorService")
@Provides
public class SlopeCollectorService implements JasmineCollectorService {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * List of collectors
     */
    private Map<String, SlopeCollector> collectorList = new HashMap<String, SlopeCollector>();

    /**
     * Info about Properties
     */
    private List<JasminePropertyInfo> properties;

    public final static String PROP_NUMERATOR = "numerator";
    public final static String PROP_DENOMINATOR = "denominator";

    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name="indicator.type", value="slope")
    private String indicatorType;

    @Validate
    public void start() {
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        properties.add(new JasminePropertyInfo(PROP_NUMERATOR, "source indicator used as numerator", true));
        properties.add(new JasminePropertyInfo(PROP_DENOMINATOR, "source indicator used as denominator", true));
    }

    @Invalidate
    public void stop() {
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    //  JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Get a Collector for this indicator and this probe
     * @param indicator
     * @param period
     * @param probeid
     * @return the JasmineCollector
     * @throws JasmineCollectorException
     */
    public JasmineCollector getCollector(JasmineIndicator indicator, int period, String probeid)
            throws JasmineCollectorException {

        // Build a unique key from the probeid and the indicator name
        String key = probeid + ":" + indicator.getName();

        SlopeCollector collector = collectorList.get(key);
        if (collector == null) {
            // sanity check, not really needed.
            if (! indicator.getType().equals(indicatorType)) {
                logger.error("Bad type: " + indicator.getType());
                throw new JasmineCollectorException("Bad type: " + indicator.getType());
            }

            Map<String, String> props = indicator.getProperties();

            // Get the collectors corresponding to the source indicators
            JasmineCollector numerator = null;
            String iname = props.get(PROP_NUMERATOR);
            if (iname == null) {
                String err = "No numerator defined in " + key;
                logger.error(err);
                throw new JasmineCollectorException(err);
            }
            try {
                numerator = probeManager.getCollector(iname, period, probeid);
            } catch (JasmineProbeException e) {
                logger.error("Cannot get collector for numerator: " + e);
                throw new JasmineCollectorException(e.getMessage());
            }

            JasmineCollector denominator = null;
            iname = props.get(PROP_DENOMINATOR);
            if (iname == null) {
                String err = "No numerator defined in " + key;
                logger.error(err);
                throw new JasmineCollectorException(err);
            }
            try {
                denominator = probeManager.getCollector(iname, period, probeid);
            } catch (JasmineProbeException e) {
                logger.error("Cannot get collector for denominator: " + e);
                throw new JasmineCollectorException(e.getMessage());
            }

            // Create the Collector
            collector = new SlopeCollector(key, indicator, period, numerator, denominator);
            collectorList.put(key, collector);
        }
        return collector;
    }

    /**
     * Retrieve the description of all specific properties hidden in JasmineOutput
     * Return a Map of all properties with their description
     * @return  a Map of property descriptions
     */
    public List<JasminePropertyInfo> getPropertiesInfo() {
        return properties;
    }


}
