/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SlopeCollector.java 9680 2012-01-27 12:49:56Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors.slope.internal;

import java.util.HashMap;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineSingleNumberResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;

/**
 * @author durieuxp
 */
public class SlopeCollector extends JCollector {

    private JasmineCollector numerator;
    private JasmineCollector denominator;

    private Number numprevious = null;
    private Number denprevious = null;

    /**
     * Constructor
     * @param probeId
     * @param indic
     * @param period
     * @param n
     * @param d
     */
    public SlopeCollector(String probeId, JasmineIndicator indic, int period, JasmineCollector n, JasmineCollector d) {
        super(probeId, indic, period);
        this.numerator = n;
        this.denominator = d;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue
     * @throws JasmineCollectorException
     */
    @Override
    public JasmineIndicatorValue getLastResult() throws JasmineCollectorException {
        logger.debug("");
        JasmineIndicatorValue jiv = new JasmineIndicatorValue();
        jiv.setName(indicator.getName());

        // get source indicator values
        JasmineIndicatorValue jivnum = numerator.getLastResult();
        if (jivnum == null || jivnum.getValues().isEmpty()) {
            logger.warn("No result available on source indicator for numerator {0}", numerator.getIndicatorName());
            return null;
        }
        JasmineIndicatorValue jivden = denominator.getLastResult();
        if (jivden == null || jivden.getValues().isEmpty()) {
            logger.warn("No result available on source indicator for denominator {0}", denominator.getIndicatorName());
            return null;
        }
        if (jivnum.getValues().size() > 1 || jivden.getValues().size() > 1) {
            logger.warn("Slope on multivalued indicator not supported");
            throw new JasmineCollectorException("Slope on multivalued indicator not supported");
        }

        // Compute target
        String tnum = jivnum.getTarget();
        String tden = jivden.getTarget();
        if (tnum != null) {
            if (tden != null) {
                if (tnum.equals(tden)) {
                    jiv.setTarget(tden);
                } else {
                    jiv.setTarget("misc");
                }
            } else {
                jiv.setTarget(tden);
            }
        } else {
            if (tden != null) {
                jiv.setTarget(tden);
            }
        }

        // compute domain
        String domain = null;
        String d = jivnum.getMetadata().get("domain");
        if (d != null) {
            domain = d;
        }
        d = jivden.getMetadata().get("domain");
        if (d != null) {
            if (domain == null) {
                domain = d;
            } else if (! d.equals(domain)) {
                domain = "misc";
            }
        }
        jiv.addMetadata("domain", domain);

        // compute server
        String server = null;
        String s = jivnum.getMetadata().get("server");
        if (s != null) {
            server = s;
        }
        s = jivden.getMetadata().get("server");
        if (s != null) {
            if (server == null) {
                server = s;
            } else if (! s.equals(server)) {
                server = "misc";
            }
        }
        jiv.addMetadata("server", server);

        // Properties for result JSR
        HashMap<String, String> jsrProps = new HashMap<String, String>();

        JasmineSingleResult jsr = new JasmineSingleNumberResult();
        Number result = 0;
        Number numval = null;
        Number denval = null;
        String numResultName = null;
        String denResultName = null;

        for (JasmineSingleResult jsr1 : jivnum.getValues()) {
            // only 1 value per JasmineIndicatorValue
            numval = (Number) jsr1.getValue();
            numResultName = jsr1.getName();
            jsrProps.putAll(jsr1.getProperties());
            break;
        }
        for (JasmineSingleResult jsr1 : jivden.getValues()) {
            // only 1 value per JasmineIndicatorValue
            denval = (Number) jsr1.getValue();
            denResultName = jsr1.getName();
            jsrProps.putAll(jsr1.getProperties());
            break;
        }

        if (denval == null || numval == null) {
            // Source result misses, can't calculate new result
            // missing result for indicator
            String missingResult = " ";
            boolean nullNumval = false;
            if (numval == null) {
                missingResult = missingResult + jivnum.getName();
                nullNumval = true;
            }
            if (denval == null) {
                if (nullNumval) {
                    missingResult = missingResult + ", ";
                }
                missingResult = missingResult + jivden.getName();
            }
            String errorMess = "Missing source result for indicator " + missingResult;
            throw new JasmineCollectorException(errorMess);
        }

        // Check that we already have previous values
        if (numprevious  == null || denprevious == null) {
            logger.debug("First call: No previous value yet");
            numprevious = numval;
            denprevious = denval;
            return null;
        }

        // compute slope
        Number diff1 = diffValues(numval, numprevious);
        Number diff2 = diffValues(denval, denprevious);
        result = divideValues(diff1, diff2);
        if (indicator.getScale() != 1) {
            result =  divideValues(result, indicator.getScale());
        }
        long numtimestamp = jivnum.getTimestamp();
        long dentimestamp = jivden.getTimestamp();
        long timestamp;
        if (numtimestamp > dentimestamp) {
            timestamp = numtimestamp;
        } else {
            timestamp = dentimestamp;
        }
        jsr.setTimestamp(timestamp);
        jsr.setValue(result);
        // No name: just use the indicator name without extension.
        jsr.setName("");
        jsr.setProperties(jsrProps);
        jiv.addValue(jsr);
        // Only one numeric simple value corresponds to an aggregate indicator
        jiv.setMultiValue(false);
        // Only one numeric simple value corresponds to a slope indicator
        jiv.setMultiValue(false);
        return jiv;
    }

    /**
     * Stop polling. Called when probe is stopped.
     */
    public void stopPolling() {
        logger.debug("Stop " + indicator.getName() + " in probe " + probeId);
        numerator.stopPolling();
        denominator.stopPolling();
    }

    /**
     * Start polling. Called when probe is started.
     */
    @Override
    public void startPolling() {
        logger.debug("Start " + indicator.getName() + " in probe " + probeId);
        numerator.startPolling();
        denominator.startPolling();
    }

    // ------------------------------------------------------------------------------------
    // private methods
    // ------------------------------------------------------------------------------------

}