/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.collectors.slope.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.ServiceProperty;
import org.apache.felix.ipojo.annotations.Validate;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.collectors.JCollectorService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;

/**
 * @author durieuxp
 */
@Component(name = "SlopeCollectorService")
@Provides
public class SlopeCollectorService extends JCollectorService {

    /**
     * Description
     */
    public final static String SHORT_DESC =
            "Calculate slope value from two source values.";
    public final static String LONG_DESC =
            "Two source indicators must be provided (s1 and s2)."
          + "\nThe order is important, since the result will be: delta(s1) / delta(s2), where delta(s) = current(s) - previous(s)."
          + "\nEach source indicator's value should be composed of only one result (current limitation). The result must be a Number.";
    /**
     * Sources
     */
    public final static String SOURCE_DESC =
            "First source is the numerator value, second source is the denominator value.";
    public final static int SOURCE_MIN = 2;
    public final static int SOURCE_MAX = 2;


    // --------------------------------------------------------------------------------
    // ipojo management
    // --------------------------------------------------------------------------------

    @ServiceProperty(name = "indicator.type", value = "slope")
    private String indicatorType;

    @Validate
    public void start() {
        // Build the CollectorInfo
        collectorInfo = new JasmineCollectorInfo();
        collectorInfo.setType(indicatorType);
        collectorInfo.setShortDesc(SHORT_DESC);
        collectorInfo.setLongDesc(LONG_DESC);
        collectorInfo.setSourceDesc(SOURCE_DESC);
        collectorInfo.setSourceMin(SOURCE_MIN);
        collectorInfo.setSourceMax(SOURCE_MAX);
        // init the list of properties
        properties = new ArrayList<JasminePropertyInfo>();
        collectorInfo.setPropertyInfos(properties);
    }

    @Invalidate
    public void stop() {
        // Mark as removed the collectors corresponding to indicatorType
        removeCollectors(null, null);
    }

    @Requires
    ProbeManager probeManager = null;

    // --------------------------------------------------------------------------------
    //  JasmineCollectorService implementation
    // --------------------------------------------------------------------------------

    /**
     * Get a Collector for this indicator and this probe.
     * First parse indicator properties, and get a collector for the numerator and denominator indicators.
     * The goal is to 'notify' these collectors that an additional 'consumer' collector exists for them.
     * This allows for collectors that manage a cache of values to update cache size.
     *
     * @param indicator the indicator for which values are collected by the collector
     * @param probe     the probe using the indicator
     * @return the JasmineCollector instance created to collect values for the given indicator
     *         within the given probe.
     * @throws JasmineCollectorException The collector could not be created.
     */
    public JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe)
            throws JasmineCollectorException {

        // sanity check, not really needed.
        if (!indicator.getType().equals(indicatorType)) {
            logger.error("Bad type: " + indicator.getType());
            throw new JasmineCollectorException("Bad type: " + indicator.getType());
        }


        // Get the collectors corresponding to the source indicators
        JasmineCollector numerator = null;
        if (indicator.getSources().size() < 2) {
            logger.error("Slope indicator without 2 sources defined: " + indicator.getName());
            throw new JasmineCollectorException("Slope indicator without 2 sources defined: " + indicator.getName());
        }
        String iname = indicator.getSources().get(0);
        if (iname == null) {
            String err = "No numerator defined in " + indicator.getName();
            logger.error(err);
            throw new JasmineCollectorException(err);
        }
        try {
            numerator = probeManager.getCollector(iname, probe.getId());
        } catch (JasmineProbeException e) {
            logger.error("Cannot get collector for numerator: " + e);
            throw new JasmineCollectorException(e.getMessage());
        }

        JasmineCollector denominator = null;
        iname = indicator.getSources().get(1);
        if (iname == null) {
            String err = "No numerator defined in " + indicator.getName();
            logger.error(err);
            throw new JasmineCollectorException(err);
        }
        try {
            denominator = probeManager.getCollector(iname, probe.getId());
        } catch (JasmineProbeException e) {
            logger.error("Cannot get collector for denominator: " + e);
            throw new JasmineCollectorException(e.getMessage());
        }

        // Create the Collector
        JCollector collector = new SlopeCollector(probe.getId(), indicator, probe.getPeriod(), numerator, denominator);
        addCollector(indicator.getName(), collector);

        return collector;
    }

    /**
     * return the List of indicators used by this indicator.
     *
     * @param indic JasmineIndicator
     * @return
     */
    @Override
    public List<String> getDependantIndicators(JasmineIndicator indic) {
        return indic.getSources();
    }

}
