package org.ow2.jasmine.probe.jmxconnection.simple;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.Map;

import javax.management.MBeanServerConnection;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.ow2.jasmine.probe.jmxconnection.JmxConnectionException;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionFactory;
import org.ow2.jasmine.probe.jmxconnection.JmxConnectionService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

public class JmxConnectionFactoryImpl implements JmxConnectionFactory {

    private JMXConnector connector = null;

    private String url = null;

    /**
     * User name to use when connecting to JMX.
     */
    private String user = null;

    /**
     * Password to use when connecting to JMX.
     */
    private String password = null;

    /**
     * Protocol provider list to use when connecting to JMX.
     */
    private String protocolProviders = null;

    private static Log logger = LogFactory.getLog(JmxConnectionService.class);

    /**
     * Constructor
     * @param url
     * @param props
     */
    public JmxConnectionFactoryImpl(String url, Map<String, String> props) throws JmxConnectionException {

        logger.debug(url);

        // Check URL
        this.url = url;
        JMXServiceURL serviceUrl = null;
        try {
            serviceUrl = new JMXServiceURL(url);
        } catch (MalformedURLException e) {
            throw new JmxConnectionException(e);
        }

        // Get properties
        if (props != null) {
            this.user = props.get(JmxConnectionServiceImpl.PROP_USER);
            this.password = props.get(JmxConnectionServiceImpl.PROP_PASSWORD);
            this.protocolProviders = props.get(JmxConnectionServiceImpl.PROP_PROTOCOL_PROVIDER_PACKAGES);
        }

        // Create a connector client for the connector server at the given serviceUrl
        ClassLoader old = Thread.currentThread().getContextClassLoader();
        try {
            Thread.currentThread().setContextClassLoader(this.getClass().getClassLoader());

            Map<String, Object> env = null;
            if (this.user != null && this.password != null) {
                if (env == null) {
                    env = new HashMap<String, Object>(2);
                }
                String[] creds = {this.user, this.password};
                env.put(JMXConnector.CREDENTIALS, creds);
            }
            if (this.protocolProviders != null) {
                if (env == null) {
                    env = new HashMap<String, Object>(2);
                }
                env.put(JMXConnectorFactory.PROTOCOL_PROVIDER_PACKAGES, this.protocolProviders);
            }

            // Workaround for WebLogic bug that does not use the CREDENTIALS key with t3!
            JMXServiceURL jmxUrl = new JMXServiceURL(this.url);
            if ("t3".equals(jmxUrl.getProtocol())) {
                env = setCredentialsForT3(env);
            }
            // END of workaround for WebLogic

            connector = JMXConnectorFactory.connect(serviceUrl, env);
            logger.info("Connector created for target {0}.", url);
        } catch (SecurityException e) {
            logger.info("SecurityException raised for target {0}.", url);
            throw new JmxConnectionException(e);
        } catch (IOException e) {
            // The connector client cannot be made or connection
            // cannot be established because of a communication problem,
            //e.printStackTrace();
            logger.info("IOException raised when trying to connect to target {0}: {1}", url, e.getMessage());
            throw new JmxConnectionException(e);
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }

    }

    public void close() throws JmxConnectionException {
        // remove the Connector
        try {
            connector.close();
            logger.info("JMX connection closed for target {0}", url);
        } catch (IOException e) {
            logger.warn("Exception when trying to close JMX connector for target {0} ({1})", url, e.toString());
            throw new JmxConnectionException(e);
        }
    }

    public MBeanServerConnection getMBeanServerConnection() throws JmxConnectionException {
        MBeanServerConnection mbsc = null;
        try {
            mbsc = connector.getMBeanServerConnection();
        } catch (IOException e) {
            logger.info("IOException raised when trying to get connection to target {0}: {1}", url, e.getMessage());
            throw new JmxConnectionException(e);
        }
        return mbsc;
    }

    /**
     * Workarround for the t3 weblogic protocol. t3 uses
     * - java.naming.security.principal (Context.SECURITY_PRINCIPAL)
     * - java.naming.security.credentials (Context.SECURITY_CREDENTIALS)
     * whareas the standard expects:
     * - jmx.remote.credentials (JMXConnector.CREDENTIALS)
     * This method translates the "standard" credentials  to that expected by t3
     * @param map
     * @return either the same or an updated map
     */
    @SuppressWarnings("unchecked")
    private Map setCredentialsForT3(Map map) {
        if (!map.containsKey("java.naming.security.principal")
                && !map.containsKey("java.naming.security.credentials")) {
            if (map.containsKey("jmx.remote.credentials")) {
                Map newMap = new HashMap();
                newMap.putAll(map);
                String[] cred = (String[]) map.get("jmx.remote.credentials");
                newMap.put("java.naming.security.principal", cred[0]);
                newMap.put("java.naming.security.credentials", cred[1]);
                return newMap;
            }
        }
        return map;
    }
}
