/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JProbeManagerImpl.java 8682 2011-07-26 13:23:18Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.manager.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Unmarshaller;

import org.apache.felix.ipojo.annotations.*;
import org.osgi.framework.ServiceReference;
import org.ow2.jasmine.probe.*;
import org.ow2.jasmine.probe.collector.JasmineAggregateService;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.manager.JProbe;
import org.ow2.jasmine.probe.outer.JasmineOuter;
import org.ow2.jasmine.probe.outer.JasmineOuterException;
import org.ow2.jasmine.probe.outer.JasmineOuterService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.jasmine.probe.probemanager.generated.*;
import org.ow2.jasmine.probe.probescheduler.SchedulerException;
import org.ow2.jasmine.probe.probescheduler.SchedulerService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implements the main service of the JasmineProbe module exposing the JasmineProbeManager interface
 * to the JasmineProbe module's clients.
 *
 * @author durieuxp
 */
@Component(name = "JasmineProbeManagerService")
@Provides
public class JProbeManagerImpl implements JasmineProbeManager, ProbeManager {

    protected static Log logger = LogFactory.getLog(JProbe.class);

    private ProbeConfig probeConfig = null;

    /**
     * JasmineProbe definitions
     */
    private Map<String, JasmineProbe> probes = new HashMap<String, JasmineProbe>();

    /**
     * JasmineOutput definitions
     */
    private Map<String, JasmineOutput> outputs = new HashMap<String, JasmineOutput>();

    /**
     * JasmineIndicator definitions
     */
    private Map<String, JasmineIndicator> indicators = new HashMap<String, JasmineIndicator>();

    /**
     * JasmineTarget definitions
     */
    private Map<String, JasmineTarget> targets = new HashMap<String, JasmineTarget>();

    /**
     * running probes
     */
    private Map<String, JProbe> runnings = new HashMap<String, JProbe>();

    /**
     * List of the available Outer services.
     */
    private Map<String, JasmineOuterService> outerservices = new HashMap<String, JasmineOuterService>();

    /**
     * List of the available Collector services.
     */
    private Map<String, JasmineCollectorService> collectorservices = new HashMap<String, JasmineCollectorService>();

    /**
     * List of the custom available Aggregate services.
     */
    private Map<String, JasmineAggregateService> aggregateservices = new HashMap<String, JasmineAggregateService>();

    /**
     * Listeners for Probe changes
     */
    private List<JasmineProbeListener> probeListeners = new ArrayList<JasmineProbeListener>();

    // -----------------------------------------------------------------------------------
    // ipojo management
    // -----------------------------------------------------------------------------------

    @Validate
    public void start() {
        logger.debug("JasmineProbeManager service activated.");
        try {
            readConfig();

        } catch (Exception e) {
            // Cannot read configuration
            logger.error("Cannot read configuration");
        }

    }

    @Invalidate
    public void stop() {
        logger.debug("JasmineProbeManager service stopped.");
    }

    @Requires
    protected SchedulerService jasmineProbeScheduler = null;

    @Bind(aggregate = true, optional = true)
    protected void bindOuterService(JasmineOuterService jos, ServiceReference sr) {
        String type = (String) sr.getProperty("output.type");
        if (type == null) {
            logger.warn("JasmineOuterService with undefined output.type");
            return;
        }
        JasmineOuterService old = outerservices.get(type);
        if (old != null) {
            logger.warn("JasmineOuterService already defined for " + type);
            return;
        }
        outerservices.put(type, jos);
        logger.info("{0} JasmineOuterService bound into JasmineProbe.", type);

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindOuterService(JasmineOuterService jos, ServiceReference sr) {
        // TODO
    }

    @Bind(aggregate = true, optional = true)
    protected void bindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        String type = (String) sr.getProperty("indicator.type");
        if (type == null) {
            logger.warn("JasmineCollectorService with undefined indicator.type");
            return;
        }
        JasmineCollectorService old = collectorservices.get(type);
        if (old != null) {
            logger.warn("JasmineCollectorService already defined for " + type);
            return;
        }
        collectorservices.put(type, jcs);
        logger.info("{0} JasmineCollectorService bound into JasmineProbe.", type);

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        // TODO
    }

    @Bind(aggregate = true, optional = true)
    protected void bindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("JasmineAggregateService with undefined aggregate.type binded");
            return;
        }
        JasmineAggregateService old = aggregateservices.get(type);
        if (old != null) {
            // A service for aggregate function already exists, use it if its not un-deployed.
            // Another approach could be to replace the old function by a new one.
            logger.warn("AggregateService already defined for " + type + " function. Please check deployment plan and undeploy old service !");
            return;
        }
        aggregateservices.put(type, jas);
        logger.info("{0} JasmineAggregateService bound into JasmineProbe.", type);
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("JasmineAggregateService with undefined aggregate.type unbinded");
            return;
        }
        if (aggregateservices.containsKey(type)) {
            aggregateservices.remove(type);
            logger.info("{0} JasmineAggregateService unbound into JasmineProbe.", type);
        }
    }

    // -----------------------------------------------------------------------------------
    //  JasmineProbeManager implementation
    // -----------------------------------------------------------------------------------

    /**
     * Create a new probe defined by its data
     *
     * @param probe probe description
     * @return the Id to be used to reference this probe later.
     */
    public synchronized String createProbe(JasmineProbe probe) throws JasmineProbeException {
        // Generate an Id if not supplied
        String id = probe.getId();
        if (id == null) {
            id = getNewProbeId();
            probe.setId(id);
        }
        // TODO Check if Id already known
        logger.debug("creating probe : " + id);

        probes.put(id, probe);
        return id;
    }

    /**
     * Change a Probe with new parameters
     *
     * @param probe probe description
     */
    public synchronized void changeProbe(JasmineProbe probe) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("changeProbe not implemented");
    }

    /**
     * Remove a probe.
     *
     * @param probeId probe identifier.
     */
    public synchronized void removeProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probes.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot remove probe with unexistent id " + probeId);
        }

        // Check if probe running
        JProbe running = runnings.get(probeId);
        if (running != null) {
            switch (running.getStatus()) {
                case JasmineProbe.PROBE_RUNNING:
                    logger.warn("Cannot remove probe running: " + probeId);
                    return;
                case JasmineProbe.PROBE_STARTED:
                    logger.warn("Cannot remove probe running: " + probeId);
                    return;
                case JasmineProbe.PROBE_FAILED:
                    logger.debug("removing a failed probe: " + probeId);
                    break;
                default:
                    // TODO
                    break;
            }
            runnings.remove(probeId);
        }
        logger.debug("removing probe " + probeId);
        probes.remove(probeId);
    }

    /**
     * Remove all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void removeAllProbes() throws JasmineProbeException {
        for (JasmineProbe probe : probes.values()) {
            try {
                removeProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not remove probe " + probe.getId() + " :" + e);
            }
        }
    }

    /**
     * @return all the probe definitions
     */
    public synchronized Collection<JasmineProbe> getProbes() {
        logger.debug("");
        return probes.values();
    }

    /**
     * Start a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be started.
     */
    public synchronized void startProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probes.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with unexistent id " + probeId);
        }

        // Check if already running
        JProbe running = runnings.get(probeId);
        if (running != null) {
            switch (running.getStatus()) {
                case JasmineProbe.PROBE_RUNNING:
                    logger.info("Probe already running: " + probeId);
                    return;
                case JasmineProbe.PROBE_STARTED:
                    logger.info("Probe already started: " + probeId);
                    return;
                case JasmineProbe.PROBE_FAILED:
                    logger.info("restarting a failed probe: " + probeId);
                    // TODO
                    break;
                default:
                    // TODO
                    break;
            }
        } else {
            // Create the JProbe object
            running = createRunningProbe(def);
            runnings.put(probeId, running);
        }

        // Start the scheduler
        try {
            // period must be given in milliseconds
            long ms = def.getPeriod() * 1000L;
            running.setTask(jasmineProbeScheduler.schedulePeriodicTask(null, running, 0, ms));
        } catch (SchedulerException e) {
            logger.error("Error when trying to start probe : {0}: {1}", probeId, e);
            setProbeState(running, JasmineProbe.PROBE_FAILED, e.getMessage());
            throw new JasmineProbeException("Cannot schedule task ");
        }

    }

    /**
     * Stop a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be stopped.
     */
    public synchronized void stopProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probes.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with unexistent id " + probeId);
        }

        // Check if already running
        JProbe running = runnings.get(probeId);
        if (running == null) {
            throw new JasmineProbeException("This probe is not running: " +  probeId);
        }

        // Stop the Probe
        running.halt();
    }

    /**
     * Start all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void startAllProbes() throws JasmineProbeException {
        for (JasmineProbe probe : probes.values()) {
            try {
                startProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not start probe " + probe.getId() + " :" + e);
            }
        }
    }

    /**
     * Stop all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void stopAllProbes() throws JasmineProbeException {
        for (JasmineProbe probe : probes.values()) {
            try {
                stopProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not stop probe " + probe.getId() + " :" + e);
            }
        }
    }

    /**
     * Register a new Output in the ProbeManager
     * A name is associated to the Output definition.
     * This Output will be usable later in probes.
     *
     * @param newoutput output description
     * @return the Id to be used to reference this output later.
     */
    public synchronized String createOutput(JasmineOutput newoutput) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newoutput.getName();
        logger.debug("Create Output " + id);

        // Check if already known
        JasmineOutput output = outputs.get(id);
        if (output != null) {
            logger.debug("output already known: " + id);
            if (output.equals(newoutput)) {
                // If same output already created, just return.
                return id;
            }
            if (isOutputUsed(output)) {
                logger.warn("oldoutput:" + output);
                logger.warn("newoutput:" + newoutput);
                throw new JasmineProbeException(id + " output already known with another definition");
            }
            // remove old definition and replace by the new one.
            outputs.remove(id);
        }
        outputs.put(id, newoutput);
        return id;
    }

    /**
     * Change an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     *
     * @param output output description
     * @throws JasmineProbeException is thrown if the output does not exist,
     *                               or if it is used by a running probe.
     */
    public void changeOutput(JasmineOutput output) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("changeOutput not implemented");
    }

    /**
     * remove an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     *
     * @param name ident of the output to remove
     * @throws JasmineProbeException is thrown if the output does not exist,
     *                               or if it is used by a running probe.
     */
    public void removeOutput(String name) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("removeOutput not implemented");
    }

    /**
     * get the list of all defined outputs.
     *
     * @return List of the outputs
     */
    public synchronized Collection<JasmineOutput> getOutputs() {
        logger.debug("");
        return outputs.values();
    }

    /**
     * get a given output's definition
     *
     * @param name the output's name
     * @return the output's definition
     */
    public synchronized JasmineOutput getOutput(String name) throws JasmineProbeException {
        logger.debug("");
        return outputs.get(name);
    }

    /**
     * get the list of possible properties for Output, depending of its type.
     *
     * @param type type of the Output (console, file, ...)
     * @return Collection of JasminePropertyInfo
     */
    public Collection<JasminePropertyInfo> getOutputPropertyInfos(String type) {
        // Find the appropriate OuterService
        JasmineOuterService jos = outerservices.get(type);
        if (jos == null) {
            logger.error("No OuterService found for type " + type);
            return null;
        }
        Collection<JasminePropertyInfo> ret = jos.getPropertiesInfo();
        return ret;
    }

    /**
     * Create a new Indicator
     *
     * @param newindic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    public synchronized String createIndicator(JasmineIndicator newindic) throws JasmineProbeException {
        // Id chosen by the user.
        // TODO: generate Id if not provided ?
        String id = newindic.getName();
        logger.debug("Create Indicator " + id);

        // Check if already known
        JasmineIndicator indic = indicators.get(id);
        if (indic != null) {
            logger.debug("indicator already known: " + id);
            if (indic.equals(newindic)) {
                // If same indicator already created, just return.
                return id;
            }
            if (isIndicatorUsed(indic)) {
                logger.warn("old indic:" + indic);
                logger.warn("new indic:" + newindic);
                throw new JasmineProbeException(id + "Indicator already known with another definition");
            }
            // remove old definition and replace by the new one.
            indicators.remove(id);
        }
        indicators.put(id, newindic);
        return id;
    }

    /**
     * Change an indicator. It must exist, otherwise an exception is thrown.
     * The indicator must not be used by a probe, otherwise an exception is thrown.
     *
     * @param indic indicator description
     * @throws JasmineProbeException is thrown if the indicator does not exist,
     *                               or if it is used by a running probe.
     */
    public void changeIndicator(JasmineIndicator indic) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("changeIndicator not implemented");
    }

    /**
     * remove an indicator
     *
     * @param name ident of the indicator to remove
     */
    public void removeIndicator(String name) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("removeIndicator not implemented");
    }

    /**
     * get the list of all defined indicators.
     *
     * @return List of the indicators
     */
    public synchronized Collection<JasmineIndicator> getIndicators() {
        logger.debug("");
        return indicators.values();
    }

    /**
     * get the list of possible properties for Indicator, depending of its type.
     *
     * @param type type of the Indicator (jmx, lewys, ...)
     * @return Collection of JasminePropertyInfo
     */
    public Collection<JasminePropertyInfo> getIndicatorPropertyInfos(String type) {
        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(type);
        if (jcs == null) {
            logger.error("No CollectorService found for type " + type);
            return null;
        }
        Collection<JasminePropertyInfo> ret = jcs.getPropertiesInfo();
        return ret;
    }

    /**
     * Create a new target
     *
     * @param newtarget target description
     * @return the Id to be used to reference this target later.
     */
    public synchronized String createTarget(JasmineTarget newtarget) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newtarget.getName();
        logger.debug("Create Target " + id);

        // Check if already known
        JasmineTarget target = targets.get(id);
        if (target != null) {
            logger.debug("target already known: " + id);
            if (target.equals(newtarget)) {
                // If same target already created, just return.
                return id;
            }
            if (isTargetUsed(target)) {
                logger.warn("oldtarget:" + target);
                logger.warn("newtarget:" + newtarget);
                throw new JasmineProbeException(id + "Target already known with another definition");
            }
            // remove old definition and replace by the new one.
            targets.remove(id);
        }
        targets.put(id, newtarget);
        return id;
    }

    /**
     * Change a target. It must exist, otherwise an exception is thrown.
     *
     * @param newtarget new target description
     * @throws JasmineProbeException is thrown if the target does not exist.
     */
    public void changeTarget(JasmineTarget newtarget) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("changeTarget not implemented");
    }

    /**
     * remove a target
     *
     * @param name ident of the target to remove
     */
    public void removeTarget(String name) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("removeTarget not implemented");
    }

    /**
     * get the list of all defined targets.
     *
     * @return List of the targets
     */
    public synchronized Collection<JasmineTarget> getTargets() {
        logger.debug("");
        return targets.values();
    }

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     *
     * @param listener object that treats the probe state change
     */
    public synchronized void addProbeListener(JasmineProbeListener listener) {
        logger.debug("");
        probeListeners.add(listener);
    }

    /**
     * Save the current configuration in the specified xml file
     *
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws JasmineProbeException
     */
    public void saveConfig(String path) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("saveConfig not implemented");
    }

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     *
     * @param path the xml file, or null if load the default configuration file.
     * @throws JasmineProbeException
     */
    public void loadConfig(String path) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("loadConfig not implemented");
    }

    // -----------------------------------------------------------------
    // ProbeManager implementation
    // -----------------------------------------------------------------

    /**
     * Get a Collector by its Indicator name
     * @param indicatorName indicator name
     * @param period the period of the probe using the indicator
     * @param probeid
     * @return The collector corresponding to an indicator.
     */
    public synchronized JasmineCollector getCollector(String indicatorName, int period, String probeid) throws JasmineProbeException {

        JasmineIndicator indicator = indicators.get(indicatorName);
        if (indicator == null) {
            throw new JasmineProbeException("Cannot return collector for unexistent indicator " + indicatorName);
        }

        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(indicator.getType());
        if (jcs == null) {
            logger.error("No CollectorService found for {0} having type {1}", indicatorName, indicator.getType());
            throw new JasmineProbeException("No CollectorService found for " + indicatorName);
        }
        // Get the Collector from the CollectorService
        JasmineCollector collector = null;
        try {
            collector = jcs.getCollector(indicator, period, probeid);
        } catch (JasmineCollectorException e) {
            logger.error("Could not get the Collector for " + indicatorName + ": " + e);
            throw new JasmineProbeException("Could not get the Collector for " + indicatorName);
        }
        return collector;
    }

    /**
     * @return If AggregateServices exist, return the functions these services implement (the values of aggregate.type properties).
     * Else, return null.
     */
    public synchronized List<String> getAggregateFuntions() {
        if (!aggregateservices.isEmpty()) {
            List<String> functions = new ArrayList<String>();
            Set<String> keySet = aggregateservices.keySet();
            for (String function : keySet) {
                functions.add(function);
            }
            return functions;
        }
        return null;
    }

    /**
     * Return the AggregateService for a given aggregate function
     * @param key the aggregate function that is searched
     * @return the AggregateService having aggregate.type equal to key. Else, return null.
     */
    public synchronized JasmineAggregateService getAggregate(String key) {
        if (aggregateservices.containsKey(key)) {
            return aggregateservices.get(key);
        }
        return null;
    }

    /**
     * Get a JasmineTarget by its name
     * @param name
     * @return JasmineTarget definition
     */
    public JasmineTarget getTarget(String name) {
        JasmineTarget target = targets.get(name);
        return target;
    }

    // --------------------------------------------------------------------------------------
    // Private methods
    // --------------------------------------------------------------------------------------

    /**
     * Generate an unique identifier
     */
    private String getNewProbeId() {
        // TODO
        return "1";
    }

    /**
     * Read configuration.
     */
    private void readConfig() throws Exception {
        // Retrieve the probe config file
        String configurationFile = "probe-config.xml";
        ClassLoader loader = ClassLoader.getSystemClassLoader();
        InputStream resource = null;
        try {
            resource = loader.getResourceAsStream(configurationFile);
        } catch (Exception e) {
            logger.error("Cannot find probe-config.xml: " + e);
            throw e;
        }
        if (resource == null) {
            logger.error("Cannot find probe-config.xml");
            return;
        }
        logger.debug("Parsing probe-config.xml");
        loadConfigFile(resource);
    }

    private synchronized void loadConfigFile(InputStream resource) throws JasmineProbeException {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            probeConfig = (ProbeConfig) unmarshaller.unmarshal(resource);

            // Create JasmineOutput objects
            for (Output output : probeConfig.getOutput()) {
                JasmineOutput jo = new JasmineOutput();
                jo.setName(output.getId());
                jo.setType(output.getType().value());
                Map<String, String> props = new HashMap<String, String>();
                for (Paramvalue param : output.getParam()) {
                    props.put(param.getName(), param.getValue());
                }
                jo.setProperties(props);
                createOutput(jo);
            }

            // Create JasmineTarget objects
            for (Target target : probeConfig.getTarget()) {
                JasmineTarget jt = new JasmineTarget();
                jt.setName(target.getId());
                jt.setUrl(target.getUrl());
                Map<String, String> props = new HashMap<String, String>();
                for (Paramvalue param : target.getParam()) {
                    props.put(param.getName(), param.getValue());
                }
                jt.setProperties(props);
                createTarget(jt);
            }

            // Create JasmineIndicator objects
            for (Indicator indic : probeConfig.getIndicator()) {
                JasmineIndicator ji = new JasmineIndicator();
                ji.setName(indic.getId());
                ji.setType(indic.getType().value());
                ji.setScale(indic.getScale().intValue());
                Map<String, String> props = new HashMap<String, String>();
                for (Paramvalue param : indic.getParam()) {
                    props.put(param.getName(), param.getValue());
                }
                ji.setProperties(props);
                createIndicator(ji);
            }

            // Create JasmineProbe objects
            for (Probe probe : probeConfig.getProbe()) {
                JasmineProbe jp = new JasmineProbe();
                jp.setId(probe.getId());
                jp.setPeriod(probe.getPeriod().intValue());
                jp.setOutputList(probe.getOutput());
                jp.setIndicatorList(probe.getIndicator());
                if (probe.getStatus() != null && probe.getStatus().value().equals("started")) {
                    jp.setStatus(JasmineProbe.PROBE_TOSTART);
                }
                createProbe(jp);
            }

        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Error in probe-config.xml: " + e);
            throw new JasmineProbeException("Error in probe-config.xml: " + e);
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    /**
     * Test if an Output is used at least in one ProbeInstance.
     *
     * @param output the output to check
     * @return true if output used in at least 1 probe.
     */
    private synchronized boolean isOutputUsed(JasmineOutput output) {
        for (JasmineProbe probe : probes.values()) {
            if (probe.getOutputList().contains(output)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Test if a Target is used
     *
     * @param target the target to check
     * @return true if target used
     */
    private boolean isTargetUsed(JasmineTarget target) {
        // TODO
        return false;
    }

    /**
     * Test if an Indicator is used at least in one ProbeInstance.
     *
     * @param indic the indicator to check
     * @return true if indic used in at least 1 probe.
     */
    private synchronized boolean isIndicatorUsed(JasmineIndicator indic) {
        for (JasmineProbe probe : probes.values()) {
            if (probe.getOutputList().contains(indic)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Notify a state change for that probe
     *
     * @param running probe
     * @param state   See JasmineProbe.state
     * @param message Error message
     * @throws JasmineProbeException
     */
    private synchronized void setProbeState(JProbe running, int state, String message) {
        logger.debug("state =" + state);
        running.setStatus(state);
        running.setError(message);
        for (JasmineProbeListener listener : probeListeners) {
            listener.notifyEvent(running.getProbeDef());
        }
    }

    /**
     * Start probes waiting to be started.
     */
    private synchronized void startWaitingProbes() {

        for (JasmineProbe probe : probes.values()) {
            if (probe.getStatus() == JasmineProbe.PROBE_TOSTART) {
                logger.debug(probe.getId() + " trying to start");
                JProbe running = null;
                try {
                    running = createRunningProbe(probe);
                } catch (JasmineProbeException e) {
                    logger.warn("Cannot start probe now: " + probe.getId());
                    continue;
                }
                runnings.put(probe.getId(), running);

                // Start the scheduler
                try {
                    // period must be given in milliseconds
                    long ms = probe.getPeriod() * 1000L;
                    running.setTask(jasmineProbeScheduler.schedulePeriodicTask(null, running, 0, ms));
                    logger.debug(probe.getId() + " should be started now");
                } catch (SchedulerException e) {
                    logger.error("Error when trying to start probe : {0}: {1}", probe.getId(), e);
                    setProbeState(running, JasmineProbe.PROBE_FAILED, e.getMessage());
                }
            }
        }
    }

    /**
     * Create a Running Probe
     * @param def probe definition
     * @return the JProbe object fully initialized
     * @throws JasmineProbeException
     */
    private JProbe createRunningProbe(JasmineProbe def) throws JasmineProbeException {
        JProbe running = new JProbe(def);

        // Build outer list.
        List<JasmineOuter> outers = new ArrayList<JasmineOuter>();
        for (String outputname : def.getOutputList()) {
            JasmineOutput output = outputs.get(outputname);
            if (output == null) {
                logger.error("unknown output : " + outputname);
                throw new JasmineProbeException("unknown output : " + outputname);
            }
            // Find the appropriate OuterService
            JasmineOuterService jos = outerservices.get(output.getType());
            if (jos == null) {
                logger.error("No OuterService found for {0} having type {1}", outputname, output.getType());
                throw new JasmineProbeException("No OuterService found for " + outputname);
            }
            // Get the Outer from the OuterService
            JasmineOuter jo = null;
            try {
                jo = jos.getOuter(output);
            } catch (JasmineOuterException e) {
                logger.error("Could not get the Outer for " + outputname + ": " + e);
                throw new JasmineProbeException("Could not get the Outer for " + outputname);
            }
            outers.add(jo);
        }
        running.setOuters(outers);

        // Build collector list
        List<JasmineCollector> collectors = new ArrayList<JasmineCollector>();
        for (String indicname : def.getIndicatorList()) {
            JasmineIndicator indic = indicators.get(indicname);
            if (indic == null) {
                logger.error("unknown indicator : " + indicname);
                throw new JasmineProbeException("unknown indicator : " + indicname);
            }
            // Find the appropriate CollectorService
            JasmineCollectorService jcs = collectorservices.get(indic.getType());
            if (jcs == null) {
                logger.error("No CollectorService found for {0} having type {1}", indicname, indic.getType());
                throw new JasmineProbeException("No CollectorService found for " + indicname);
            }
            // Get the Collector from the CollectorService
            JasmineCollector coll = null;
            try {
                coll = jcs.getCollector(indic, def.getPeriod(), def.getId());
            } catch (JasmineCollectorException e) {
                logger.error("Could not get the Collector for " + indicname + ": " + e);
                throw new JasmineProbeException("Could not get the Collector for " + indicname);
            }
            collectors.add(coll);
        }
        running.setCollectors(collectors);

        // Set status to STARTED only if running is OK.
        def.setStatus(JasmineProbe.PROBE_STARTED);

        return running;
    }
}
