/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JOuter.java 8479 2011-07-01 14:40:17Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.outers;

import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbeResult;
import org.ow2.jasmine.probe.outer.JasmineOuter;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.text.SimpleDateFormat;
import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * Outer instance.
 * this abstract class should be derived in different forms,
 * depending of the type of output:
 * console, file, mule, etc...
 * These implementations are in separate modules.
 * @author durieuxp
 */
public abstract class JOuter extends Thread implements JasmineOuter {

    /**
     * logger used for outers
     */
    protected Log logger = LogFactory.getLog(JOuter.class);

    /**
     * Queue of JasmineProbeResult elements to be sent
     */
    private LinkedList<JasmineProbeResult> dataqueue = new LinkedList<JasmineProbeResult>();

    /**
     * CSV separator.
     */
    protected static final String SEPARATOR = ";";

    private boolean valid = false;
    private boolean waiting = false;
    private long outerTimeout = 10 * 1000;

    /**
     * Std format for the date.
     */
    protected SimpleDateFormat dateformat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    abstract public void processData(JasmineProbeResult result);

    public JOuter(JasmineOutput output) {
        super("JasmineOuter-" + output.getName());
        valid = true;
    }

    /**
     * run method for Thread implementation.
     */
    public void run() {
        while (valid) {
            JasmineProbeResult data = null;
            synchronized (this) {
                if (dataqueue.isEmpty()) {
                    try {
                        waiting = true;
                        wait(outerTimeout);
                        waiting = false;
                    } catch (InterruptedException e) {
                        logger.warn(getName() + ": outer interrupted", e);
                    } catch (Exception e) {
                        logger.warn(getName() + ": outer exception", e);
                    }
                }
                try {
                    data = dataqueue.removeFirst();
                } catch (NoSuchElementException e) {
                    data = null;
                }
            }
            if (data != null) {
                processData(data);
            }
        }
    }

    /**
     * stop the AbstractOuter
     */
    public synchronized void stopIt() {
        valid = false;
        notify();
    }

    // ------------------------------------------------------------------------------
    // JasmineOuter implementation
    // ------------------------------------------------------------------------------

    /**
     * Publish data on this Output
     * @param result data to publish
     */
    public synchronized void publishData(JasmineProbeResult result) {
        logger.debug("");
        dataqueue.addLast(result);
        if (waiting) {
            notify();
        }
    }

}
