/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineProbeCommand.java 8491 2011-07-05 13:09:56Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.ParameterException;
import org.apache.commons.beanutils.PropertyUtils;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasminePropertyInfo;

import java.io.PrintStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * Common code for all JasmineProbe Commands.
 * Contains ProbeManager reference and command line parsing.
 * @author danesa
 */
public abstract class JasmineProbeCommand {


    public JasmineProbeCommand() {
        // In case there are no parameters
        params = new JasmineProbeCommandParams();
    }

    public abstract String getName();

    /**
     * Contains the command parameters set by JCommander.
     */
    private JasmineProbeCommandParams params = null;

    public void setParams(JasmineProbeCommandParams params) {
        this.params = params;
    }

    public JasmineProbeCommandParams getParams() {
        return params;
    }

    /**
     * Called by all the commands that have parameters
     * @param line
     * @param out
     * @param err
     * @return
     */
    protected boolean parseOptions(final String line, PrintStream out, PrintStream err) {

        // Extract command name to get options
        String options = line.substring(getName().length()).trim();

        if (options.length() == 0) {
            err.println(getName() + " command: options missing");
            // un-set params object in order to avoid using it
            return false;
        }

        // Parse parameters
        StringTokenizer st = new StringTokenizer(options);

        /**
         * Contains the command line arguments.
         */
        String[] argv = new String[st.countTokens()];
        int i = 0;
        while (st.hasMoreTokens()) {
            argv[i++] = st.nextToken();
        }

        JCommander commander = null;
        try {
            commander = new JCommander(params, argv);
        } catch (ParameterException pe) {
            err.println(pe.getMessage());
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    /**
     * Construct a properties Map, where the keys are property names provided by propsInfo and the values of the
     * properties should be read from the command parameters
     * @param propsInfo contains the properties description
     * @param params contains the command parameters
     * @return the resulting properties map
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    protected Map<String, String> parseProperties(Collection<JasminePropertyInfo> propsInfo, JasmineProbeCommandParams params) throws JasmineProbeException {
        Map<String, String> properties = new HashMap<String, String>();
        for (JasminePropertyInfo propInfo : propsInfo) {
            // get property name
            String key = propInfo.getName();
            // get property value from the params object
            String value = null;
            try {
                value = (String) PropertyUtils.getSimpleProperty(params, key);
            } catch (Exception e) {
                throw new JasmineProbeException(getName() + " command: cannot get options for " + key + " property (" + e.toString() + ")");
            }
            properties.put(key, value);
            //System.out.println("- " + key + " = " + value);
        }
        return properties;
    }

}
