/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JCollector.java 8847 2011-09-13 08:09:38Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.*;

/**
 * This object represents a Collector in use.
 * It may be associated to more than one JProbe
 * It is abstract, because implementation may be various.
 * These implementations are in separate modules.
 * @author durieuxp
 */
public abstract class JCollector implements JasmineCollector {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * Indicator definition
     */
    protected JasmineIndicator indicator;

    /**
     * period of polling in seconds
     */
    protected int period;

    /**
     * Name used to identify this Collector
     */
    protected String name;

    /**
     * Constructor
     */
    public JCollector(String name, JasmineIndicator def, int period) {
        this.name = name;
        this.indicator = def;
        this.period = period;
    }

    public JasmineIndicator getIndicator() {
        return indicator;
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * @return name of this Collector
     */
    public String getName() {
        return name;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue or null if no value available
     * @throws JasmineCollectorException
     */
    public abstract JasmineIndicatorValue getLastResult() throws JasmineCollectorException;

    /**
     * Stop polling
     */
    public abstract void stopPolling();

    /**
     * Restart polling
     */
    public abstract void startPolling();

    // ------------------------------------------------------------------------
    // Utility methods
    // ------------------------------------------------------------------------

    /**
     * Make a diff between 2 values
     * @param v1
     * @param v2
     * @return v1 - v2
     */
    protected long diffValues(Number v1, Number v2) throws JasmineCollectorException {
        return (longValue(v1) - longValue(v2));
    }

    /**
     * Make a div between 2 values
     * @param v1
     * @param v2
     * @return v1 / v2
     */
    protected long divideValues(Number v1, Number v2) throws JasmineCollectorException {
        if (longValue(v2) == 0) {
            // Cannot make the divide: Choose to return 1 for now.
            // TODO
            return 1;
        }
        return (longValue(v1) / longValue(v2));
    }

    /**
     * Add a list of values of same type
     * @param values
     * @return Object of same type representing the sum of all values
     */
    protected long addValues(Collection<Number> values) throws JasmineCollectorException {
        long ret = 0L;
        for (Number o : values) {
            ret += longValue(o);
        }
        return ret;
    }

    /**
     * Multiply a list of values of same type
     * @param values
     * @return Object of same type representing the product of all values
     */
    protected long multiplyValues(Collection<Number> values) throws JasmineCollectorException {
        long ret = 0L;
        for (Number o : values) {
            ret *= longValue(o);
        }
        return ret;
    }

    /**
     * Try to cast a Number into a long value
     * @param nb
     * @return
     * @throws JasmineCollectorException
     */
    protected long longValue(Number nb) throws JasmineCollectorException {
        long val;
        if (nb instanceof Integer) {
            val = ((Integer)nb).longValue();
        } else if (nb instanceof Long) {
            val = ((Long)nb).longValue();
        } else if (nb instanceof Short) {
            val = ((Short)nb).longValue();
        } else {
            throw new JasmineCollectorException("This type is not implemented");
        }
        return val;
    }

}
