/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JProbe.java 8847 2011-09-13 08:09:38Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.manager;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.outer.JasmineOuter;
import org.ow2.jasmine.probe.probescheduler.TaskReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.List;

/**
 * This object represents a Running Probe.
 * @author durieuxp
 */
public class JProbe implements Runnable {

    protected static Log logger = LogFactory.getLog(JProbe.class);

    /**
     * Probe definition + probe state
     */
    JasmineProbe probeDef;

    /**
     * Task returned by the scheduler
     */
    TaskReference task;

    /**
     * List of Collectors used by the probe.
     */
    List<JasmineCollector> collectors;

    /**
     * List of Outers used by the probe.
     */
    List<JasmineOuter> outers;

    /**
     * Constructor
     */
    public JProbe(JasmineProbe def) {
        probeDef = def;
        def.setError("");
    }

    // -----------------------------------------------------------------------------
    // Getters and Setters
    // -----------------------------------------------------------------------------

    public JasmineProbe getProbeDef() {
        return probeDef;
    }

    /**
     * Get the Probe status, kept in the JasmineProbe object for convenience.
     * @return the probe status
     */
    public int getStatus() {
        return probeDef.getStatus();
    }

    /**
     * Set the Probe status, kept in the JasmineProbe object for convenience.
     * @param state the probe status
     */
    public void setStatus(int state) {
        probeDef.setStatus(state);
    }

    /**
     * Set the Probe error message, kept in the JasmineProbe object for convenience.
     * @param mess the error message
     */
    public void setError(String mess) {
        probeDef.setError(mess);
    }

    /**
     * Set the task reference returned by the scheduler.
     */
    public void setTask(TaskReference task) {
        this.task = task;
    }

    public List<JasmineOuter> getOuters() {
        return outers;
    }

    public void setOuters(List<JasmineOuter> outers) {
        this.outers = outers;
    }

    public List<JasmineCollector> getCollectors() {
        return collectors;
    }

    public void setCollectors(List<JasmineCollector> collectors) {
        this.collectors = collectors;
    }

    // -----------------------------------------------------------------------------
    // Interface for the ProbeManager
    // -----------------------------------------------------------------------------

    /**
     * Stop the Probe by stopping the collectors' polling.
     */
    public void halt() {
        logger.debug("");

        // Stop the scheduler
        if (task != null) {
            task.cancel(true);
            task = null;
        }

        // Stop Collectors
        for (JasmineCollector collector : collectors) {
            collector.stopPolling();
        }

        // Set the Probe status
        probeDef.setStatus(JasmineProbe.PROBE_STOPPED);
    }

    /**
     * Called by the JasmineProbeManager in startProbe(), when the probe is in the STOPPED state.
     * (restartProbe() operation).
     */
    public void launch() {
        logger.debug("");

        // Start Collectors
        for (JasmineCollector collector : collectors) {
            collector.startPolling();
        }

        // Set the Probe status
        probeDef.setStatus(JasmineProbe.PROBE_STARTED);
    }

    // -----------------------------------------------------------------------------
    // Runnable implementation
    // -----------------------------------------------------------------------------

    /**
     * Called by the Scheduler to make a polling on all indicators
     */
    public void run() {
        logger.debug("");

        // Init a JasmineProbeResult with the probe Id
        JasmineProbeResult result = new JasmineProbeResult(probeDef.getId());

        // Get results for each indicator
        for (JasmineCollector collector : collectors) {
            try {
                JasmineIndicatorValue val = collector.getLastResult();
                if (val != null) {
                    result.addValue(val);
                } else {
                    logger.debug("No result found for " + collector.getName());
                }
            } catch (JasmineCollectorException e) {
                logger.warn("Cannot get results on " + collector.getName() + " :" + e);
            } catch (NullPointerException e) {
                logger.error("NPE in run method");
                e.printStackTrace();
            }
        }

        // Publish the results on each Outer
        for (JasmineOuter outer : outers) {
            outer.publishData(result);
        }
    }

}
