/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JOuter.java 8825 2011-09-09 07:51:15Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.outers;

import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbeResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.outer.JasmineOuter;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * Outer instance.
 * this abstract class should be derived in different forms,
 * depending of the type of output:
 * console, file, mule, etc...
 * These implementations are in separate modules.
 * @author durieuxp
 */
public abstract class JOuter extends Thread implements JasmineOuter {

    /**
     * logger used for outers
     */
    protected Log logger = LogFactory.getLog(JOuter.class);

    /**
     * Queue of JasmineProbeResult elements to be sent
     */
    private LinkedList<JasmineProbeResult> dataqueue = new LinkedList<JasmineProbeResult>();

    /**
     * CSV separator.
     */
    protected static final String SEPARATOR = ";";

    /**
     * Separator used to construct data names from indicator name and result name.
     */
    protected static final String NAME_SEPARATOR = ".";


    private boolean valid = false;
    private boolean waiting = false;
    private long outerTimeout = 10 * 1000;

    /**
     * Std format for the date.
     */
    protected SimpleDateFormat dateformat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    abstract public void processData(JasmineProbeResult result);

    public JOuter(JasmineOutput output) {
        super("JasmineOuter-" + output.getName());
        valid = true;
    }

    /**
     * run method for Thread implementation.
     */
    public void run() {
        while (valid) {
            JasmineProbeResult data = null;
            synchronized (this) {
                if (dataqueue.isEmpty()) {
                    try {
                        waiting = true;
                        wait(outerTimeout);
                        waiting = false;
                    } catch (InterruptedException e) {
                        logger.warn(getName() + ": outer interrupted", e);
                    } catch (Exception e) {
                        logger.warn(getName() + ": outer exception", e);
                    }
                }
                try {
                    data = dataqueue.removeFirst();
                } catch (NoSuchElementException e) {
                    data = null;
                }
            }
            if (data != null) {
                processData(data);
            }
        }
    }

    /**
     * stop the AbstractOuter
     */
    public synchronized void stopIt() {
        valid = false;
        notify();
    }

    // ------------------------------------------------------------------------------
    // JasmineOuter implementation
    // ------------------------------------------------------------------------------

    /**
     * Publish data on this Output
     * @param result data to publish
     */
    public synchronized void publishData(JasmineProbeResult result) {
        logger.debug("");
        dataqueue.addLast(result);
        if (waiting) {
            notify();
        }
    }

    /**
     * Check if simple name (the indicator's name) is to be published, or
     * if the published name is to be composed from the indicator name and
     * a simple result name.
     * This method implements the following rule:
     * If one result, then use a simple name, the indicator name
     * Otherwise:
     *   If the results have different names, then use a composed name
     *   Otherwise:
     *      If the names are equal to the indicator name, then use a simple name, the indicator name
     *      Otherwise, use a composed name
     * @param name the indicator name
     * @param jsrs the JasmineSingleResults of the indicator
     * @return true if
     */
    protected boolean simpleDataName(final String name, final Collection<JasmineSingleResult> jsrs) {
        boolean result = true;
        if (jsrs.size() > 1) {
            String oneResultName = null;
            for (JasmineSingleResult jsr : jsrs) {
                // Check that JSR names are different from each other
                // different from the indicator name
                String resultName = jsr.getName();
                if (oneResultName == null) {
                    oneResultName = resultName;
                }
                if (!oneResultName.equals(resultName)) {
                    // JSR names are different
                    result = false;
                    return result;
                }
            }
            // all the JSR names are equal to oneResName
            if (oneResultName.equals(name)) {
                result = true;
            } else {
                // but different from the indicator name => use composed names
                result = false;
            }

        }
        return result;
    }

    protected String composeName(final String indicatorName, final String resultName) {
        return indicatorName + NAME_SEPARATOR + resultName;
    }

}
