/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: TargetCreateCommand.java 8788 2011-09-01 15:00:57Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.JasmineTarget;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.ParameterException;

/**
 * Implements target-create command.
 * @author danesa
 */
@Component(name="TargetCreateCommand")
@Provides
public class TargetCreateCommand extends JasmineProbeCommand implements Command {

        /**
         * command name
         */
        private String name = "target-create";

        /**
         * necessary to be able to print out usage even when wrong options are provided by the user
         */
        private String requiredOptions = "-n targetName -u targetUrl";

        /**
         * command description
         */
        private String description = "Create a Target";

        // ----------------------------------------------------------------------------
        // ipojo management
        // ----------------------------------------------------------------------------

        @Requires
        JasmineProbeManager probeManager = null;

        // ----------------------------------------------------------------------------
        // constructors
        // ----------------------------------------------------------------------------

        /**
         * Default Constructor
         */
        public TargetCreateCommand() {
            super();
            setParams(new TargetCreateCommandParams());
        }

        // ----------------------------------------------------------------------------
        // org.apache.felix.shell.Command implementation
        // ----------------------------------------------------------------------------

        /**
         * Returns the name of the command that is implemented by the interface.
         * The command name should not contain whitespace and should also be unique.
         * @return the name of the command.
         */
        public String getName() {
            return name;
        }

        /**
         * Returns a short description of the command; this description should be as short as possible.
         * This information is used when generating the command help information.
         * @return a short description of the command.
         */
        public String getShortDescription() {
            return description;
        }

        public String getProperties() {
            String props = new String("  Properties: ....");
            return props;
        }

        public String getRequiredOptions() {
            return requiredOptions;
        }

        public JasmineProbeManager getProbeManager() {
            return probeManager;
        }

        /**
         * Executes the command using the supplied command line, output print stream, and error print stream.
         * @param line the complete command line, including the command name.
         * @param out  the print stream to use for standard output.
         * @param err  the print stream to use for standard error.
         */
        public void execute(String line, PrintStream out, PrintStream err) {
            boolean validOptions = parseOptions(line, out, err);
            if (!validOptions) {
                err.println(getUsage());
                return;
            }

            TargetCreateCommandParams params = ((TargetCreateCommandParams) getParams());

            // get command arguments
            String targetName = params.getName();
            String targetUrl = params.getUrl();
            List<String> targetParams = params.getProps();
            Map<String, String> properties = null;
            if (!targetParams.isEmpty()) {
                properties = new HashMap<String, String>();
            }
            for (String targetParam : targetParams) {
                if (validParam(targetParam)) {
                    int indexEqual = targetParam.indexOf("=");
                    String targetParamName = targetParam.substring(0, indexEqual);
                    String targetParamValue = targetParam.substring(indexEqual + 1, targetParam.length());
                    properties.put(targetParamName, targetParamValue);
                } else {
                    err.println(name + " " + targetName + " failed");
                    // wrong parameter format
                    err.println("Wrong properties format for " + targetParam);
                    err.println("Expected format is paramName=paramValue");
                    err.println(getUsage());
                    params.reset();
                    return;
                }
            }
            params.reset();

            // Create the JasmineTarget
            JasmineTarget newTarget = new JasmineTarget();
            newTarget.setName(targetName);
            newTarget.setUrl(targetUrl);
            newTarget.setProperties(properties);

            try {
                probeManager.createTarget(newTarget);
            } catch (JasmineProbeException e) {
                err.println(name + " " + targetName + " failed");
                err.println(e);
                return;
            }
            out.println(name + " " + targetName +  " was successfull");

        }

}
