/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JProbeManager.java 9026 2011-09-28 14:32:11Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.manager.internal;

import java.io.*;
import java.math.BigInteger;
import java.util.*;

import javax.management.ObjectName;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.felix.ipojo.annotations.*;
import org.osgi.framework.ServiceReference;
import org.ow2.jasmine.probe.*;
import org.ow2.jasmine.probe.collector.JasmineAggregateService;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.manager.JProbe;
import org.ow2.jasmine.probe.mbeans.AB;
import org.ow2.jasmine.probe.mbeans.CD;
import org.ow2.jasmine.probe.mbeans.TestOpenMBean;
import org.ow2.jasmine.probe.outer.JasmineOuterService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.jasmine.probe.probemanager.generated.*;
import org.ow2.jasmine.probe.probescheduler.SchedulerService;
import org.ow2.jonas.jmx.JmxService;
import org.ow2.jonas.lib.bootstrap.JProp;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implements the main service of the JasmineProbe module exposing the JasmineProbeManager interface
 * to the JasmineProbe module's clients.
 *
 * @author durieuxp
 */
@Component(name = "JasmineProbeManagerService")
@Provides
public class JProbeManager implements JasmineProbeManager, ProbeManager, JProbeManagerMXBean {

    protected static Log logger = LogFactory.getLog(JProbe.class);

    private ProbeConfig probeConfig = null;

    /**
     * JasmineProbe definitions
     */
    private Map<String, JasmineProbe> probes = new HashMap<String, JasmineProbe>();

    /**
     * JasmineOutput definitions
     */
    private Map<String, JasmineOutput> outputs = new HashMap<String, JasmineOutput>();

    /**
     * JasmineIndicator definitions
     */
    private Map<String, JasmineIndicator> indicators = new HashMap<String, JasmineIndicator>();

    /**
     * JasmineTarget definitions
     */
    private Map<String, JasmineTarget> targets = new HashMap<String, JasmineTarget>();

    /**
     * running probes
     */
    private Map<String, JProbe> runnings = new HashMap<String, JProbe>();

    /**
     * List of the available Outer services.
     */
    private Map<String, JasmineOuterService> outerservices = new HashMap<String, JasmineOuterService>();

    /**
     * List of the available Collector services.
     */
    private Map<String, JasmineCollectorService> collectorservices = new HashMap<String, JasmineCollectorService>();

    /**
     * List of the custom available Aggregate services.
     */
    private Map<String, JasmineAggregateService> aggregateservices = new HashMap<String, JasmineAggregateService>();

    /**
     * Listeners for Probe changes
     */
    private List<JasmineProbeListener> probeListeners = new ArrayList<JasmineProbeListener>();

    /**
     * List of Outputs to be used when not defined in a probe.
     */
    private List<String> defaultOutputList = new ArrayList<String>();

    /**
     * Count used to generate probe ids
     */
    private int probeCount = 0;

    private ObjectName mbeanObjectName = null;

    // -----------------------------------------------------------------------------------
    // ipojo management
    // -----------------------------------------------------------------------------------

    @Validate
    public void start() {
        logger.debug("JasmineProbeManager service activated.");
        try {
            readConfig();

        } catch (Exception e) {
            // Cannot read configuration
            logger.error("Cannot read configuration");
        }

        // register mbean
        try {
            mbeanObjectName = new ObjectName("jasmine:dest=probe-manager");
            jmxService.registerMBean(this, mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot register mbean: " + e);
        }

        // register test mbeans
        registerMBeans();
    }

    private void registerMBeans() {
        try {
            ObjectName mbeanObjectName = ObjectName.getInstance("jasmine:dest=open-manager");

            AB ab_1 = new AB(1, 2);
            AB ab_2 = new AB(3, 4);
            AB ab_3 = new AB(5, 6);

            Long l_1 = new Long(2356);
            Long l_2 = new Long(45678);
            Long l_3 = new Long(1000004);
            Long[] longs = new Long[] {l_1, l_2, l_3};

            CD cd = new CD(longs, ab_3);

            ObjectName[] ons = new ObjectName[] {mbeanObjectName};

            AB[] elems = new AB[] {ab_1, ab_2};

            TestOpenMBean testMbean = new TestOpenMBean(ab_1, ab_2, cd, longs, elems, ons);

            jmxService.registerMBean(testMbean, mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot register mbean: " + e);
        }
    }

    private void unregisterMBeans() {
        try {
            ObjectName mbeanObjectName = ObjectName.getInstance("jasmine:dest=open-manager");
            jmxService.unregisterMBean(mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot unregister mbean: " + e);
        }
    }


    @Invalidate
    public void stop() {
        logger.debug("JasmineProbeManager service stopped.");

        // unregister mbean
        jmxService.unregisterMBean(mbeanObjectName);

        // register test mbeans
        unregisterMBeans();
    }

    @Requires
    protected SchedulerService jasmineProbeScheduler = null;

    @Requires
    protected JmxService jmxService = null;

    @Bind(aggregate = true, optional = true)
    protected void bindOuterService(JasmineOuterService jos, ServiceReference sr) {
        String type = (String) sr.getProperty("output.type");
        if (type == null) {
            logger.warn("JasmineOuterService with undefined output.type");
            return;
        }
        JasmineOuterService old = outerservices.get(type);
        if (old != null) {
            logger.warn("JasmineOuterService already defined for " + type);
            return;
        }
        outerservices.put(type, jos);
        logger.info("{0} JasmineOuterService bound into JasmineProbe.", type);

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindOuterService(JasmineOuterService jos, ServiceReference sr) {
        // TODO
    }

    @Bind(aggregate = true, optional = true)
    protected void bindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        String type = (String) sr.getProperty("indicator.type");
        if (type == null) {
            logger.warn("JasmineCollectorService with undefined indicator.type");
            return;
        }
        JasmineCollectorService old = collectorservices.get(type);
        if (old != null) {
            logger.warn("JasmineCollectorService already defined for " + type);
            return;
        }
        collectorservices.put(type, jcs);
        logger.info("{0} JasmineCollectorService bound into JasmineProbe.", type);

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        // TODO
    }

    @Bind(aggregate = true, optional = true)
    protected void bindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("JasmineAggregateService with undefined aggregate.type binded");
            return;
        }
        JasmineAggregateService old = aggregateservices.get(type);
        if (old != null) {
            // A service for aggregate function already exists, use it if its not un-deployed.
            // Another approach could be to replace the old function by a new one.
            logger.warn("AggregateService already defined for " + type + " function. Please check deployment plan and undeploy old service !");
            return;
        }
        aggregateservices.put(type, jas);
        logger.info("{0} JasmineAggregateService bound into JasmineProbe.", type);
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("JasmineAggregateService with undefined aggregate.type unbinded");
            return;
        }
        if (aggregateservices.containsKey(type)) {
            aggregateservices.remove(type);
            logger.info("{0} JasmineAggregateService unbound into JasmineProbe.", type);
        }
    }

    // -----------------------------------------------------------------------------------
    //  internal interface
    // -----------------------------------------------------------------------------------

    /**
     * return the scheduler
     * @return referencee on the SchedulerService
     */
    public SchedulerService getScheduler() {
        return jasmineProbeScheduler;
    }

    /**
     * Get the OuterService for a given type
     */
    public JasmineOuterService getOuterService(String type) {
        return outerservices.get(type);
    }

    /**
     * Get the CollectorService for a given type
     */
    public JasmineCollectorService getCollectorService(String type) {
        return collectorservices.get(type);
    }

    // -----------------------------------------------------------------------------------
    //  JasmineProbeManager implementation
    // -----------------------------------------------------------------------------------

    /**
     * Create a new probe defined by its data
     *
     * @param probe probe description
     * @return the Id to be used to reference this probe later.
     */
    public synchronized String createProbe(JasmineProbe probe) throws JasmineProbeException {
        // Generate an Id if not supplied
        String id = probe.getId();
        if (id == null) {
            // If possible, take the name of unique indicator
            if (probe.getIndicatorList().size() == 1) {
                String strindic = probe.getIndicatorList().get(0);
                boolean found = false;
                for (JasmineProbe p : probes.values()) {
                    if (p.getId().equals(strindic)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    id = strindic;
                }
            }
            if (id == null) {
                id = getNewProbeId();
            }
            probe.setId(id);
        } else {
            // Check Id not already used
            for (JasmineProbe p : probes.values()) {
                if (p.getId().equals(id)) {
                    logger.error("Id already used: " + id);
                    throw new JasmineProbeException("Id already used");
                }
            }
        }

        // Check validity of the JasmineProbe
        if (probe.getIndicatorList().isEmpty()) {
             throw new JasmineProbeException("A probe must have at least one indicator");
        }
        if (probe.getOutputList().isEmpty()) {
            probe.setOutputList(defaultOutputList);
        }

        logger.debug("creating probe : " + id);

        probes.put(id, probe);
        return id;
    }

    /**
     * Remove a probe.
     *
     * @param probeId probe identifier.
     */
    public synchronized void removeProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probes.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot remove probe with unexistent id " + probeId);
        }

        // Check if probe running
        JProbe running = runnings.get(probeId);
        if (running != null) {
            running.suspend();
            runnings.remove(probeId);
        }
        logger.debug("removing probe " + probeId);
        probes.remove(probeId);
    }

    /**
     * Remove all the managed probes.
     * Stop probe running before.
     *
     * @throws JasmineProbeException
     */
    public synchronized void removeAllProbes() throws JasmineProbeException {
        logger.debug("");

        // must build a list first, to avoid ConcurrentModificationException.
        List<String> namelist = new ArrayList<String>();
        for (String n : probes.keySet()) {
            namelist.add(n);
        }
        for (String name : namelist) {
            removeProbe(name);
        }
    }

    /**
     * Get all the Probe definitions in one operation.
     * @return all the probe definitions
     */
    public synchronized List<JasmineProbe> getProbes() {
        logger.debug("");
        List<JasmineProbe> ret = new ArrayList<JasmineProbe>();
        ret.addAll(probes.values());
        return ret;
    }

    /**
     * @return all the probe ids
     */
    public synchronized String [] listProbes() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (String n : probes.keySet()) {
            ret.add(n);
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get a Probe by its name
     * @param probeId probe identifier.
     * @return the Probe definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineProbe getProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probes.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot find probe with id " + probeId);
        }
        return def;
     }

    /**
     * Start a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be started.
     */
    public synchronized void startProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probes.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with unexistent id " + probeId);
        }

        // Check if already running
        JProbe running = runnings.get(probeId);
        if (running != null) {
            switch (running.getStatus()) {
                case JasmineProbe.PROBE_RUNNING:
                    logger.info("Probe already running: " + probeId);
                    return;
                case JasmineProbe.PROBE_STARTED:
                    logger.info("Probe already started: " + probeId);
                    return;
                case JasmineProbe.PROBE_FAILED:
                    logger.info("restarting a failed probe: " + probeId);
                    break;
                case JasmineProbe.PROBE_STOPPED:
                    logger.info("restarting a stopped probe: " + probeId);
                    break;
                default:
                    logger.warn("Bad state for: " + probeId);
                    break;
            }
        } else {
            // Create the JProbe object (a running probe)
            running = new JProbe(def, this);
            runnings.put(probeId, running);
        }
        running.resume();
    }

    /**
     * Stop a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be stopped.
     */
    public synchronized void stopProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probes.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with unexistent id " + probeId);
        }

        // Check if already running
        JProbe running = runnings.get(probeId);
        if (running == null) {
            throw new JasmineProbeException("This probe is not running: " +  probeId);
        }

        // Stop the probe by halting polling and changing status to JasmineProbe.PROBE_STOPPED
        running.suspend();
    }

    /**
     * Start all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void startAllProbes() throws JasmineProbeException {
        for (JasmineProbe probe : probes.values()) {
            try {
                startProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not start probe " + probe.getId() + " :" + e);
            }
        }
    }

    /**
     * Stop all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void stopAllProbes() throws JasmineProbeException {
        for (JProbe probe : runnings.values()) {
            try {
                probe.suspend();
            } catch (Exception e) {
                logger.warn("could not stop probe " + probe.getProbeDef().getId() + " :" + e);
            }
        }
    }

    /**
     * Change a Probe with new parameters
     * All parameters not supplied are let inchanged.
     *
     * @param newdef  partial probe description
     */
    public synchronized void changeProbe(JasmineProbe newdef) throws JasmineProbeException {

        // Retrieve the Probe by its Id
        String probeId = newdef.getId();
        JasmineProbe olddef = probes.get(probeId);
        if (olddef == null) {
            throw new JasmineProbeException("Probe not found: " + probeId);
        }

        // Stop it first if it is running.
        // It will be restarted after changed
        boolean restart = false;
        JProbe running = runnings.get(probeId);
        if (running != null) {
            running.suspend();
            restart = true;
        }

        // Change period
        if (newdef.getPeriod() > 0) {
            olddef.setPeriod(newdef.getPeriod());
        }

        // Change output list
        if (newdef.getOutputList().size() > 0) {
            olddef.setOutputList(newdef.getOutputList());
        }

        // Change indicator list
        if (newdef.getIndicatorList().size() > 0) {
            olddef.setIndicatorList(newdef.getIndicatorList());
        }

        // Change target list
        if (newdef.getTargetList().size() > 0) {
            olddef.setTargetList(newdef.getTargetList());
        }

        // Restart probe if it was suspended
        if (restart) {
            running.resume();
        }
    }

    /**
     * Register a new Output in the ProbeManager
     * A name is associated to the Output definition.
     * This Output will be usable later in probes.
     *
     * @param newoutput output description
     * @return the Id to be used to reference this output later.
     */
    public String createOutput(JasmineOutput newoutput) throws JasmineProbeException {
        return createOutputInternal(newoutput, true);
    }

    /**
     * Change an output. The output must exist.
     *
     * @param newdef  output changes description
     * @throws JasmineProbeException is thrown if the output does not exist.
     */
    public void changeOutput(JasmineOutput newdef) throws JasmineProbeException {
        logger.debug("");

        // Find the JasmineOutput to be changed
        String name = newdef.getName();
        JasmineOutput output = outputs.get(name);
        if (output == null) {
            throw new JasmineProbeException("This output does not exist: " +  name);
        }

        // Stop all running probes using this output.
        List<JProbe> prlist = new ArrayList<JProbe>();
        for (JProbe running : runnings.values()) {
            JasmineProbe probe = running.getProbeDef();
            for (String oname : probe.getOutputList()) {
                if (oname.equals(name)) {
                    running.suspend();
                    prlist.add(running);
                    break;
                }
            }
        }

        // remove Outers defined on this Output
        JasmineOuterService jos = outerservices.get(output.getType());
        if (jos != null) {
            jos.removeOuters(name);
        }

        // Change type
        if (newdef.getType() != null) {
            output.setType(newdef.getType());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            String oldval = output.getProperties().get(key);
            if (oldval != null) {
                output.getProperties().remove(key);
            }
            output.getProperties().put(key, newval);
        }

        // restart probes
        for (JProbe running : prlist) {
            running.resume();
        }
    }

    /**
     * remove an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     *
     * @param name ident of the output to remove
     * @throws JasmineProbeException is thrown if the output does not exist,
     *                               or if it is used by a running probe.
     */
    public void removeOutput(String name) throws JasmineProbeException {
        logger.debug("");
        JasmineOutput output = outputs.get(name);
        if (output == null) {
            throw new JasmineProbeException("This output does not exist: " +  name);
        }
        if (isOutputUsed(output, true)) {
            throw new JasmineProbeException("This output is used and cannot be removed: " +  name);
        }
        JasmineOuterService jos = outerservices.get(output.getType());
        if (jos != null) {
            jos.removeOuters(name);
        }
        outputs.remove(name);
    }

    /**
     * @return all the output names
     */
    public synchronized String [] listOutputs() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (JasmineOutput output : outputs.values()) {
            ret.add(output.getName());
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all the Output definitions in one operation.
     * @return all the output definitions
     */
    public synchronized List<JasmineOutput> getOutputs() {
        logger.debug("");
        List<JasmineOutput> ret = new ArrayList<JasmineOutput>();
        ret.addAll(outputs.values());
        return ret;
    }


    /**
     * Get an Output by its name
     * @param id output identifier.
     * @return the Output definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineOutput getOutput(String id) throws JasmineProbeException {
        logger.debug("");
        JasmineOutput def = outputs.get(id);
        if (def == null) {
            throw new JasmineProbeException("Cannot find output with id " + id);
        }
        return def;
     }

    /**
     * get the list of possible properties for Output, depending of its type.
     *
     * @param type type of the Output (console, file, ...)
     * @return List of JasminePropertyInfo
     */
    public List<JasminePropertyInfo> getOutputPropertyInfos(String type) {
        // Find the appropriate OuterService
        JasmineOuterService jos = outerservices.get(type);
        if (jos == null) {
            logger.error("No OuterService found for type " + type);
            return null;
        }
        return jos.getPropertiesInfo();
    }

    /**
     * @return the available output types.
     */
    public synchronized Set<String> getOutputTypes() {
        return outerservices.keySet();
    }

    /**
     * Get the list of probes using this output
     * @param name  Output name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    public synchronized Set<String> getOutputCurrentUse(String name) throws JasmineProbeException {
        Set<String> ret = new HashSet<String>();
        for (JasmineProbe probe : probes.values()) {
            for (String output : probe.getOutputList()) {
                if (output.equals(name)) {
                    ret.add(probe.getId());
                }
            }
        }
        return ret;
    }

    /**
     * Create a new Indicator
     *
     * @param newindic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    public String createIndicator(JasmineIndicator newindic) throws JasmineProbeException {
        return createIndicatorInternal(newindic, true);
    }

    /**
     * Change an indicator. It must exist.
     *
     * @param newdef indicator description
     * @throws JasmineProbeException is thrown if the indicator does not exist,
     *                               or if it is used by a running probe.
     */
    public void changeIndicator(JasmineIndicator newdef) throws JasmineProbeException {
        logger.debug("");

        // Find the JasmineIndicator to be changed
        String name = newdef.getName();
        JasmineIndicator indic = indicators.get(name);
        if (indic == null) {
            throw new JasmineProbeException("This indcator does not exist: " +  name);
        }

        // Stop all running probes using this indicator.
        List<JProbe> prlist = new ArrayList<JProbe>();
        for (JProbe running : runnings.values()) {
            JasmineProbe probe = running.getProbeDef();
            for (String oname : probe.getIndicatorList()) {
                if (oname.equals(name)) {
                    running.suspend();
                    prlist.add(running);
                    break;
                }
            }
        }

        // remove Collectors defined on this Indicator
        JasmineCollectorService jcs = collectorservices.get(indic.getType());
        if (jcs != null) {
            jcs.removeCollectors(name);
        }

        // Change type
        if (newdef.getType() != null) {
            indic.setType(newdef.getType());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            String oldval = indic.getProperties().get(key);
            if (oldval != null) {
                indic.getProperties().remove(key);
            }
            indic.getProperties().put(key, newval);
        }

        // restart probes
        for (JProbe running : prlist) {
            running.resume();
        }
    }


    /**
     * remove an indicator
     *
     * @param name ident of the indicator to remove
     */
    public void removeIndicator(String name) throws JasmineProbeException {
        logger.debug("");
        JasmineIndicator indic = indicators.get(name);
        if (indic == null) {
            throw new JasmineProbeException("This indicator does not exist: " +  name);
        }
        if (isIndicatorUsed(indic, true)) {
            throw new JasmineProbeException("This indicator is used and cannot be removed: " +  name);
        }

        JasmineCollectorService jcs = collectorservices.get(indic.getType());
        if (jcs != null) {
            jcs.removeCollectors(name);
        }
        indicators.remove(name);
    }

    /**
     * Get an Indicator by its name
     * @param id identifier.
     * @return the definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineIndicator getIndicator(String id) throws JasmineProbeException {
        logger.debug("");
        JasmineIndicator def = indicators.get(id);
        if (def == null) {
            throw new JasmineProbeException("Cannot find indicator with id " + id);
        }
        return def;
     }

    /**
     * Get all the Indicator definitions in one operation.
     * @return all the indicator definitions
     */
    public synchronized List<JasmineIndicator> getIndicators() {
        logger.debug("");
        List<JasmineIndicator> ret = new ArrayList<JasmineIndicator>();
        ret.addAll(indicators.values());
        return ret;
    }


    /**
     * @return all the indicator names
     */
    public synchronized String [] listIndicators() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (JasmineIndicator indic : indicators.values()) {
            ret.add(indic.getName());
        }
        return ret.toArray(new String [0]);
    }


    /**
     * get the list of possible properties for Indicator, depending of its type.
     *
     * @param type type of the Indicator (jmx, lewys, ...)
     * @return List of JasminePropertyInfo
     */
    public List<JasminePropertyInfo> getIndicatorPropertyInfos(String type) {
        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(type);
        if (jcs == null) {
            logger.error("No CollectorService found for type " + type);
            return null;
        }
        return jcs.getPropertiesInfo();
    }

    /**
     * @return the available indicator types.
     */
    public synchronized Set<String> getIndicatorTypes() {
        return collectorservices.keySet();
    }

    /**
     * Get the list of probes using this indicator
     * @param name  Indicator name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    public synchronized Set<String> getIndicatorCurrentUse(String name) throws JasmineProbeException {
        Set<String> ret = new HashSet<String>();
        for (JasmineProbe probe : probes.values()) {
            for (String indic : getRecursiveIndicatorList(probe)) {
                if (indic.equals(name)) {
                    ret.add(probe.getId());
                }
            }
        }
        return ret;
    }

    /**
     * Create a new target
     *
     * @param newtarget target description
     * @return the Id to be used to reference this target later.
     */
    public synchronized String createTarget(JasmineTarget newtarget) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newtarget.getName();
        if (id == null || id.length() == 0) {
            throw new JasmineProbeException("No valid target name");
        }
        logger.debug("Create Target " + id);

        // Check if already known
        JasmineTarget target = targets.get(id);
        if (target != null) {
            logger.debug("target already known: " + id);
            if (target.equals(newtarget)) {
                // If same target already created, just return.
                return id;
            }
            if (isTargetUsed(target)) {
                logger.warn("oldtarget:" + target);
                logger.warn("newtarget:" + newtarget);
                throw new JasmineProbeException(id + "Target already known with another definition");
            }
            // remove old definition and replace by the new one.
            targets.remove(id);
        }
        targets.put(id, newtarget);
        return id;
    }

    /**
     * Change a target. It must exist, otherwise an exception is thrown.
     *
     * @param newtarget new target description
     * @throws JasmineProbeException is thrown if the target does not exist.
     */
    public void changeTarget(JasmineTarget newtarget) throws JasmineProbeException {
        // TODO
        throw new JasmineProbeException("changeTarget not implemented");
    }

    /**
     * remove a target
     *
     * @param name ident of the target to remove
     */
    public void removeTarget(String name) throws JasmineProbeException {
        logger.debug("");
        JasmineTarget target = targets.get(name);
        if (target == null) {
            throw new JasmineProbeException("This target does not exist: " +  name);
        }
        if (isTargetUsed(target)) {
            throw new JasmineProbeException("This target is used and cannot be removed: " +  name);
        }
        targets.remove(name);
    }

    /**
     * @return all the target names
     */
    public synchronized String [] listTargets() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (JasmineTarget target : targets.values()) {
            ret.add(target.getName());
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all the Target definitions in one operation.
     * @return all the target definitions
     */
    public synchronized List<JasmineTarget> getTargets() {
        logger.debug("");
        List<JasmineTarget> ret = new ArrayList<JasmineTarget>();
        ret.addAll(targets.values());
        return ret;
    }

    /**
     * Get a JasmineTarget by its name
     * @param name
     * @return JasmineTarget definition
     */
    public JasmineTarget getTarget(String name) {
        JasmineTarget target = targets.get(name);
        return target;
    }

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     *
     * @param listener object that treats the probe state change
     */
    public synchronized void addProbeListener(JasmineProbeListener listener) {
        logger.debug("");
        probeListeners.add(listener);
    }

    /**
     * Remove a ProbeListener previusly registered.
     *
     * @param listener object that treats the probe state change
     */
    public synchronized void removeProbeListener(JasmineProbeListener listener) {
        logger.debug("");
        probeListeners.remove(listener);
    }

    /**
     * Save the current configuration in the specified xml file
     *
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws JasmineProbeException
     */
    public void saveConfig(String path) throws JasmineProbeException {
        logger.debug("saveConfig");
        FileOutputStream outputStream = null;
        try {
            outputStream = new FileOutputStream(new File(JProp.getConfDir(), "probe-config.xml"));
        } catch (FileNotFoundException e) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'" + e);
        }
        if (outputStream == null) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'");
        }
        try {
            saveConfigFile(outputStream);
        } catch (Exception e) {
            logger.warn("could not save config: " + e);
            throw new JasmineProbeException("could not save config: " + e.getMessage());
        }
    }

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     *
     * @param path the xml file.
     * @throws JasmineProbeException
     */
    public void loadConfig(String path) throws JasmineProbeException {
        logger.debug("loadConfig: " + path);
        try {
            loadConfigFile(new FileInputStream(path));
        } catch (Exception e) {
            logger.warn("could not load config: " + e);
            throw new JasmineProbeException("could not load config: " + e.getMessage());
        }
    }

    // -----------------------------------------------------------------
    // ProbeManager implementation
    // -----------------------------------------------------------------

    /**
     * Get a Collector by its Indicator name
     * @param indicatorName indicator name
     * @param period the period of the probe using the indicator
     * @param probeid
     * @return The collector corresponding to an indicator.
     */
    public synchronized JasmineCollector getCollector(String indicatorName, int period, String probeid) throws JasmineProbeException {

        JasmineIndicator indicator = indicators.get(indicatorName);
        if (indicator == null) {
            throw new JasmineProbeException("Cannot return collector for unexistent indicator " + indicatorName);
        }

        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(indicator.getType());
        if (jcs == null) {
            logger.error("No CollectorService found for {0} having type {1}", indicatorName, indicator.getType());
            throw new JasmineProbeException("No CollectorService found for " + indicatorName);
        }
        // Get the Collector from the CollectorService
        JasmineCollector collector = null;
        try {
            collector = jcs.getCollector(indicator, period, probeid);
        } catch (JasmineCollectorException e) {
            logger.error("Could not get the Collector for " + indicatorName + ": " + e);
            throw new JasmineProbeException("Could not get the Collector for " + indicatorName);
        }
        return collector;
    }

    /**
     * @return If AggregateServices exist, return the functions these services implement (the values of aggregate.type properties).
     * Else, return null.
     */
    public synchronized List<String> getAggregateFuntions() {
        if (!aggregateservices.isEmpty()) {
            List<String> functions = new ArrayList<String>();
            Set<String> keySet = aggregateservices.keySet();
            for (String function : keySet) {
                functions.add(function);
            }
            return functions;
        }
        return null;
    }

    /**
     * Return the AggregateService for a given aggregate function
     * @param key the aggregate function that is searched
     * @return the AggregateService having aggregate.type equal to key. Else, return null.
     */
    public synchronized JasmineAggregateService getAggregate(String key) {
        if (aggregateservices.containsKey(key)) {
            return aggregateservices.get(key);
        }
        return null;
    }

    /**
     * @return the current domain name.
     */
    public String getDomainName() {
       return jmxService.getDomainName();
    }

    /**
     * @return the current server name.
     */
    public String getServerName() {
       return jmxService.getJonasServerName();
    }


    // --------------------------------------------------------------------------------------
    // Private methods
    // --------------------------------------------------------------------------------------

    /**
     * Create a new Output
     * @param newoutput
     * @param check
     * @return
     * @throws JasmineProbeException
     */
    private synchronized String createOutputInternal(JasmineOutput newoutput, boolean check) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newoutput.getName();
        if (id == null || id.length() == 0) {
            throw new JasmineProbeException("No valid output name");
        }

        // Check that a type is provided
        String type = newoutput.getType();
        if (type == null) {
            throw new JasmineProbeException("No type provided");
        }

        if (check) {
            // Check that the type is supported
            JasmineOuterService jos = outerservices.get(type);
            if (jos == null) {
                throw new TypeNotSupportedException("Output type not supported: " + type);
            }

            // Check that mandatory properties are supplied
            String missinglist = "";
            for (JasminePropertyInfo jpi : jos.getPropertiesInfo()) {
                if (jpi.isRequired()) {
                    boolean found = false;
                    for (String key : newoutput.getProperties().keySet()) {
                        if (key.equals(jpi.getName())) {
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        missinglist += " " + jpi.getName();
                    }
                }
            }
            if (missinglist.length() > 0) {
                throw new PropertyMissingException("Missing mandatory properties:" + missinglist);
            }
        }

        logger.debug("Create Output " + id);

        // Check if already known
        JasmineOutput output = outputs.get(id);
        if (output != null) {
            logger.debug("output already known: " + id);
            if (output.equals(newoutput)) {
                // If same output already created, just return.
                return id;
            }
            if (isOutputUsed(output, true)) {
                logger.warn("oldoutput:" + output);
                logger.warn("newoutput:" + newoutput);
                throw new JasmineProbeException(id + " output already known with another definition");
            }
            // remove old definition and replace by the new one.
            outputs.remove(id);
        }

        outputs.put(id, newoutput);
        return id;
    }

    /**
     * Create a new Indicator
     *
     * @param newindic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    public synchronized String createIndicatorInternal(JasmineIndicator newindic, boolean check) throws JasmineProbeException {

        // Id chosen by the user.
        String id = newindic.getName();
        if (id == null || id.length() == 0) {
            // TODO: generate Id if not provided ?
            throw new JasmineProbeException("Automatic indicator name is not supported yet");
        }

        // Check that a type is provided
        String type = newindic.getType();
        if (type == null) {
            throw new TypeNotSupportedException("No type provided");
        }

        if (check) {
            // Check that the type is supported
            JasmineCollectorService jcs = collectorservices.get(type);
            if (jcs == null) {
                throw new TypeNotSupportedException("Indicator type not supported: " + type);
            }

            // Check that mandatory properties are supplied
            String missinglist = "";
            for (JasminePropertyInfo jpi : jcs.getPropertiesInfo()) {
                if (jpi.isRequired()) {
                    boolean found = false;
                    for (String key : newindic.getProperties().keySet()) {
                        if (key.equals(jpi.getName())) {
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        missinglist += " " + jpi.getName();
                    }
                }
            }
            if (missinglist.length() > 0) {
                throw new PropertyMissingException("Missing mandatory properties:" + missinglist);
            }
        }

        logger.debug("Create Indicator " + id);

        // Check if already known
        JasmineIndicator indic = indicators.get(id);
        if (indic != null) {
            logger.debug("indicator already known: " + id);
            if (indic.equals(newindic)) {
                // If same indicator already created, just return.
                return id;
            }
            if (isIndicatorUsed(indic, true)) {
                logger.warn("old indic:" + indic);
                logger.warn("new indic:" + newindic);
                throw new JasmineProbeException(id + "Indicator already known with another definition");
            }
            // remove old definition and replace by the new one.
            indicators.remove(id);
        }

        indicators.put(id, newindic);
        return id;
    }

    /**
     * Generate an unique identifier
     */
    private synchronized String getNewProbeId() {
        String ret = "probe-0";
        boolean found = false;
        while (! found) {
            ret = "probe-" + ++probeCount;
            found = true;
            for (JasmineProbe p : probes.values()) {
                if (p.getId().equals(ret)) {
                    found = false;
                    break;
                }
            }
        }
        return ret;
    }

    /**
     * Read configuration.
     */
    private void readConfig() throws Exception {
        // Retrieve the probe config file
        String configurationFile = "probe-config.xml";
        ClassLoader loader = ClassLoader.getSystemClassLoader();
        InputStream resource = null;
        try {
            resource = loader.getResourceAsStream(configurationFile);
        } catch (Exception e) {
            logger.error("Cannot find probe-config.xml: " + e);
            throw e;
        }
        if (resource == null) {
            logger.error("Cannot find probe-config.xml");
            return;
        }
        logger.debug("Parsing probe-config.xml");
        loadConfigFile(resource);
    }

    private void saveConfigFile(OutputStream resource) throws Exception {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

            // rebuild probeConfig from current configuration
            probeConfig = new ProbeConfig();

            // outputs
            ArrayList<Output> olist = new ArrayList<Output>();
            for (JasmineOutput jo : outputs.values()) {
                Output output = new Output();
                output.setId(jo.getName());
                output.setType(jo.getType());
                for (String key : jo.getProperties().keySet()) {
                    Paramvalue pv =  new Paramvalue();
                    pv.setName(key);
                    pv.setValue(jo.getProperties().get(key));
                    output.getParam().add(pv);
                }
                output.setDefault(false);
                for (String on : defaultOutputList) {
                    if (on.equals(jo.getName())) {
                        output.setDefault(true);
                    }
                }
                olist.add(output);
            }
            probeConfig.setOutput(olist);

            // targets
            ArrayList<Target> tlist = new ArrayList<Target>();
            for (JasmineTarget jt : targets.values()) {
                Target target = new Target();
                target.setId(jt.getName());
                target.setUrl(jt.getUrl());
                tlist.add(target);
            }
            probeConfig.setTarget(tlist);

            // indicators
            ArrayList<Indicator> ilist = new ArrayList<Indicator>();
            for (JasmineIndicator ji : indicators.values()) {
                Indicator indic = new Indicator();
                indic.setId(ji.getName());
                indic.setType(ji.getType());
                String scale = new Long(ji.getScale()).toString();
                indic.setScale(new BigInteger(scale));
                for (String key : ji.getProperties().keySet()) {
                    Paramvalue pv =  new Paramvalue();
                    pv.setName(key);
                    pv.setValue(ji.getProperties().get(key));
                    indic.getParam().add(pv);
                }
                ilist.add(indic);
            }
            probeConfig.setIndicator(ilist);

            // probes
            ArrayList<Probe> plist = new ArrayList<Probe>();
            for (JasmineProbe jp : probes.values()) {
                Probe pdata = new Probe();
                pdata.setId(jp.getId());
                String period = new Long(jp.getPeriod()).toString();
                pdata.setPeriod(new BigInteger(period));
                ArrayList<String> onlist = new ArrayList<String>();
                for (String output : jp.getOutputList()) {
                    onlist.add(output);
                }
                pdata.setOutput(onlist);
                ArrayList<String> inlist = new ArrayList<String>();
                for (String ind : jp.getIndicatorList()) {
                    inlist.add(ind);
                }
                pdata.setIndicator(inlist);
                switch (jp.getStatus()) {
                case JasmineProbe.PROBE_FAILED:
                case JasmineProbe.PROBE_STOPPED:
                    pdata.setStatus(StatusType.STOPPED);
                    break;
                case JasmineProbe.PROBE_RUNNING:
                case JasmineProbe.PROBE_STARTED:
                    pdata.setStatus(StatusType.STARTED);
                    break;
                }
                plist.add(pdata);
            }
            probeConfig.setProbe(plist);

            // Build the xml file
            marshaller.marshal(probeConfig, resource);
        } catch (Exception e) {
            logger.error("Error while writing probe-config.xml: " + e);
            throw new JasmineProbeException("Error while writing probe-config.xml: " + e);
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    private synchronized void loadConfigFile(InputStream resource) throws JasmineProbeException {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            probeConfig = (ProbeConfig) unmarshaller.unmarshal(resource);

            // Create JasmineOutput objects
            for (Output output : probeConfig.getOutput()) {
                JasmineOutput jo = new JasmineOutput();
                jo.setName(output.getId());
                jo.setType(output.getType());
                if (output.getDefault() == true) {
                    defaultOutputList.add(output.getId());
                }
                Map<String, String> props = new HashMap<String, String>();
                for (Paramvalue param : output.getParam()) {
                    props.put(param.getName(), param.getValue());
                }
                jo.setProperties(props);
                createOutputInternal(jo, false);
            }

            // Create JasmineTarget objects
            for (Target target : probeConfig.getTarget()) {
                JasmineTarget jt = new JasmineTarget();
                jt.setName(target.getId());
                jt.setUrl(target.getUrl());
                Map<String, String> props = new HashMap<String, String>();
                for (Paramvalue param : target.getParam()) {
                    props.put(param.getName(), param.getValue());
                }
                jt.setProperties(props);
                createTarget(jt);
            }

            // Create JasmineIndicator objects
            for (Indicator indic : probeConfig.getIndicator()) {
                JasmineIndicator ji = new JasmineIndicator();
                ji.setName(indic.getId());
                ji.setType(indic.getType());
                ji.setScale(indic.getScale().intValue());
                Map<String, String> props = new HashMap<String, String>();
                for (Paramvalue param : indic.getParam()) {
                    props.put(param.getName(), param.getValue());
                }
                ji.setProperties(props);
                createIndicatorInternal(ji, false);
            }

            // Create JasmineProbe objects
            for (Probe probe : probeConfig.getProbe()) {
                JasmineProbe jp = new JasmineProbe();
                jp.setId(probe.getId());
                jp.setPeriod(probe.getPeriod().intValue());
                jp.setOutputList(probe.getOutput());
                jp.setIndicatorList(probe.getIndicator());
                if (probe.getStatus() != null && probe.getStatus().value().equals("started")) {
                    jp.setStatus(JasmineProbe.PROBE_TOSTART);
                }
                createProbe(jp);
            }

        } catch (Exception e) {
            logger.error("Error in probe-config.xml: " + e);
            throw new JasmineProbeException("Error in probe-config.xml: " + e);
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    /**
     * Test if an Output is used at least in one ProbeInstance.
     *
     * @param output the output to check
     * @param checkrun  If true, check only if used by running probes
     * @return true if output used in at least 1 probe.
     */
    private synchronized boolean isOutputUsed(JasmineOutput output, boolean checkrun) {
        for (JasmineProbe probe : probes.values()) {
            for (String name : probe.getOutputList()) {
                if (name.equals(output.getName())) {
                    if (!checkrun || (probe.getStatus() != JasmineProbe.PROBE_STOPPED)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Test if a Target is used
     *
     * @param target the target to check
     * @return true if target used
     */
    private boolean isTargetUsed(JasmineTarget target) {
        for (JasmineProbe probe : probes.values()) {
            if (probe.getTargetList().contains(target)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Test if an Indicator is used at least in one ProbeInstance.
     * @param indicator the indicator to check
     * @param checkrun  If true, check only if used by running probes
     * @return true if indic used in at least 1 probe.
     */
    private synchronized boolean isIndicatorUsed(JasmineIndicator indicator, boolean checkrun) {
        for (JasmineProbe probe : probes.values()) {
            for (String indic : getRecursiveIndicatorList(probe)) {
                if (indic.equals(indicator.getName())) {
                    if (!checkrun || (probe.getStatus() != JasmineProbe.PROBE_STOPPED)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Notify a state change for that probe
     *
     * @param running probe
     * @param state   See JasmineProbe.state
     * @param message Error message
     */
    public synchronized void setProbeState(JProbe running, int state, String message) {
        logger.debug("state =" + state);
        running.setStatus(state);
        running.setError(message);
        for (JasmineProbeListener listener : probeListeners) {
            listener.notifyEvent(running.getProbeDef());
        }
    }

    /**
     * Start probes waiting to be started.
     */
    private synchronized void startWaitingProbes() {

        for (JasmineProbe probe : probes.values()) {
            if (probe.getStatus() == JasmineProbe.PROBE_TOSTART) {
                logger.debug(probe.getId() + " trying to start");
                JProbe running = new JProbe(probe, this);
                try {
                    running.resume();
                } catch (JasmineProbeException e) {
                    logger.warn("Cannot start probe now: " + probe.getId());
                    continue;
                }
                runnings.put(probe.getId(), running);
            }
        }
    }

    private synchronized List<String> getRecursiveIndicatorList(JasmineProbe probe) {
        List<String> ret = new ArrayList<String>();
        for (String iname : probe.getIndicatorList()) {
            ret.add(iname);
            JasmineIndicator indic = indicators.get(iname);
            JasmineCollectorService jcs = collectorservices.get(indic.getType());
            if (jcs != null) {
                List<String> full = jcs.getDependantIndicators(indic);
                ret.addAll(full);
            }
        }
        return ret;
    }
}
