/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineProbeCommand.java 8785 2011-09-01 13:16:54Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

import org.apache.commons.beanutils.PropertyUtils;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.JasminePropertyInfo;

import com.beust.jcommander.JCommander;
import com.beust.jcommander.ParameterException;

/**
 * Common code for all JasmineProbe Commands.
 * Contains command line parsing.
 * @author danesa
 */
public abstract class JasmineProbeCommand {


    public JasmineProbeCommand() {
        // In case there are no parameters
        params = new JasmineProbeCommandParams();
    }

    public abstract String getName();

    public abstract JasmineProbeManager getProbeManager();

    private JCommander commander = null;

    /**
     * Contains the command parameters set by JCommander.
     */
    private JasmineProbeCommandParams params = null;

    public void setParams(JasmineProbeCommandParams params) {
        this.params = params;
    }

    public JasmineProbeCommandParams getParams() {
        return params;
    }

    /**
     * Called by all the commands that have parameters
     * @param line
     * @param out
     * @param err
     * @return
     */
    protected boolean parseOptions(final String line, PrintStream out, PrintStream err) {

        // Extract command name to get command line arguments
        String[] args = getArgs(line.substring(getName().length()).trim());

        try {
            commander = new JCommander(params, args);
            // JCommander program name set to command name allows to have a proper usage message
            commander.setProgramName(getName());
        } catch (ParameterException pe) {
            err.println(pe.getMessage());
            return false;
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }

        return true;
    }

    /**
     * By default, there are no properties.
     * To be extended by the sub-class corresponding to commands with properties.
     * @return
     */
    public String getProperties() {
        return null;
    }

    /**
     * Returns the usage string for the command.
     * The usage string is a short string that illustrates how to use the command on the command line.
     * This information is used when generating command help information.
     * @return the usage string for the command.
     */
    public String getUsage() {
        if (commander == null) {
            // create a fake object
            String[] args = getArgs(getRequiredOptions());
            commander = new JCommander(params, args);
            // JCommander program name set to command name allows to have a proper usage message
            commander.setProgramName(getName());
        }
        StringBuilder sb = new StringBuilder();
        commander.usage(sb);
        String props = getProperties();
        if (props != null) {
            sb.append("\n");
            sb.append(props);
        }
        // remove "Usage: ", already displayed by Felix.
        return new String(sb).substring(7);
    }

    /**
     * To be extended by commands having required options
     * @return
     */
    public String getRequiredOptions() {
        return "";
    }

    /**
     * Construct a properties Map, where the keys are property names provided by propsInfo and the values of the
     * properties should be read from the command parameters
     * @param propsInfo contains the properties description
     * @param params contains the command parameters
     * @return the resulting properties map
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
    protected Map<String, String> parseProperties(Collection<JasminePropertyInfo> propsInfo, JasmineProbeCommandParams params) throws JasmineProbeException {
        Map<String, String> properties = new HashMap<String, String>();
        for (JasminePropertyInfo propInfo : propsInfo) {
            // get property name
            String key = propInfo.getName();
            // get property value from the params object
            String value = null;
            try {
                value = (String) PropertyUtils.getSimpleProperty(params, key);
            } catch (Exception e) {
                throw new JasmineProbeException(getName() + " command: cannot get options for " + key + " property (" + e.toString() + ")");
            }
            properties.put(key, value);
            //System.out.println("- " + key + " = " + value);
        }
        return properties;
    }

    private String[] getArgs(final String argsLine) {

        String[] args = null;

        if (argsLine.length() == 0) {
            args = new String[0];
        } else {
            // Parse parameters
            StringTokenizer st = new StringTokenizer(argsLine);
            args = new String[st.countTokens()];
            int i = 0;
            while (st.hasMoreTokens()) {
                args[i++] = st.nextToken();
            }
        }

        return args;
    }

    /**
     * Verify that the command parameter has the following format : paramName=paramValue
     * @param param
     * @return true if format ok, false if the format is wrong
     */
    protected boolean validParam(String param) {
        int indexEqual = param.indexOf("=");
        if (indexEqual == -1) {
            // there is no = sign
            return false;
        } else if (indexEqual < 1) {
            // there is no paramName before the =
            return false;
        } else {
            try {
                param.substring(indexEqual + 1);
                return true;
            } catch (IndexOutOfBoundsException obe) {
                // there is no paramValue after the =
                return false;
            }
        }
    }

    protected void createIndicator(String commandName, JasmineIndicator indicator, PrintStream out, PrintStream err) {
        try {
            getProbeManager().createIndicator(indicator);
            out.println(commandName + " " + indicator.getName() +  " was successfull");
        } catch (JasmineProbeException e) {
            err.println(commandName + " " + indicator.getName() + " failed");
            err.println(e);
            return;
        }
    }

}
