/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JProbe.java 8917 2011-09-16 12:09:23Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.manager;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.manager.internal.JProbeManager;
import org.ow2.jasmine.probe.outer.JasmineOuter;
import org.ow2.jasmine.probe.outer.JasmineOuterException;
import org.ow2.jasmine.probe.outer.JasmineOuterService;
import org.ow2.jasmine.probe.probescheduler.SchedulerException;
import org.ow2.jasmine.probe.probescheduler.TaskReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * This object represents a Running Probe.
 *
 * @author durieuxp
 */
public class JProbe implements Runnable {

    protected static Log logger = LogFactory.getLog(JProbe.class);

    /**
     * Probe definition + probe state
     */
    private JasmineProbe probeDef;

    /**
     * Ref to the JProbeManager
     */
    private JProbeManager probeMgr;

    /**
     * Task returned by the scheduler
     */
    private TaskReference task;

    /**
     * List of Collectors used by the probe.
     */
    private List<JasmineCollector> collectors;

    /**
     * List of Outers used by the probe.
     */
    private List<JasmineOuter> outers;

    /**
     * Constructor
     */
    public JProbe(JasmineProbe def, JProbeManager mgr) {
        probeDef = def;
        probeMgr = mgr;
        def.setError("");
    }

    // -----------------------------------------------------------------------------
    // Getters and Setters
    // -----------------------------------------------------------------------------

    public JasmineProbe getProbeDef() {
        return probeDef;
    }

    /**
     * Get the Probe status, kept in the JasmineProbe object for convenience.
     *
     * @return the probe status
     */
    public int getStatus() {
        return probeDef.getStatus();
    }

    /**
     * Set the Probe status, kept in the JasmineProbe object for convenience.
     *
     * @param state the probe status
     */
    public void setStatus(int state) {
        probeDef.setStatus(state);
    }

    /**
     * Set the Probe error message, kept in the JasmineProbe object for convenience.
     *
     * @param mess the error message
     */
    public void setError(String mess) {
        probeDef.setError(mess);
    }

    // -----------------------------------------------------------------------------
    // Interface for the ProbeManager
    // -----------------------------------------------------------------------------

    /**
     * Start the Probe, or resume it if it was previously suspended.
     */
    public void resume() throws JasmineProbeException {
        logger.debug("");

        buildOuterList();
        buildCollectorList();

        // Start Collectors
        for (JasmineCollector collector : collectors) {
            collector.startPolling();
        }

        // Start the scheduler
        try {
            // period must be given in milliseconds
            long ms = probeDef.getPeriod() * 1000L;
            task = probeMgr.getScheduler().schedulePeriodicTask(null, this, 0, ms);
            logger.debug(probeDef.getId() + " should be started now");
            // Set the Probe status
            probeDef.setStatus(JasmineProbe.PROBE_STARTED);
        } catch (SchedulerException e) {
            logger.error("Error when trying to start probe : {0}: {1}", probeDef.getId(), e);
            probeMgr.setProbeState(this, JasmineProbe.PROBE_FAILED, e.getMessage());
        }
    }

    /**
     * Suspend the Probe.
     * It may be restarted later by resume().
     */
    public void suspend() {
        logger.debug("");

        // Stop the scheduler
        if (task != null) {
            task.cancel(true);
            task = null;
        }

        // Stop Collectors
        for (JasmineCollector collector : collectors) {
            collector.stopPolling();
        }

        // Remove lists to avoid reusing an obsolete object
        outers = new ArrayList<JasmineOuter>();
        collectors = new ArrayList<JasmineCollector>();

        // Set the Probe status
        probeDef.setStatus(JasmineProbe.PROBE_STOPPED);
    }

    // -----------------------------------------------------------------------------
    // Runnable implementation
    // -----------------------------------------------------------------------------

    /**
     * Called by the Scheduler to make a polling on all indicators
     */
    public void run() {
        logger.debug("");

        // Init a JasmineProbeResult with the probe Id
        JasmineProbeResult result = new JasmineProbeResult(probeDef.getId());

        // Get results for each indicator
        for (JasmineCollector collector : collectors) {
            try {
                JasmineIndicatorValue val = collector.getLastResult();
                if (val != null) {
                    result.addValue(val);
                } else {
                    logger.debug("No result found for " + collector.getName());
                }
            } catch (JasmineCollectorException e) {
                logger.warn("Cannot get results on " + collector.getName() + " :" + e);
            } catch (NullPointerException e) {
                logger.error("NPE in run method");
                e.printStackTrace();
            }
        }

        // Publish the results on each Outer
        for (JasmineOuter outer : outers) {
            outer.publishData(result);
        }
    }

    // -------------------------------------------------------------------------------
    // private methods
    // -------------------------------------------------------------------------------

    /**
     * Build the Outer list
     *
     * @throws JasmineProbeException
     */
    private void buildOuterList() throws JasmineProbeException {
        outers = new ArrayList<JasmineOuter>();
        for (String outputname : probeDef.getOutputList()) {
            JasmineOutput output = probeMgr.getOutput(outputname);
            if (output == null) {
                logger.error("unknown output : " + outputname);
                throw new JasmineProbeException("unknown output : " + outputname);
            }
            // Find the appropriate OuterService
            JasmineOuterService jos = probeMgr.getOuterService(output.getType());
            if (jos == null) {
                logger.error("No OuterService found for {0} having type {1}", outputname, output.getType());
                throw new JasmineProbeException("No OuterService found for " + outputname);
            }
            // Get the Outer from the OuterService
            JasmineOuter jo = null;
            try {
                jo = jos.getOuter(output);
            } catch (JasmineOuterException e) {
                logger.error("Could not get the Outer for " + outputname + ": " + e);
                throw new JasmineProbeException("Could not get the Outer for " + outputname);
            }
            outers.add(jo);
        }
    }

    /**
     * Build the Collector list
     *
     * @throws JasmineProbeException
     */
    private void buildCollectorList() throws JasmineProbeException {
        collectors = new ArrayList<JasmineCollector>();
        for (String indicname : probeDef.getIndicatorList()) {
            JasmineIndicator indic = probeMgr.getIndicator(indicname);
            if (indic == null) {
                logger.error("unknown indicator : " + indicname);
                throw new JasmineProbeException("unknown indicator : " + indicname);
            }
            // Find the appropriate CollectorService
            JasmineCollectorService jcs = probeMgr.getCollectorService(indic.getType());
            if (jcs == null) {
                logger.error("No CollectorService found for {0} having type {1}", indicname, indic.getType());
                throw new JasmineProbeException("No CollectorService found for " + indicname);
            }
            // Get the Collector from the CollectorService
            JasmineCollector coll = null;
            try {
                coll = jcs.getCollector(indic, probeDef.getPeriod(), probeDef.getId());
            } catch (JasmineCollectorException e) {
                logger.error("Could not get the Collector for " + indicname + ": " + e);
                throw new JasmineProbeException("Could not get the Collector for " + indicname);
            }
            collectors.add(coll);
        }
    }

}
