/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ProbeListCommand.java 8830 2011-09-09 09:44:03Z durieuxp $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.Collection;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.JasmineProbe;

/**
 * Implements probe-list command that list the name of all the created probes.
 * @author danesa
 */
@Component(name="ProbeListCommand")
@Provides
public class ProbeListCommand extends JasmineProbeCommand implements Command {

        /**
         * command name
         */
        private String name = "probe-list";

        /**
         * command description
         */
        private String description = "List all JASMINe Probes, or show a particular Probe";


        // ----------------------------------------------------------------------------
        // ipojo management
        // ----------------------------------------------------------------------------

        @Requires
        JasmineProbeManager probeManager = null;

        // ----------------------------------------------------------------------------
        // constructors
        // ----------------------------------------------------------------------------

        /**
         * Default Constructor
         */
        public ProbeListCommand() {
            super();
            setParams(new ProbeListCommandParams());
        }

        // ----------------------------------------------------------------------------
        // org.apache.felix.shell.Command implementation
        // ----------------------------------------------------------------------------

        /**
         * Returns the name of the command that is implemented by the interface.
         * The command name should not contain whitespace and should also be unique.
         * @return the name of the command.
         */
        public String getName() {
            return name;
        }

        /**
         * Returns a short description of the command; this description should be as short as possible.
         * This information is used when generating the command help information.
         * @return a short description of the command.
         */
        public String getShortDescription() {
            return description;
        }

        public JasmineProbeManager getProbeManager() {
            return probeManager;
        }

        /**
         * Executes the command using the supplied command line, output print stream, and error print stream.
         * @param line the complete command line, including the command name.
         * @param out  the print stream to use for standard output.
         * @param err  the print stream to use for standard error.
         */
        public void execute(String line, PrintStream out, PrintStream err) {
            boolean validOptions = parseOptions(line, out, err);
            if (!validOptions) {
                err.println(getUsage());
                return;
            }

            ProbeListCommandParams params = ((ProbeListCommandParams) getParams());

            // get command arguments
            String probeName = params.getName();
            boolean verbose = params.getVerbose();
            if (probeName == null) {
                listProbes(verbose, out);
            } else {
                showProbe(probeName, out, err);
            }

            params.reset();
        }

        private void listProbes(boolean verbose, PrintStream out) {
            Collection<JasmineProbe> probes = probeManager.getProbes();
            if (probes.isEmpty()) {
                out.println("There are no JasmineProbes");
                return;
            }
            for (JasmineProbe probe : probes) {
                if (verbose) {
                    out.println(probe.toString());
                    out.println();
                } else {
                    out.println(probe.getId());
                }
            }
        }

    private void showProbe(String probeName, PrintStream out, PrintStream err) {
        try {
            JasmineProbe probe = probeManager.getProbe(probeName);
            out.println(probe.toString());
        } catch (JasmineProbeException e) {
            err.println("There is no JasmineProbe named " + probeName);
        }
    }
}
