/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JCollector.java 9650 2012-01-23 08:40:59Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.collectors;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.*;

/**
 * This object represents a Collector in use.
 * It is abstract, because implementation may be various.
 * These implementations are in separate modules.
 * @author durieuxp
 * @author danesa
 */
public abstract class JCollector implements JasmineCollector {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * Indicator definition
     */
    protected JasmineIndicator indicator;

    /**
     * period of polling in seconds
     */
    protected int period;

    /**
     * Probe id
     */
    protected String probeId;

    /**
     * reference to the ProbeManager.
     */
    protected ProbeManager probeManager = null;

    /**
     * Host name
     */
    protected String hostName = null;

    /**
     * Marker allowing to a probe that uses this collector
     * to get rid of it.
     */
    private boolean removed = false;

    /**
     * Mark collector as removed.
     * Called when a JasmineCollectorService stops.
     */
    public void remove() {
        logger.debug("Mark removed " + indicator.getName() + " in probe " + probeId);
        removed = true;
    }

    /**
     * @return true if collector marked as removed.
     */
    public boolean isRemoved() {
        return removed;
    }

    /**
     * Constructor
     */
    public JCollector(String probeId, JasmineIndicator def, int period) {
        this.indicator = def;
        this.period = period;
        this.probeId = probeId;
        try {
            this.hostName = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            logger.warn("Cannot get host name for indicator {0} ({1})", def.getName(), e.toString());
        }
    }

    public JasmineIndicator getIndicator() {
        return indicator;
    }

    /**
     * Set the probeManager ref when it is needed, for some Collectors.
     * @param probeManager
     */
    public void setProbeManager(ProbeManager probeManager) {
        this.probeManager = probeManager;
    }

    protected String getDomainName() {
        return probeManager.getDomainName();
    }

    protected String getServerName() {
        return probeManager.getServerName();
    }

    // ----------------------------------------------------------
    // JasmineCollector implementation
    // ----------------------------------------------------------

    /**
     * Return the name of the indicator corresponding to this Collector
     */
    public String getIndicatorName() {
        return getIndicator().getName();
    }

    /**
     *
     * @return Return the Id of the probe corresponding to this Collector
     */
    public String getProbeId() {
        return probeId;
    }

    /**
     * return the last result for this indicator
     * @return JasmineIndicatorValue or null if no value available
     * @throws JasmineCollectorException
     */
    public abstract JasmineIndicatorValue getLastResult() throws JasmineCollectorException;

    /**
     * Stop polling
     */
    public abstract void stopPolling();

    /**
     * Restart polling
     */
    public abstract void startPolling();

    // ------------------------------------------------------------------------
    // Utility methods
    // ------------------------------------------------------------------------

    /**
     * Implements the "max" operation.
     * @param coll values to aggregate
     * @return value
     * @throws org.ow2.jasmine.probe.collector.JasmineCollectorException
     */
    private Number max(Collection<Number> coll) throws JasmineCollectorException {
        Number ret = null;
        for (Number val : coll) {
            if (ret == null) {
                ret = val;
            } else {
                Number diff = diffValues(val, ret);
                if (castLong(diff)) {
                    if (diff.longValue() > 0) {
                        ret = val;
                    }
                } else if (diff instanceof Float) {
                    if (diff.floatValue() > 0) {
                        ret = val;
                    }
                }  else if (diff instanceof Double) {
                    if (diff.doubleValue() > 0) {
                        ret = val;
                    }
                }
            }
        }
        return ret;
    }

    /**
     * Return the max value in a collection of JSRs.
     * @param jsrs collection of JSRs to be aggregated
     * @return the maximum numeric value
     * @throws JasmineCollectorException
     */
    protected Number maxJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        try {
            Collection<Number> numList = getNumList(jsrs);
            return max(numList);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Aggregation of incompatible types (max operation expects only Number values)");
        }
    }

    /**
     * Implements the "min" operation.
     * @param coll values to aggregate
     * @return value
     * @throws org.ow2.jasmine.probe.collector.JasmineCollectorException
     */
    private Number min(Collection<Number> coll) throws JasmineCollectorException {
        Number ret = null;
        for (Number val : coll) {
            if (ret == null) {
                ret = val;
            } else {
                Number diff = diffValues(val, ret);
                if (castLong(diff)) {
                    if (diff.longValue() < 0) {
                        ret = val;
                    }
                } else if (diff instanceof Float) {
                    if (diff.floatValue() < 0) {
                        ret = val;
                    }
                }  else if (diff instanceof Double) {
                    if (diff.doubleValue() < 0) {
                        ret = val;
                    }
                }
            }
        }
        return ret;
    }

    /**
     * Return the min value in a collection of JSRs.
     * @param jsrs collection of JSRs to be aggregated
     * @return the minimum numeric value
     * @throws JasmineCollectorException
     */
    protected Number minJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        try {
            Collection<Number> numList = getNumList(jsrs);
            return min(numList);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Aggregation of incompatible types (min operation expects only Number values)");
        }
    }

    /**
     * Implements the "average" operation.
     * @param coll values to aggregate
     * @return value
     * @throws org.ow2.jasmine.probe.collector.JasmineCollectorException
     */
    private Number average(Collection<Number> coll) throws JasmineCollectorException {
        Number ret = addValues(coll);
        int nbval = coll.size();
        if (castLong(ret)) {
            return ((float) ret.doubleValue()) / nbval;
        } else if (ret instanceof Float) {
            return ret.floatValue() / nbval;
        } else if (ret instanceof Double) {
            double dret = ret.doubleValue();
            return dret / nbval;
        } else {
            throw new JasmineCollectorException("Unsupported values in a collection of numbers for average operation");
        }
    }

    /**
     */
    protected Number averageJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        try {
            Collection<Number> numList = getNumList(jsrs);
            return average(numList);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Aggregation of incompatible types (min operation expects only Number values)");
        }
    }

    /**
     * Make a diff between 2 values
     * @param v1
     * @param v2
     * @return v1 - v2
     */
    protected long diffLongValues(Number v1, Number v2) throws JasmineCollectorException {
        return (longValue(v1) - longValue(v2));
    }


    /**
     * Make a diff between 2 same type values of a list containing 2 JSRs.
     * @param jsrs the list of JSRs
     * @return Object of same type representing the diff
     */
    protected Number diffJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        if (jsrs.size() != 2) {
            throw new JasmineCollectorException("Diff correlation with more that 2 values not supported");
        }
        JasmineSingleResult jsr1 = null;
        JasmineSingleResult jsr2 = null;
        for (JasmineSingleResult jsr : jsrs) {
            if (jsr1 == null) {
                // first JSR
                jsr1 = jsr;
            } else {
                jsr2 = jsr;
            }
        }
        try {
            Number num1 = (Number) jsr1.getValue();
            Number num2 = (Number) jsr2.getValue();
            return diffValues(num1, num2);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Correlation of incompatible types (diff operation expects only Number values)");
        }
    }

    /**
     * Make a div between 2 values
     * @param v1
     * @param v2
     * @return v1 / v2
     */
    protected float divideLongValues(Number v1, Number v2) throws JasmineCollectorException {
        if (longValue(v2) == 0) {
            // Cannot make the divide: Choose to return 1 for now.
            // TODO
            return 1;
        }
        return (longValue(v1) / longValue(v2));
    }

    /**
     * Make a div between 2 values
     * @param v1
     * @param v2
     * @return v1 / v2
     */
    protected Number divideValues(Number v1, Number v2) throws JasmineCollectorException {
        // Check that v2 is not null
        if (castLong(v2)) {
            if (longValue(v2) == 0) {
                // Cannot make the divide: Choose to return 1 for now.
                // TODO
                return 1;
            }
        } else if (v2 instanceof Float || v2 instanceof Double) {
            if (v2.doubleValue() == 0) {
                // Cannot make the divide: Choose to return 1 for now.
                // TODO
                return 1;
            }
        }

        return (v1.doubleValue() / v2.doubleValue());
    }

    /**
     * Make a division of 2 values of 2 JSRs contained in a given JSRs list
     * @param jsrs the list of  2 JSRs
     * @return Object of same type representing the division
     */
    protected Number divideJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        if (jsrs.size() != 2) {
            throw new JasmineCollectorException("Diff correlation with more that 2 values not supported");
        }
        JasmineSingleResult jsr1 = null;
        JasmineSingleResult jsr2 = null;
        for (JasmineSingleResult jsr : jsrs) {
            if (jsr1 == null) {
                // first JSR
                jsr1 = jsr;
            } else {
                jsr2 = jsr;
            }
        }
        try {
            Number num1 = (Number) jsr1.getValue();
            Number num2 = (Number) jsr2.getValue();
            return divideValues(num1, num2);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Correlation of incompatible types (divide operation expects only Number values)");
        }
    }

    /**
     * Add a list of values of one of the following types: short, integer, long, float or doubles
     * @param values to sum
     * @return Number representing the sum of all values
     */
    protected Number addValues(Collection<Number> values) throws JasmineCollectorException {
        long longRes = 0L;
        float floatRes = 0;
        double doubleRes = 0;
        boolean isFloat = false;
        boolean isDouble = false;
        for (Number o : values) {
            if (castLong(o)) {
                longRes += o.longValue();
            } else if (o instanceof Float) {
                isFloat = true;
                floatRes += o.floatValue();
            } else if (o instanceof Double) {
                isDouble = true;
                doubleRes += o.doubleValue();
            } else {
                throw new JasmineCollectorException("This Number cannot be converted to a long or to a float: " + o);
            }
        }
        if (isDouble) {
            double res = doubleRes + floatRes + longRes;
            return (Number) res;
        } else if (isFloat) {
            float res = floatRes + longRes;
            return (Number) res;
        } else {
            return (Number) longRes;
        }
    }

    /**
     * Make a difference between 2 values of one of the following types: short, integer, long, float or doubles
     * @param v1
     * @param v2
     * @return v1 - v2
     */
    protected Number diffValues(Number v1, Number v2) throws JasmineCollectorException {
        if (castLong(v1) && castLong(v2)) {
            return (Number) (longValue(v1) - longValue(v2));
        } else {
            if (v1 instanceof Float || v1 instanceof Double) {
                double d_v1 = v1.doubleValue();
                if (castLong(v2)) {
                    return (Number) (d_v1 - v2.longValue());
                } else if (v2 instanceof Float) {
                    return (Number) (d_v1 - v2.floatValue());
                } else if (v2 instanceof Double) {
                    return (Number) (d_v1 - v2.doubleValue());
                } else {
                    throw new JasmineCollectorException("This Number cannot be converted to a long or to a float: " + v2);
                }
            } else if (castLong(v1)) {
                if (v2 instanceof Float) {
                    return (Number) (v1.floatValue() - v2.floatValue());
                } else if (v2 instanceof Double) {
                    return (Number) (v1.doubleValue() - v2.doubleValue());
                } else {
                    throw new JasmineCollectorException("This Number cannot be converted to a long or to a float: " + v2);
                }
            } else {
                 throw new JasmineCollectorException("This Number cannot be converted to a long or to a float: " + v1);
            }
        }
    }
    /**
     * Add a list of values of same type (short, int, long or float casted to long)
     * @param values
     * @return Object a long representing the sum of all the values
     */
    protected long addLongValues(Collection<Number> values) throws JasmineCollectorException {
        long ret = 0L;
        for (Number o : values) {
            ret += longValue(o);
        }
        return ret;
    }

    /**
     * Add the values of a list of JSRs.
     * @param jsrs the list of JSRs
     * @return Number representing the sum of all the JSR values casted to long
     */
    protected long addLongJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        try {
            Collection<Number> numList = getNumList(jsrs);
            return addLongValues(numList);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Correlation of incompatible types (add operation expects only Number values)");
        }
    }

    /**
     * Add the values of a list of JSRs.
     * @param jsrs the list of JSRs
     * @return Number representing the sum of all the JSR values casted to long, float or double
     */
    protected Number addJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        try {
            Collection<Number> numList = getNumList(jsrs);
            return addValues(numList);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Correlation of incompatible types (add operation expects only Number values)");
        }
    }

    /**
     * Multiply a list of values of same type
     * @param values
     * @return Object of same type representing the product of all values
     */
    protected long multiplyLongValues(Collection<Number> values) throws JasmineCollectorException {
        long ret = 1L;
        for (Number o : values) {
            ret *= longValue(o);
        }
        return ret;
    }

    /**
     * Multiply a list of values of one of the following types: short, integer, long, float or doubles
     * @param values to multiply
     * @return Object object representing the product of all values
     */
    protected Number multiplyValues(Collection<Number> values) throws JasmineCollectorException {
        long longRes = 1L;
        float floatRes = 1;
        double doubleRes = 1;
        boolean isFloat = false;
        boolean isDouble = false;
        for (Number o : values) {
            if (castLong(o)) {
                longRes = longRes * o.longValue();
            } else if (o instanceof Float) {
                isFloat = true;
                floatRes = floatRes * o.floatValue();
            }  else if (o instanceof Double) {
                isDouble = true;
                doubleRes = doubleRes * o.doubleValue();
            } else {
                throw new JasmineCollectorException("This Number cannot be converted to a long or to a float: " + o);
            }

        }
        if (isDouble) {
            double res = doubleRes * floatRes * longRes;
            return (Number) res;
        } else if (isFloat) {
            float res = floatRes * longRes;
            return (Number) res;
        } else {
            return (Number) longRes;
        }
    }

    /**
     * Multiply the values of a list of JSRs.
     * @param jsrs the list of JSRs
     * @return Object of same type representing the product of all the JSR values.
     * @throws JasmineCollectorException
     */
    protected Number multiplyJsr(Collection<JasmineSingleResult> jsrs) throws JasmineCollectorException {
        try {
            Collection<Number> numList = getNumList(jsrs);
            return multiplyValues(numList);
        } catch (ClassCastException ce) {
            throw new JasmineCollectorException("Correlation of incompatible types (multiply operation expects only Number values)");
        }
    }

    /**
     * Transforms a list of JSRs into a list of Numbers corresponding to each JSR's value.
     * @param jsrs the list of JSRs
     * @return the list of Numbers
     * @throws ClassCastException At least one vale is not a Number
     */
    private Collection<Number> getNumList(Collection<JasmineSingleResult> jsrs) throws ClassCastException {
        Collection<Number> numList = new ArrayList<Number>();
        for (JasmineSingleResult jsr : jsrs) {
                Number value = (Number) jsr.getValue();
                numList.add(value);
        }
        return numList;
    }


    /**
     * Try to cast a Number into a long value
     * @param nb
     * @return
     * @throws JasmineCollectorException
     */
    protected long longValue(Number nb) throws JasmineCollectorException {
        long val;
        if (nb instanceof Integer) {
            val = ((Integer)nb).longValue();
        } else if (nb instanceof Long) {
            val = ((Long)nb).longValue();
        } else if (nb instanceof Short) {
            val = ((Short)nb).longValue();
        } else {
            throw new JasmineCollectorException("This Number cannot be converted to a long: " + nb);
        }
        return val;
    }

    /**
     * Construct properties for a JSR by aggregating the properties of a collection of JSRs.
     * @param jsrs
     * @return
     */
    protected HashMap<String, String> jsrProps(Collection<JasmineSingleResult> jsrs) {
        HashMap<String, String> jsrProps = new HashMap<String, String>();
        for (JasmineSingleResult jsr : jsrs) {
            jsrProps.putAll(jsr.getProperties());
        }
        return jsrProps;
    }


    /**
     * Return true if a Number has one of the following types: short, int, long.
     * @param nb the Number
     * @return
     */
    private boolean castLong(Number nb) {
        if (nb instanceof Integer) {
            return true;
        } else if (nb instanceof Long) {
            return true;
        } else if (nb instanceof Short) {
            return true;
        } else {
            return false;
        }
    }

}
