/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JProbe.java 9794 2012-02-08 13:32:59Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.manager;

import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeResult;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.collectors.JCollector;
import org.ow2.jasmine.probe.manager.internal.JProbeManager;
import org.ow2.jasmine.probe.outer.JasmineOuter;
import org.ow2.jasmine.probe.outer.JasmineOuterException;
import org.ow2.jasmine.probe.outer.JasmineOuterService;
import org.ow2.jasmine.probe.outers.JOuter;
import org.ow2.jasmine.probe.probescheduler.SchedulerException;
import org.ow2.jasmine.probe.probescheduler.TaskReference;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * This object represents a Running Probe.
 *
 * @author durieuxp
 */
public class JProbe implements Runnable {

    protected static Log logger = LogFactory.getLog(JProbe.class);

    /**
     * Probe definition + probe state
     */
    private JasmineProbe probeDef;

    /**
     * Ref to the JProbeManager
     */
    private JProbeManager probeMgr;

    /**
     * Task returned by the scheduler
     */
    private TaskReference task;

    /**
     * List of Collectors used by the probe.
     */
    private List<JasmineCollector> collectors;

    /**
     * List of Outers used by the probe.
     */
    private List<JasmineOuter> outers;

    /**
     * Constructor
     */
    public JProbe(JasmineProbe def, JProbeManager mgr) {
        probeDef = def;
        probeMgr = mgr;
        def.setError("");
        outers = new ArrayList<JasmineOuter>();
        collectors = new ArrayList<JasmineCollector>();
    }

    // -----------------------------------------------------------------------------
    // Getters and Setters
    // -----------------------------------------------------------------------------

    public JasmineProbe getProbeDef() {
        return probeDef;
    }

    /**
     * Get the Probe status, kept in the JasmineProbe object for convenience.
     *
     * @return the probe status
     */
    public int getStatus() {
        return probeDef.getStatus();
    }

    /**
     * Set the Probe status, kept in the JasmineProbe object for convenience.
     *
     * @param state the probe status
     */
    public void setStatus(int state) {
        probeDef.setStatus(state);
    }

    /**
     * Set the Probe error message, kept in the JasmineProbe object for convenience.
     *
     * @param mess the error message
     */
    public void setError(String mess) {
        probeDef.setError(mess);
    }

    // -----------------------------------------------------------------------------
    // Interface for the ProbeManager
    // -----------------------------------------------------------------------------

    /**
     * @return true if probe is running
     */
    public boolean isRunning() {
        if (getStatus() == JasmineProbe.PROBE_RUNNING || getStatus() == JasmineProbe.PROBE_STARTED) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Start the Probe, or resume it if it was previously suspended.
     */
    public void resume() throws JasmineProbeException {
        logger.debug("");

        try {
            buildOuters();
        } catch (JasmineProbeException e) {
            outers.clear();
            throw e;
        }
        try {
            buildCollectors();
        } catch (JasmineProbeException e) {
            collectors.clear();
            throw e;
        }

        // Start Collectors
        for (JasmineCollector collector : collectors) {
            if (collector != null) {
                collector.startPolling();
            } else {
                logger.error("Cannot start probe {0}. Found null elements in its collectors list", probeDef.getId());
                probeMgr.setProbeState(this, JasmineProbe.PROBE_FAILED, "Found null in collector list");
                return;
            }
        }

        // Start the scheduler
        try {
            // period must be given in milliseconds
            long ms = probeDef.getPeriod() * 1000L;
            task = probeMgr.getScheduler().schedulePeriodicTask(null, this, 0, ms);
            logger.debug(probeDef.getId() + " should be started now");
            // Set the Probe status
            probeMgr.setProbeState(this, JasmineProbe.PROBE_RUNNING, null);
        } catch (SchedulerException e) {
            logger.error("Error when trying to start probe : {0}: {1}", probeDef.getId(), e);
            probeMgr.setProbeState(this, JasmineProbe.PROBE_FAILED, e.getMessage());
        }
    }

    /**
     * reset timer value with probe period
     */
    public void resetTimer() {
        logger.debug(probeDef.getId() + " timer reset to " +  probeDef.getPeriod());
        long ms = probeDef.getPeriod() * 1000L;
        task.changeTimeout(ms);
    }

    /**
     * Not only reset collectors list, but insure that collector instances are removed
     * from the collector service.
     */
    public void resetIndicators() {
        try {
            removeCollectors();
        } catch (JasmineProbeException e) {
            probeMgr.setProbeState(this, JasmineProbe.PROBE_FAILED, null);

        }
    }

    /**
     * Clear outers list.
     */
    public void resetOutputs() {
        removeOuters();
    }

    /**
     * Allow to change a probe's target.
     * TODO
     */
    public void resetTargets() {
        // TODO
    }

    /**
     * Suspend the Probe.
     * It may be restarted later by resume().
     * @return true if probe was running
     */
    public boolean suspend() {
        logger.debug("");

        boolean ret = isRunning();

        // Stop the scheduler
        if (task != null) {
            task.cancel(true);
            task = null;
        }

        // Flush the Outers
        for (JasmineOuter outer : outers) {
            outer.flushData();
        }

        // Stop Collectors
        for (JasmineCollector collector : collectors) {
            collector.stopPolling();
        }

        // Set the Probe status
        probeMgr.setProbeState(this, JasmineProbe.PROBE_STOPPED, null);

        return ret;
    }

    // -----------------------------------------------------------------------------
    // Runnable implementation
    // -----------------------------------------------------------------------------

    /**
     * Called by the Scheduler to make a polling on all indicators
     */
    public void run() {
        logger.debug("");

        // Init a JasmineProbeResult with the probe Id
        JasmineProbeResult result = new JasmineProbeResult(probeDef.getId());

        // Get results for each indicator
        for (JasmineCollector collector : collectors) {
            try {
                JasmineIndicatorValue val = collector.getLastResult();
                if (val != null) {
                    result.addValue(val);
                } else {
                    logger.debug("No result found for indicator {0} in probe {1}", collector.getIndicatorName(), probeDef.getId());
                }
            } catch (JasmineCollectorException e) {
                logger.warn("Cannot get results for {0} in probe {1} : " + e, collector.getIndicatorName(), probeDef.getId());
            } catch (NullPointerException e) {
                logger.error("NPE in run method", e);
            }
        }

        // Publish the results on each Outer
        for (JasmineOuter outer : outers) {
            outer.publishData(result);
        }
    }

    // -------------------------------------------------------------------------------
    // private methods
    // -------------------------------------------------------------------------------

    /**
     * Build the Outer list
     *
     * @throws JasmineProbeException
     */
    private void buildOuters() throws JasmineProbeException {
        if (!outers.isEmpty()) {
            boolean rebuild = false;
            for (JasmineOuter outer : outers) {
                if (outer.isRemoved()) {
                    rebuild = true;
                    break;
                }
            }
            if (rebuild) {
                outers.clear();
            } else {
                return;
            }
        }
        for (String outputname : probeDef.getOutputList()) {
            JasmineOutput output = probeMgr.getOutput(outputname);
            if (output == null) {
                logger.error("unknown output : " + outputname);
                throw new JasmineProbeException("unknown output : " + outputname);
            }
            // Find the appropriate OuterService
            JasmineOuterService jos = probeMgr.getOuterService(output.getType());
            if (jos == null) {
                logger.error("No OuterService found for {0} having type {1}", outputname, output.getType());
                throw new JasmineProbeException("No OuterService found for " + outputname);
            }
            // Get the Outer from the OuterService
            JasmineOuter jo = null;
            try {
                jo = jos.getOuter(output);
            } catch (JasmineOuterException e) {
                logger.error("Could not get the Outer for " + outputname + ": " + e);
                throw new JasmineProbeException("Could not get the Outer for " + outputname);
            }
            outers.add(jo);
        }
    }

    /**
     * Remove all the outers created for this probe.
     * Then clear the outers list.
     */
    private void removeOuters() {
        for (JasmineOuter outer : outers) {
            JasmineOutput output = ((JOuter) outer).getOutput();
            JasmineOuterService jos = probeMgr.getOuterService(output.getType());
            if (jos != null) {
                jos.removeOuters(output.getName());
            }
        }
        outers.clear();
    }

    /**
     * Build the Collector list
     *
     * @throws JasmineProbeException
     */
    private void buildCollectors() throws JasmineProbeException {
        if (!collectors.isEmpty()) {
            boolean rebuild = false;
            for (JasmineCollector coll : collectors) {
                if (coll.isRemoved()) {
                    rebuild = true;
                    break;
                }
            }
            if (rebuild) {
                for (JasmineCollector coll : collectors) {
                    if (!coll.isRemoved()) {
                        coll.stopPolling();
                        coll.remove();
                    }
                }
                collectors.clear();
            } else {
                return;
            }
        }
        for (String indicname : probeDef.getIndicatorList()) {
            JasmineIndicator indic = probeMgr.getIndicator(indicname);
            if (indic == null) {
                logger.error("unknown indicator : " + indicname);
                throw new JasmineProbeException("unknown indicator : " + indicname);
            }
            // Find the appropriate CollectorService
            JasmineCollectorService jcs = probeMgr.getCollectorService(indic.getType());
            if (jcs == null) {
                logger.error("No CollectorService found for {0} having type {1}", indicname, indic.getType());
                throw new JasmineProbeException("No CollectorService found for " + indicname);
            }
            // Get the Collector from the CollectorService
            JasmineCollector coll = null;
            try {
                coll = jcs.getCollector(indic, probeDef);
            } catch (JasmineCollectorException e) {
                logger.error("Could not get the Collector for " + indicname + ": " + e);
                throw new JasmineProbeException("Could not get the Collector for " + indicname);
            }
            collectors.add(coll);
        }
    }

    /**
     * Remove all the collectors created for this probe.
     * First the collectors corresponding to each indicator are stopped
     * and removed from the corresponding collector service's data structure.
     * Then, collectors list is cleared.
     * @throws JasmineProbeException
     */
    private void removeCollectors() throws JasmineProbeException {
        boolean ok = true;
        for (JasmineCollector coll : collectors) {
            String indicname = coll.getIndicatorName();
            JasmineIndicator indic;
            try {
                indic = probeMgr.getIndicator(indicname);
                JasmineCollectorService jcs = probeMgr.getCollectorService(indic.getType());
                jcs.removeCollectors(indicname, probeDef.getId());
            } catch (JasmineProbeException e) {
                logger.error("Cannot remove collectors for probe {0}. Seems that JasmineIndicator {1} does not exists anymore.", probeDef.getId(), indicname);
                ok = false;
            }
        }
        collectors.clear();
        if (!ok) {
            throw new JasmineProbeException("Errors when trying to remove the collectors for probe: " + probeDef.getId());
        }
    }
}
