/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JProbeManager.java 9778 2012-02-07 10:05:03Z durieuxp $
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.manager.internal;

import java.io.*;
import java.math.BigInteger;
import java.util.*;

import javax.management.ObjectName;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.felix.ipojo.annotations.*;
import org.osgi.framework.ServiceReference;
import org.ow2.jasmine.probe.*;
import org.ow2.jasmine.probe.collector.JasmineAggregateService;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.manager.JProbe;
import org.ow2.jasmine.probe.mbeans.AB;
import org.ow2.jasmine.probe.mbeans.CD;
import org.ow2.jasmine.probe.mbeans.TestOpenMBean;
import org.ow2.jasmine.probe.outer.JasmineOuterService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.jasmine.probe.probemanager.generated.*;
import org.ow2.jasmine.probe.probescheduler.SchedulerService;
import org.ow2.jonas.jmx.JmxService;
import org.ow2.jonas.lib.bootstrap.JProp;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implements the main service of the JasmineProbe module exposing the JasmineProbeManager interface
 * to the JasmineProbe module's clients.
 *
 * @author durieuxp
 */
@Component(name = "JasmineProbeManagerService")
@Provides
public class JProbeManager implements JasmineProbeManager, ProbeManager, JProbeManagerMXBean {

    protected static Log logger = LogFactory.getLog(JProbe.class);

    private ProbeConfig probeConfig = null;

    /**
     * JasmineProbe definitions
     */
    private Map<String, JasmineProbe> probedefs = new TreeMap<String, JasmineProbe>();

    /**
     * JasmineOutput definitions
     */
    private Map<String, JasmineOutput> outputs = new TreeMap<String, JasmineOutput>();

    /**
     * JasmineIndicator definitions
     */
    private Map<String, JasmineIndicator> indicators = new TreeMap<String, JasmineIndicator>();

    /**
     * JasmineTarget definitions
     */
    private Map<String, JasmineTarget> targets = new TreeMap<String, JasmineTarget>();

    /**
     *  probe objects
     */
    private Map<String, JProbe> probes = new HashMap<String, JProbe>();

    /**
     * List of the available Outer services.
     */
    private Map<String, JasmineOuterService> outerservices = new HashMap<String, JasmineOuterService>();

    /**
     * List of the available Collector services.
     */
    private Map<String, JasmineCollectorService> collectorservices = new HashMap<String, JasmineCollectorService>();

    /**
     * List of the custom available Aggregate services.
     */
    private Map<String, JasmineAggregateService> aggregateservices = new HashMap<String, JasmineAggregateService>();

    /**
     * Listeners for Probe changes
     */
    private List<JasmineProbeListener> probeListeners = new ArrayList<JasmineProbeListener>();

    /**
     * List of Outputs to be used when not defined in a probe.
     */
    private List<String> defaultOutputList = new ArrayList<String>();

    /**
     * Count used to generate probe ids
     */
    private int probeCount = 0;

    private ObjectName mbeanObjectName = null;

    // -----------------------------------------------------------------------------------
    // ipojo management
    // -----------------------------------------------------------------------------------

    @Validate
    public void start() {
        logger.debug("JasmineProbeManager service activated.");
        try {
            readConfig();

        } catch (Exception e) {
            // Cannot read configuration
            logger.error("Cannot read configuration");
        }

        // register mbean
        try {
            mbeanObjectName = new ObjectName("jasmine:dest=probe-manager");
            jmxService.registerMBean(this, mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot register mbean: " + e);
        }

        // register test mbeans
        registerMBeans();
    }

    private void registerMBeans() {
        try {
            ObjectName mbeanObjectName = ObjectName.getInstance("jasmine:dest=open-manager");

            AB ab_1 = new AB(1, 2);
            AB ab_2 = new AB(3, 4);
            AB ab_3 = new AB(5, 6);

            Long l_1 = new Long(2356);
            Long l_2 = new Long(45678);
            Long l_3 = new Long(1000004);
            Long[] longs = new Long[] {l_1, l_2, l_3};

            CD cd = new CD(longs, ab_3);

            ObjectName[] ons = new ObjectName[] {mbeanObjectName};

            AB[] elems = new AB[] {ab_1, ab_2};

            TestOpenMBean testMbean = new TestOpenMBean(ab_1, ab_2, cd, longs, elems, ons);

            jmxService.registerMBean(testMbean, mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot register mbean: " + e);
        }
    }

    private void unregisterMBeans() {
        try {
            ObjectName mbeanObjectName = ObjectName.getInstance("jasmine:dest=open-manager");
            jmxService.unregisterMBean(mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot unregister mbean: " + e);
        }
    }


    @Invalidate
    public void stop() {
        logger.debug("JasmineProbeManager service stopped.");

        // unregister mbean
        jmxService.unregisterMBean(mbeanObjectName);

        // register test mbeans
        unregisterMBeans();
    }

    @Requires
    protected SchedulerService jasmineProbeScheduler = null;

    @Requires
    protected JmxService jmxService = null;

    @Bind(aggregate = true, optional = true)
    protected void bindOuterService(JasmineOuterService jos, ServiceReference sr) {
        String type = (String) sr.getProperty("output.type");
        if (type == null) {
            logger.warn("A service providing implementation for a JASMINe Probe Output type must use 'output.type' property to specify that type");
            return;
        }
        String mess = "A service for " + type + " output";
        JasmineOuterService old = outerservices.get(type);
        if (old != null) {
            logger.warn(mess + " already deployed");
            return;
        }
        outerservices.put(type, jos);
        logger.info(mess + " deployed to JASMINe Probe");

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindOuterService(JasmineOuterService jos, ServiceReference sr) {
        String type = (String) sr.getProperty("output.type");
        if (type == null) {
            logger.warn("Undeploying a service providing implementation for a JASMINe Probe Output that has not defined 'output.type' property");
            return;
        }
        if (outerservices.get(type) == null) {
            return;
        }
        // Get outputs having as type the service type
        String[] outputNames = listOutputs(type);
        for (String outputName : outputNames) {
            for (JasmineProbe probe : probedefs.values()) {
                if (probe.getStatus() == JasmineProbe.PROBE_RUNNING) {
                    // check that this probe does not uses one of the outputs
                    for (String output : probe.getOutputList()) {
                        if (output.equals(outputName)) {
                            logger.warn("Stop probe {0}", probe.getId());
                            try {
                                stopProbe(probe.getId());
                            } catch (JasmineProbeException e) {
                                logger.error(e);
                                //e.printStackTrace();
                            }
                            break;
                        }
                    }
                }
            }
        }
        outerservices.remove(type);
    }

    @Bind(aggregate = true, optional = true)
    protected void bindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        String type = (String) sr.getProperty("indicator.type");
        if (type == null) {
            logger.warn("A service providing implementation for a JASMINe Probe Indicator type must use 'indicator.type' property to specify that type");
            return;
        }
        String mess = "A service for " + type + " indicator";
        JasmineCollectorService old = collectorservices.get(type);
        if (old != null) {
            logger.warn(mess + " already deployed");
            return;
        }
        collectorservices.put(type, jcs);
        logger.info(mess + " deployed to JASMINe Probe");

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        String type = (String) sr.getProperty("indicator.type");
        if (type == null) {
            logger.warn("Undeploying a service providing implementation for a JASMINe Probe Indicator that has not defined 'indicator.type' property");
            return;
        }
        if (collectorservices.get(type) == null) {
            return;
        }
        String[] indicatorNames = listIndicators(type);
        for (String indicatorName : indicatorNames) {
            for (JasmineProbe probe : probedefs.values()) {
                if (probe.getStatus() == JasmineProbe.PROBE_RUNNING) {
                    // check that this probe does not uses one of the indicators of the service's type
                    for (String indic : getRecursiveIndicatorList(probe)) {
                        if (indic.equals(indicatorName)) {
                            logger.warn("Stop probe {0}", probe.getId());
                            try {
                                stopProbe(probe.getId());
                            } catch (JasmineProbeException e) {
                                logger.error(e);
                            }
                            break;
                        }
                    }
                }
            }
        }
        collectorservices.remove(type);
    }

    @Bind(aggregate = true, optional = true)
    protected void bindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("A service providing specific implementation for an aggregate JASMINe Probe Indicator must define 'aggregate.type' property to specify its aggregate function");
            return;
        }
        JasmineAggregateService old = aggregateservices.get(type);
        if (old != null) {
            // A service for aggregate function already exists, use it if its not un-deployed.
            // Another approach could be to replace the old function by a new one.
            logger.warn("A service providing implementation for an aggregate JASMINe Probe Indicator already defined for " + type + " function. Please check deployment plan and undeploy old service !");
            return;
        }
        aggregateservices.put(type, jas);
        logger.info("A service providing implementation for an aggregate JASMINe Probe Indicator with {0} function deployed to JasmineProbe", type);
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("Undeploying a service providing implementation for an aggregate JASMINe Probe Indicator that has not defined 'indicator.type' property");
            return;
        }
        if (aggregateservices.containsKey(type)) {
            aggregateservices.remove(type);
            logger.info("A service providing implementation for an aggregate JASMINe Probe Indicator with {0} function undeployed from JasmineProbe", type);
            logger.info("{0} JasmineAggregateService unbound into JasmineProbe.", type);
        }
    }

    // -----------------------------------------------------------------------------------
    //  internal interface
    // -----------------------------------------------------------------------------------

    /**
     * return the scheduler
     * @return referencee on the SchedulerService
     */
    public SchedulerService getScheduler() {
        return jasmineProbeScheduler;
    }

    /**
     * Get the OuterService for a given type
     */
    public JasmineOuterService getOuterService(String type) {
        return outerservices.get(type);
    }

    /**
     * Get the CollectorService for a given type
     */
    public JasmineCollectorService getCollectorService(String type) {
        return collectorservices.get(type);
    }

    // -----------------------------------------------------------------------------------
    //  JasmineProbeManager implementation
    // -----------------------------------------------------------------------------------

    /**
     * Create a new probe defined by its data
     *
     * @param probe probe description
     * @return the Id to be used to reference this probe later.
     */
    public synchronized String createProbe(JasmineProbe probe) throws JasmineProbeException {
        // Generate an Id if not supplied
        String id = probe.getId();
        logger.debug("creating probe " + id);
        if (id == null) {
            // If possible, take the name of unique indicator
            if (probe.getIndicatorList().size() == 1) {
                String strindic = probe.getIndicatorList().get(0);
                boolean found = false;
                for (JasmineProbe p : probedefs.values()) {
                    if (p.getId().equals(strindic)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    id = strindic;
                }
            }
            if (id == null) {
                id = getNewProbeId();
            }
            probe.setId(id);
        } else {
            // Check Id not already used
            /*
            for (JasmineProbe p : probedefs.values()) {
                if (p.getId().equals(id)) {
                    logger.error("Id already used: " + id);
                    throw new JasmineProbeException("Id already used");
                }
            }*/
            JasmineProbe p = probedefs.get(id);
            if (p != null) {
                throw new JasmineProbeException("Probe id " + id + " already used");
            }
        }

        // Check validity of the JasmineProbe
        if (probe.getIndicatorList() == null || probe.getIndicatorList().isEmpty()) {
             throw new JasmineProbeException("A probe must have at least one indicator");
        }
        if (probe.getOutputList() == null || probe.getOutputList().isEmpty()) {
            probe.setOutputList(defaultOutputList);
        }

        probedefs.put(id, probe);
        return id;
    }

    /**
     * Remove a probe.
     *
     * @param probeId probe identifier.
     */
    public synchronized void removeProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot remove probe with unexistent id " + probeId);
        }

        // Check if probe is running
        JProbe probe = probes.get(probeId);
        if (probe != null) {
            probe.suspend();
            probe.resetIndicators();
            probe.resetOutputs();
            probes.remove(probeId);
        }
        logger.debug("removing probe " + probeId);
        probedefs.remove(probeId);
    }

    /**
     * Remove all the managed probes.
     * Stop probe running before.
     *
     * @throws JasmineProbeException
     */
    public synchronized void removeAllProbes() throws JasmineProbeException {
        logger.debug("");

        // must build a list first, to avoid ConcurrentModificationException.
        List<String> namelist = new ArrayList<String>();
        for (String n : probedefs.keySet()) {
            namelist.add(n);
        }
        for (String name : namelist) {
            removeProbe(name);
        }
    }

    /**
     * Get all the Probe definitions in one operation.
     * @return all the probe definitions
     */
    public synchronized List<JasmineProbe> getProbes() {
        logger.debug("");
        List<JasmineProbe> ret = new ArrayList<JasmineProbe>();
        /** Descending order
        for (Iterator<String> keys = ((TreeMap<String, JasmineProbe>) probedefs).descendingKeySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineProbe) probedefs.get(keys.next()));
        }
        */
        /**
         * Iterate over keys (probe ids) in ascending order.
         */
        for (Iterator<String> keys = ((TreeMap<String, JasmineProbe>) probedefs).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineProbe) probedefs.get(keys.next()));
        }
        return ret;
    }

    /**
     * @return all the probe ids
     */
    public synchronized String [] listProbes() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineProbe>) probedefs).keySet().iterator(); keys.hasNext(); ) {
            ret.add(keys.next());
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get a Probe by its name
     * @param probeId probe identifier.
     * @return the Probe definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineProbe getProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot find probe with id " + probeId);
        }
        return def;
     }

    /**
     * Start a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be started.
     */
    public synchronized void startProbe(String probeId) throws JasmineProbeException {
        logger.debug(probeId);

        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with unexistent id " + probeId);
        }

        // Check if already running
        JProbe running = probes.get(probeId);
        if (running != null) {
            switch (running.getStatus()) {
                case JasmineProbe.PROBE_RUNNING:
                    logger.info("Probe already running: " + probeId);
                    return;
                case JasmineProbe.PROBE_STARTED:
                    logger.info("Probe already started: " + probeId);
                    return;
                case JasmineProbe.PROBE_FAILED:
                    logger.info("restarting a failed probe: " + probeId);
                    break;
                case JasmineProbe.PROBE_STOPPED:
                    logger.info("restarting a stopped probe: " + probeId);
                    break;
                default:
                    logger.warn("Bad state for: " + probeId);
                    break;
            }
        } else {
            // Create the JProbe object (a running probe)
            running = new JProbe(def, this);
            probes.put(probeId, running);
        }
        running.resume();
    }

    /**
     * Stop a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be stopped.
     */
    public synchronized void stopProbe(String probeId) throws JasmineProbeException {
        logger.debug(probeId);

        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with unexistent id " + probeId);
        }

        // Check if already running
        JProbe running = probes.get(probeId);
        if (running == null) {
            throw new JasmineProbeException("This probe is not running: " +  probeId);
        }

        // Stop the probe by halting polling and changing status to JasmineProbe.PROBE_STOPPED
        running.suspend();
    }

    /**
     * Start all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void startAllProbes() throws JasmineProbeException {
        logger.debug("");
        for (JasmineProbe probe : probedefs.values()) {
            try {
                startProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not start probe " + probe.getId() + " :" + e);
            }
        }
    }

    /**
     * Stop all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void stopAllProbes() throws JasmineProbeException {
        logger.debug("");
        for (JProbe probe : probes.values()) {
            try {
                probe.suspend();
            } catch (Exception e) {
                logger.warn("could not stop probe " + probe.getProbeDef().getId() + " :" + e);
            }
        }
    }

    /**
     * Change a Probe with new parameters
     * All parameters that are not supplied rest unchanged.
     *
     * @param newdef  partial probe description
     */
    public synchronized void changeProbe(JasmineProbe newdef) throws JasmineProbeException {

        // Retrieve the Probe by its Id
        String probeId = newdef.getId();
        logger.debug(probeId);

        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Probe not found: " + probeId);
        }

        // Stop it first if it is running.
        // It will be restarted after changed
        boolean restart = false;
        JProbe running = probes.get(probeId);
        if (running != null) {
            restart = running.suspend();
        }

        // Change period
        if (newdef.getPeriod() > 0) {
            def.setPeriod(newdef.getPeriod());
        }

        // Change output list
        if (newdef.getOutputList().size() > 0) {
            def.setOutputList(newdef.getOutputList());
            if (running != null) {
                running.resetOutputs();
            }
        }

        // Change indicator list
        if (newdef.getIndicatorList().size() > 0) {
            def.setIndicatorList(newdef.getIndicatorList());
            if (running != null) {
                running.resetIndicators();
            }
        }

        // Change target list
        if (newdef.getTargetList().size() > 0) {
            def.setTargetList(newdef.getTargetList());
            if (running != null) {
                running.resetTargets();
            }
        }

        // Restart probe if it was suspended
        if (restart) {
            try {
                running.resume();
            } catch (JasmineProbeException e) {
                logger.warn("Probe {0} changed. It was stopped but could not be restarted: {1}", probeId, e.getMessage());
            }
        }
    }

    /**
     * Set a new value for probe period
     * @param probeId probe identifier.
     * @param period in seconds
     */
    public void changeProbePeriod(String probeId, int period) throws JasmineProbeException {
        // Change value in definition
        JasmineProbe def = probedefs.get(probeId);
        def.setPeriod(period);

        // If probe is running, must change its timer.
        JProbe running = probes.get(probeId);
        if (running != null && running.isRunning()) {
             running.resetTimer();
        }
    }

    /**
     * Register a new Output in the ProbeManager
     * A name is associated to the Output definition.
     * This Output will be usable later in probes.
     *
     * @param newoutput output description
     * @return the Id to be used to reference this output later.
     */
    public String createOutput(JasmineOutput newoutput) throws JasmineProbeException {
        logger.debug("");
        return createOutputInternal(newoutput, true);
    }

    /**
     * Change an output. The output must exist.
     *
     * @param newdef  output changes description
     * @throws JasmineProbeException is thrown if the output does not exist.
     */
    public void changeOutput(JasmineOutput newdef) throws JasmineProbeException {
        logger.debug("");

        // Find the JasmineOutput to be changed
        String name = newdef.getName();
        JasmineOutput output = outputs.get(name);
        if (output == null) {
            throw new JasmineProbeException("This output does not exist: " +  name);
        }

        // Stop all running probes using this output.
        List<JProbe> prlist = new ArrayList<JProbe>();
        for (JProbe running : probes.values()) {
            JasmineProbe probe = running.getProbeDef();
            for (String oname : probe.getOutputList()) {
                if (oname.equals(name)) {
                    running.suspend();
                    running.resetOutputs();
                    prlist.add(running);
                    break;
                }
            }
        }

        // remove Outers defined on this Output
        JasmineOuterService jos = outerservices.get(output.getType());
        if (jos != null) {
            jos.removeOuters(name);
        }

        // Change type
        if (newdef.getType() != null) {
            output.setType(newdef.getType());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            if (newval == null || newval.isEmpty()) {
                output.getProperties().remove(key);
            } else {
                // got new value to set
                // If an old value exists, it is replaced by the new value
                output.getProperties().put(key, newval);
            }
        }

        // restart probes
        for (JProbe running : prlist) {
            running.resume();
        }
    }

    /**
     * remove an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     *
     * @param name ident of the output to remove
     * @throws JasmineProbeException is thrown if the output does not exist,
     *                               or if it is used by a running probe.
     */
    public void removeOutput(String name) throws JasmineProbeException {
        logger.debug("");
        JasmineOutput output = outputs.get(name);
        if (output == null) {
            throw new JasmineProbeException("This output does not exist: " +  name);
        }
        if (isOutputUsed(output, true)) {
            throw new JasmineProbeException("This output is used and cannot be removed: " +  name);
        }
        JasmineOuterService jos = outerservices.get(output.getType());
        if (jos != null) {
            jos.removeOuters(name);
        }
        outputs.remove(name);
    }

    /**
     * Get the list of all outputs or of outputs having a given type
     * @param type the given type. If null, get all the outputs.
     */
    public synchronized String [] listOutputs(final String type) {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineOutput>) outputs).keySet().iterator(); keys.hasNext(); ) {
            String name = keys.next();
            if (type == null) {
                ret.add(name);
            } else {
                JasmineOutput out = outputs.get(name);
                if (type.equals(out.getType())) {
                    ret.add(name);
                }
            }
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all the Output definitions in one operation.
     * @return all the output definitions
     */
    public synchronized List<JasmineOutput> getOutputs() {
        logger.debug("");
        List<JasmineOutput> ret = new ArrayList<JasmineOutput>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineOutput>) outputs).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineOutput) outputs.get(keys.next()));
        }
        return ret;
    }

    /**
     * Get the list of all outputs of a given type.
     * @param type the given output type
     * @return List of the outputs having that type
     */
    public synchronized List<JasmineOutput> getOutputs(final String type) {
        logger.debug("");
        List<JasmineOutput> ret = new ArrayList<JasmineOutput>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineOutput>) outputs).keySet().iterator(); keys.hasNext(); ) {
            JasmineOutput output = outputs.get(keys.next());
            if (output.getType().equals(type)) {
                ret.add(output);
            }
        }
        return ret;
    }

    /**
     * Get an Output by its name
     * @param id output identifier.
     * @return the Output definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineOutput getOutput(String id) throws JasmineProbeException {
        logger.debug("");
        JasmineOutput def = outputs.get(id);
        if (def == null) {
            throw new JasmineProbeException("Cannot find output with id " + id);
        }
        return def;
     }

    /**
     * get the list of possible properties for Output, depending of its type.
     *
     * @param type type of the Output (console, file, ...)
     * @return List of JasminePropertyInfo
     */
    public List<JasminePropertyInfo> getOutputPropertyInfos(String type) {
        if (type == null || type.length() == 0) {
            logger.error("No type provided");
            return null;
        }
        logger.debug(type);

        // Find the appropriate OuterService
        JasmineOuterService jos = outerservices.get(type);
        if (jos == null) {
            logger.error("No OuterService found for type " + type);
            return null;
        }
        return jos.getPropertiesInfo();
    }

    /**
     * @return the available output types.
     */
    public synchronized Set<String> getOutputTypes() {
        return outerservices.keySet();
    }

    /**
     * Get the list of probes using this output
     * @param name  Output name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    public synchronized Set<String> getOutputCurrentUse(String name) throws JasmineProbeException {
        Set<String> ret = new HashSet<String>();
        for (JasmineProbe probe : probedefs.values()) {
            for (String output : probe.getOutputList()) {
                if (output.equals(name)) {
                    ret.add(probe.getId());
                }
            }
        }
        return ret;
    }

    /**
     * Create a new Indicator
     *
     * @param newindic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    public String createIndicator(JasmineIndicator newindic) throws JasmineProbeException {
        return createIndicatorInternal(newindic, true);
    }

    /**
     * Change an indicator. It must exist.
     *
     * @param newdef indicator description
     * @throws JasmineProbeException is thrown if the indicator does not exist,
     *                               or if it is used by a running probe.
     */
    public void changeIndicator(JasmineIndicator newdef) throws JasmineProbeException {

        // Find the JasmineIndicator to be changed
        String name = newdef.getName();
        logger.debug(name);

        JasmineIndicator indic = indicators.get(name);
        if (indic == null) {
            throw new JasmineProbeException("This indicator does not exists: " +  name);
        }

        // Stop all running probes using this indicator.
        List<JProbe> prlist = new ArrayList<JProbe>();
        for (JProbe running : probes.values()) {
            JasmineProbe probe = running.getProbeDef();
            for (String oname : probe.getIndicatorList()) {
                if (oname.equals(name)) {
                    running.suspend();
                    running.resetIndicators();
                    prlist.add(running);
                    break;
                }
            }
        }

        // Remove Collectors defined on this Indicator
        JasmineCollectorService jcs = collectorservices.get(indic.getType());
        if (jcs != null) {
            jcs.removeCollectors(name, null);
        }

        // Change type
        if (newdef.getType() != null) {
            indic.setType(newdef.getType());
        }

        // Change scale
        if (newdef.getScale() != indic.getScale()) {
            indic.setScale(newdef.getScale());
        }

        // Change modified sources
        if (! newdef.getSources().isEmpty()) {
            indic.setSources(newdef.getSources());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            if (newval == null || newval.isEmpty()) {
                indic.getProperties().remove(key);
            } else {
                // got new value to set
                // If an old value exists, it is replaced by the new value
                indic.getProperties().put(key, newval);
            }
        }

        // restart probes
        for (JProbe running : prlist) {
            running.resume();
        }
    }


    /**
     * remove an indicator
     *
     * @param name ident of the indicator to remove
     */
    public void removeIndicator(String name) throws JasmineProbeException {
        logger.debug(name);
        JasmineIndicator indic = indicators.get(name);
        if (indic == null) {
            throw new JasmineProbeException("This indicator does not exist: " +  name);
        }
        if (isIndicatorUsed(indic, true)) {
            throw new JasmineProbeException("This indicator is used and cannot be removed: " +  name);
        }

        JasmineCollectorService jcs = collectorservices.get(indic.getType());
        if (jcs != null) {
            jcs.removeCollectors(name, null);
        }
        indicators.remove(name);
    }

    /**
     * Get an Indicator by its name
     * @param name indicator name.
     * @return the definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineIndicator getIndicator(String name) throws JasmineProbeException {
        logger.debug(name);
        JasmineIndicator def = indicators.get(name);
        if (def == null) {
            /*
            JasmineProbeException e =  new JasmineProbeException("Cannot find indicator with id " + id);
            e.printStackTrace();
            throw e;*/
            throw new JasmineProbeException("Cannot find indicator " + name);
        }
        return def;
     }

    /**
     * Get all the Indicator definitions in one operation.
     * @return all the indicator definitions
     */
    public synchronized List<JasmineIndicator> getIndicators() {
        logger.debug("");
        List<JasmineIndicator> ret = new ArrayList<JasmineIndicator>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineIndicator>) indicators).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineIndicator) indicators.get(keys.next()));
        }
        return ret;
    }

    /**
     * Get the list of indicators having a given type.
     * @param type the given indicator type
     * @return List of the indicators having that type
     */
    public synchronized List<JasmineIndicator> getIndicators(final String type) {
        logger.debug("");
        List<JasmineIndicator> ret = new ArrayList<JasmineIndicator>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineIndicator>) indicators).keySet().iterator(); keys.hasNext(); ) {
            JasmineIndicator indic = indicators.get(keys.next());
            if (indic.getType().equals(type)) {
                ret.add(indic);
            }
        }
        return ret;
    }

    /**
     * @return all the indicator names
     */
    public synchronized String [] listIndicators(final String type) {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineIndicator>) indicators).keySet().iterator(); keys.hasNext(); ) {
            String name = keys.next();
            if (type == null) {
                ret.add(name);
            } else {
                JasmineIndicator indic = indicators.get(name);
                if (type.equals(indic.getType())) {
                    ret.add(name);
                }
            }
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all infiormation about the type of indicator
     * @param type type of the Indicator (Jmx, Lewys, ...)
     * @return JasmineIndicatorTypeInfo
     */
    public JasmineCollectorInfo getIndicatorTypeInfo(String type) {
        if (type == null || type.length() == 0) {
            logger.error("No type provided");
            return null;
        }
        logger.debug(type);

        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(type);
        if (jcs == null) {
            logger.error("No CollectorService found for type " + type);
            return null;
        }
        return jcs.getCollectorInfo();
    }

    /**
     * @return the available indicator types.
     */
    public synchronized Set<String> getIndicatorTypes() {
        logger.debug("");
        return collectorservices.keySet();
    }

    /**
     * Get the list of probes using this indicator
     * @param name  Indicator name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    public synchronized Set<String> getIndicatorCurrentUse(String name) throws JasmineProbeException {
        Set<String> ret = new HashSet<String>();
        for (JasmineProbe probe : probedefs.values()) {
            for (String indic : getRecursiveIndicatorList(probe)) {
                if (indic.equals(name)) {
                    ret.add(probe.getId());
                }
            }
        }
        return ret;
    }

    /**
     * Create a new target
     *
     * @param newtarget target description
     * @return the Id to be used to reference this target later.
     */
    public synchronized String createTarget(JasmineTarget newtarget) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newtarget.getName();
        if (id == null || id.length() == 0) {
            throw new JasmineProbeException("No valid target name");
        }
        logger.debug(id);

        // Check if already known
        JasmineTarget target = targets.get(id);
        if (target != null) {
            logger.debug("target already known: " + id);
            if (target.equals(newtarget)) {
                // If same target already created, just return.
                return id;
            }
            if (isTargetUsed(target)) {
                logger.warn("oldtarget:" + target);
                logger.warn("newtarget:" + newtarget);
                throw new JasmineProbeException(id + "Target already known with another definition");
            }
            // remove old definition and replace by the new one.
            targets.remove(id);
        }
        targets.put(id, newtarget);
        return id;
    }

    /**
     * Change a target. It must exist, otherwise an exception is thrown.
     *
     * @param newdef new target description
     * @throws JasmineProbeException is thrown if the target does not exist.
     */
    public void changeTarget(JasmineTarget newdef) throws JasmineProbeException {
        // Find the JasmineTarget to be changed
        String name = newdef.getName();
        logger.debug(name);

        JasmineTarget target = targets.get(name);
        if (target == null) {
            throw new JasmineProbeException("This target does not exist: " +  name);
        }

        // Stop all running probes using this target.
        List<JProbe> prlist = new ArrayList<JProbe>();
        for (JProbe running : probes.values()) {
            JasmineProbe probe = running.getProbeDef();
            for (String oname : probe.getTargetList()) {
                if (oname.equals(name)) {
                    running.suspend();
                    running.resetTargets();
                    prlist.add(running);
                    break;
                }
            }
        }

        // Change type
        if (newdef.getType() != null) {
            target.setType(newdef.getType());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            if (newval == null || newval.isEmpty()) {
                target.getProperties().remove(key);
            } else {
                logger.debug(key+"="+newval);
                // got new value to set
                // If an old value exists, it is replaced by the new value
                target.getProperties().put(key, newval);
            }
        }

        // restart probes
        for (JProbe running : prlist) {
            running.resume();
        }
    }

    /**
     * remove a target
     *
     * @param name ident of the target to remove
     */
    public void removeTarget(String name) throws JasmineProbeException {
        logger.debug(name);
        JasmineTarget target = targets.get(name);
        if (target == null) {
            throw new JasmineProbeException("This target does not exist: " +  name);
        }
        if (isTargetUsed(target)) {
            throw new JasmineProbeException("This target is used and cannot be removed: " +  name);
        }
        targets.remove(name);
    }

    /**
     * @return all the target names
     */
    public synchronized String [] listTargets() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineTarget>) targets).keySet().iterator(); keys.hasNext(); ) {
            ret.add(keys.next());
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all the Target definitions in one operation.
     * @return all the target definitions
     */
    public synchronized List<JasmineTarget> getTargets() {
        logger.debug("");
        List<JasmineTarget> ret = new ArrayList<JasmineTarget>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineTarget>) targets).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineTarget) targets.get(keys.next()));
        }
        return ret;
    }

    /**
     * Get a JasmineTarget by its name
     * @param name
     * @return JasmineTarget definition
     */
    public JasmineTarget getTarget(String name) {
        JasmineTarget target = targets.get(name);
        return target;
    }

    /**
     * get the list of possible properties for Target, depending of its type.
     *
     * @param type type of the Target (jmx, ...)
     * @return List of JasminePropertyInfo
     */
    public List<JasminePropertyInfo> getTargetPropertyInfos(String type) {
        if (type == null || type.length() == 0) {
            logger.error("No type provided");
            return null;
        }
        logger.debug(type);

        // TODO This must be changed when targets may have other type than jmx
        ArrayList<JasminePropertyInfo> ret = new ArrayList<JasminePropertyInfo>();
        ret.add(new JasminePropertyInfo("url", "target jmx URL", true));
        ret.add(new JasminePropertyInfo("user", "user name to access the target", false));
        ret.add(new JasminePropertyInfo("password", "password associated to the user name", false));
        ret.add(new JasminePropertyInfo("protocolProviders", "optional protocol provider", false));
        return ret;
    }

    /**
     * @return the available indicator types.
     */
    public synchronized Set<String> getTargetTypes() {
        logger.debug("");
        // TODO This must be changed when targets may have other type than jmx
        HashSet<String> ret = new HashSet<String>();
        ret.add("jmx");
        return ret;
    }

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     *
     * @param listener object that treats the probe state change
     */
    public synchronized void addProbeListener(JasmineProbeListener listener) {
        logger.debug("");
        probeListeners.add(listener);
    }

    /**
     * Remove a ProbeListener previously registered.
     *
     * @param listener object that treats the probe state change
     */
    public synchronized void removeProbeListener(JasmineProbeListener listener) {
        logger.debug("");
        probeListeners.remove(listener);
    }

    /**
     * Save the current configuration in the specified xml file
     *
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws JasmineProbeException
     */
    public void saveConfig(String path) throws JasmineProbeException {
        logger.debug(path);
        FileOutputStream outputStream = null;
        try {
            outputStream = new FileOutputStream(new File(JProp.getConfDir(), "probe-config.xml"));
        } catch (FileNotFoundException e) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'" + e);
        }
        if (outputStream == null) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'");
        }
        try {
            saveConfigFile(outputStream);
        } catch (Exception e) {
            logger.warn("could not save config: " + e);
            throw new JasmineProbeException("could not save config: " + e.getMessage());
        }
    }

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     *
     * @param path the xml file.
     * @throws JasmineProbeException
     */
    public void loadConfig(String path) throws JasmineProbeException {
        logger.debug(path);
        try {
            loadConfigFile(new FileInputStream(path));
        } catch (Exception e) {
            logger.warn("could not load config: " + e);
            throw new JasmineProbeException("could not load config: " + e.getMessage());
        }
        // Start probes marked as started in the file
        startWaitingProbes();
    }

    // -----------------------------------------------------------------
    // ProbeManager implementation
    // -----------------------------------------------------------------

    /**
     * Get a Collector by its Indicator name
     * @param probeid
     * @param indicatorName indicator name
     * @return The collector corresponding to an indicator.
     */
    public synchronized JasmineCollector getCollector(String indicatorName, String probeid) throws JasmineProbeException {

        JasmineIndicator indicator = indicators.get(indicatorName);
        if (indicator == null) {
            throw new JasmineProbeException("Cannot return collector for unexistent indicator " + indicatorName);
        }

        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(indicator.getType());
        if (jcs == null) {
            logger.error("No CollectorService found for {0} having type {1}", indicatorName, indicator.getType());
            throw new JasmineProbeException("No CollectorService found for " + indicatorName);
        }
        // Get the Collector from the CollectorService
        JasmineCollector collector = null;
        try {

            collector = jcs.getCollector(indicator, probedefs.get(probeid));
        } catch (JasmineCollectorException e) {
            logger.error("Could not get the Collector for " + indicatorName + ": " + e);
            throw new JasmineProbeException("Could not get the Collector for " + indicatorName);
        }
        return collector;
    }

    /**
     * @return If AggregateServices exist, return the functions these services implement (the values of aggregate.type properties).
     * Else, return null.
     */
    public synchronized List<String> getAggregateFuntions() {
        if (!aggregateservices.isEmpty()) {
            List<String> functions = new ArrayList<String>();
            Set<String> keySet = aggregateservices.keySet();
            for (String function : keySet) {
                functions.add(function);
            }
            return functions;
        }
        return null;
    }

    /**
     * Return the AggregateService for a given aggregate function
     * @param key the aggregate function that is searched
     * @return the AggregateService having aggregate.type equal to key. Else, return null.
     */
    public synchronized JasmineAggregateService getAggregate(String key) {
        if (aggregateservices.containsKey(key)) {
            return aggregateservices.get(key);
        }
        return null;
    }

    /**
     * @return the current domain name.
     */
    public String getDomainName() {
       return jmxService.getDomainName();
    }

    /**
     * @return the current server name.
     */
    public String getServerName() {
       return jmxService.getJonasServerName();
    }


    // --------------------------------------------------------------------------------------
    // Private methods
    // --------------------------------------------------------------------------------------

    /**
     * Create a new Output
     * @param newoutput
     * @param check
     * @return
     * @throws JasmineProbeException
     */
    private synchronized String createOutputInternal(JasmineOutput newoutput, boolean check) throws JasmineProbeException {
        // Id chosen by the user.
        String id = newoutput.getName();
        if (id == null || id.length() == 0) {
            throw new JasmineProbeException("No valid output name");
        }

        // Check that a type is provided
        String type = newoutput.getType();
        if (type == null) {
            throw new JasmineProbeException("No type provided");
        }

        if (check) {
            // Check that the type is supported
            JasmineOuterService jos = outerservices.get(type);
            if (jos == null) {
                throw new TypeNotSupportedException("Output type not supported: " + type);
            }

            // Check that mandatory properties are supplied
            String missinglist = "";
            for (JasminePropertyInfo jpi : jos.getPropertiesInfo()) {
                if (jpi.isRequired()) {
                    boolean found = false;
                    for (String key : newoutput.getProperties().keySet()) {
                        if (key.equals(jpi.getName())) {
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        missinglist += " " + jpi.getName();
                    }
                }
            }
            if (missinglist.length() > 0) {
                logger.warn("Missing mandatory properties:" + missinglist);
                for (String key : newoutput.getProperties().keySet()) {
                    logger.warn("found " + key);
                }
                throw new PropertyMissingException("Missing mandatory properties:" + missinglist);
            }
        }

        logger.debug("Create Output " + id);

        // Check if already known
        JasmineOutput output = outputs.get(id);
        if (output != null) {
            logger.debug("output already known: " + id);
            if (output.equals(newoutput)) {
                // If same output already created, just return.
                return id;
            }
            if (isOutputUsed(output, true)) {
                logger.warn("oldoutput:" + output);
                logger.warn("newoutput:" + newoutput);
                throw new JasmineProbeException(id + " output already known with another definition");
            }
            // remove old definition and replace by the new one.
            outputs.remove(id);
        }

        outputs.put(id, newoutput);
        return id;
    }

    /**
     * Create a new Indicator
     *
     * @param newindic indicator description
     * @param check true if check properties against the Collector (that should be there)
     * @return the Id to be used to reference this indicator later.
     */
    public synchronized String createIndicatorInternal(JasmineIndicator newindic, boolean check) throws JasmineProbeException {

        // Id chosen by the user.
        String id = newindic.getName();
        if (id == null || id.length() == 0) {
            // TODO: generate Id if not provided ?
            throw new JasmineProbeException("Automatic indicator name is not supported yet");
        }

        // Check that a type is provided
        String type = newindic.getType();
        if (type == null) {
            throw new TypeNotSupportedException("No type provided");
        }

        if (check) {
            // Check that the type is supported
            JasmineCollectorService jcs = collectorservices.get(type);
            if (jcs == null) {
                throw new TypeNotSupportedException("Indicator type not supported: " + type);
            }

            JasmineCollectorInfo info = jcs.getCollectorInfo();

            // Check that mandatory properties are supplied
            String missinglist = "";
            for (JasminePropertyInfo jpi : info.getPropertyInfos()) {
                if (jpi.isRequired()) {
                    boolean found = false;
                    for (String key : newindic.getProperties().keySet()) {
                        if (key.equals(jpi.getName())) {
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        missinglist += " " + jpi.getName();
                    }
                }
            }
            if (missinglist.length() > 0) {
                throw new PropertyMissingException("Missing mandatory properties:" + missinglist);
            }

            // Check that a correct number of indicator sources are supplied
            if (newindic.getSources().size() < info.getSourceMin()) {
                throw new SourceNumberException("Missing source indicator. Minimum number is " + info.getSourceMin());
            }
            if (newindic.getSources().size() > info.getSourceMax() && info.getSourceMax() < 3) {
                throw new SourceNumberException("Too manysource indicators. Maximum number is " + info.getSourceMax());
            }
        }

        logger.debug("Create Indicator " + id);

        // Check if already known
        JasmineIndicator indic = indicators.get(id);
        if (indic != null) {
            logger.debug("indicator already known: " + id);
            if (indic.equals(newindic)) {
                // If same indicator already created, just return.
                return id;
            }
            if (isIndicatorUsed(indic, true)) {
                logger.warn("old indic:" + indic);
                logger.warn("new indic:" + newindic);
                throw new JasmineProbeException(id + "Indicator already known with another definition");
            }
            // remove old definition and replace by the new one.
            indicators.remove(id);
        }

        indicators.put(id, newindic);
        return id;
    }

    /**
     * Generate an unique identifier
     */
    private synchronized String getNewProbeId() {
        String ret = "probe-0";
        boolean found = false;
        while (! found) {
            ret = "probe-" + ++probeCount;
            found = true;
            for (JasmineProbe p : probedefs.values()) {
                if (p.getId().equals(ret)) {
                    found = false;
                    break;
                }
            }
        }
        return ret;
    }

    /**
     * Read configuration.
     */
    private void readConfig() throws Exception {
        // Retrieve the probe config file
        String configurationFile = "probe-config.xml";
        ClassLoader loader = ClassLoader.getSystemClassLoader();
        InputStream resource = null;
        try {
            resource = loader.getResourceAsStream(configurationFile);
        } catch (Exception e) {
            logger.error("Cannot find probe-config.xml: " + e);
            throw e;
        }
        if (resource == null) {
            logger.error("Cannot find probe-config.xml");
            return;
        }
        logger.debug("Parsing probe-config.xml");
        loadConfigFile(resource);
    }

    private void saveConfigFile(OutputStream resource) throws Exception {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

            // rebuild probeConfig from current configuration
            probeConfig = new ProbeConfig();

            // outputs
            ArrayList<Output> olist = new ArrayList<Output>();
            for (JasmineOutput jo : outputs.values()) {
                Output output = new Output();
                output.setName(jo.getName());
                output.setType(jo.getType());
                for (String key : jo.getProperties().keySet()) {
                    Proptype pv =  new Proptype();
                    pv.setKey(key);
                    pv.setValue(jo.getProperties().get(key));
                    output.getProperty().add(pv);
                }
                output.setDefault(false);
                for (String on : defaultOutputList) {
                    if (on.equals(jo.getName())) {
                        output.setDefault(true);
                    }
                }
                olist.add(output);
            }
            probeConfig.setOutput(olist);

            // targets
            ArrayList<Target> tlist = new ArrayList<Target>();
            for (JasmineTarget jt : targets.values()) {
                Target target = new Target();
                target.setName(jt.getName());
                target.setType(jt.getType());
                for (String key : jt.getProperties().keySet()) {
                    Proptype pv =  new Proptype();
                    pv.setKey(key);
                    pv.setValue(jt.getProperties().get(key));
                    target.getProperty().add(pv);
                }
                tlist.add(target);
            }
            probeConfig.setTarget(tlist);

            // indicators
            ArrayList<Indicator> ilist = new ArrayList<Indicator>();
            for (JasmineIndicator ji : indicators.values()) {
                Indicator indic = new Indicator();
                indic.setName(ji.getName());
                indic.setType(ji.getType());
                String scale = new Long(ji.getScale()).toString();
                indic.setScale(new BigInteger(scale));
                indic.setSource(ji.getSources());
                for (String key : ji.getProperties().keySet()) {
                    Proptype pv =  new Proptype();
                    pv.setKey(key);
                    pv.setValue(ji.getProperties().get(key));
                    indic.getProperty().add(pv);
                }
                ilist.add(indic);
            }
            probeConfig.setIndicator(ilist);

            // probedefs
            ArrayList<Probe> plist = new ArrayList<Probe>();
            for (JasmineProbe jp : probedefs.values()) {
                Probe pdata = new Probe();
                pdata.setId(jp.getId());
                String period = new Long(jp.getPeriod()).toString();
                pdata.setPeriod(new BigInteger(period));
                ArrayList<String> onlist = new ArrayList<String>();
                for (String output : jp.getOutputList()) {
                    onlist.add(output);
                }
                pdata.setOutput(onlist);
                ArrayList<String> talist = new ArrayList<String>();
                for (String target : jp.getTargetList()) {
                    talist.add(target);
                }
                pdata.setTarget(talist);
                ArrayList<String> inlist = new ArrayList<String>();
                for (String ind : jp.getIndicatorList()) {
                    inlist.add(ind);
                }
                pdata.setIndicator(inlist);
                switch (jp.getStatus()) {
                case JasmineProbe.PROBE_FAILED:
                case JasmineProbe.PROBE_STOPPED:
                    pdata.setStatus(StatusType.STOPPED);
                    break;
                case JasmineProbe.PROBE_RUNNING:
                case JasmineProbe.PROBE_STARTED:
                    pdata.setStatus(StatusType.STARTED);
                    break;
                }
                plist.add(pdata);
            }
            probeConfig.setProbe(plist);

            // Build the xml file
            marshaller.marshal(probeConfig, resource);
        } catch (Exception e) {
            logger.error("Error while writing probe-config.xml: " + e);
            throw new JasmineProbeException("Error while writing probe-config.xml: " + e);
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    private synchronized void loadConfigFile(InputStream resource) throws JasmineProbeException {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            probeConfig = (ProbeConfig) unmarshaller.unmarshal(resource);

            // Create JasmineOutput objects
            for (Output output : probeConfig.getOutput()) {
                JasmineOutput jo = new JasmineOutput();
                jo.setName(output.getName());
                jo.setType(output.getType());
                if (output.getDefault() == true) {
                    defaultOutputList.add(output.getName());
                }
                Map<String, String> props = new HashMap<String, String>();
                for (Proptype param : output.getProperty()) {
                    props.put(param.getKey(), param.getValue());
                }
                jo.setProperties(props);
                createOutputInternal(jo, false);
            }

            // Create JasmineTarget objects
            for (Target target : probeConfig.getTarget()) {
                JasmineTarget jt = new JasmineTarget();
                jt.setName(target.getName());
                Map<String, String> props = new HashMap<String, String>();
                for (Proptype param : target.getProperty()) {
                    props.put(param.getKey(), param.getValue());
                }
                jt.setProperties(props);
                createTarget(jt);
            }

            // Create JasmineIndicator objects
            for (Indicator indic : probeConfig.getIndicator()) {
                JasmineIndicator ji = new JasmineIndicator();
                ji.setName(indic.getName());
                ji.setType(indic.getType());
                ji.setScale(indic.getScale().intValue());
                ji.setSources(indic.getSource());
                Map<String, String> props = new HashMap<String, String>();
                for (Proptype param : indic.getProperty()) {
                    props.put(param.getKey(), param.getValue());
                }
                ji.setProperties(props);
                createIndicatorInternal(ji, false);
            }

            // Create JasmineProbe objects
            for (Probe probe : probeConfig.getProbe()) {
                JasmineProbe jp = new JasmineProbe();
                jp.setId(probe.getId());
                jp.setPeriod(probe.getPeriod().intValue());
                jp.setOutputList(probe.getOutput());
                jp.setIndicatorList(probe.getIndicator());
                jp.setTargetList(probe.getTarget());
                if (probe.getStatus() != null && probe.getStatus().value().equals("started")) {
                    jp.setStatus(JasmineProbe.PROBE_TOSTART);
                }
                createProbe(jp);
            }

        } catch (Exception e) {
            logger.error("Error in probe-config.xml: " + e);
            throw new JasmineProbeException("Error in probe-config.xml: " + e);
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    /**
     * Test if an Output is used at least in one ProbeInstance.
     *
     * @param output the output to check
     * @param checkrun  If true, check only if used by running probes
     * @return true if output used in at least 1 probe.
     */
    private synchronized boolean isOutputUsed(JasmineOutput output, boolean checkrun) {
        for (JasmineProbe probe : probedefs.values()) {
            for (String name : probe.getOutputList()) {
                if (name.equals(output.getName())) {
                    if (!checkrun || (probe.getStatus() != JasmineProbe.PROBE_STOPPED)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Test if a Target is used
     *
     * @param target the target to check
     * @return true if target used
     */
    private boolean isTargetUsed(JasmineTarget target) {
        for (JasmineProbe probe : probedefs.values()) {
            if (probe.getTargetList().contains(target)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Test if an Indicator is used at least in one ProbeInstance.
     * @param indicator the indicator to check
     * @param checkrun  If true, check only if used by running probes
     * @return true if indic used in at least 1 probe.
     */
    private synchronized boolean isIndicatorUsed(JasmineIndicator indicator, boolean checkrun) {
        for (JasmineProbe probe : probedefs.values()) {
            for (String indic : getRecursiveIndicatorList(probe)) {
                if (indic.equals(indicator.getName())) {
                    if (!checkrun || (probe.getStatus() != JasmineProbe.PROBE_STOPPED)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Notify a state change for that probe
     *
     * @param running probe
     * @param state   See JasmineProbe.state
     * @param message Error message
     */
    public synchronized void setProbeState(JProbe running, int state, String message) {
        logger.debug("state =" + state);
        running.setStatus(state);
        running.setError(message);
        for (JasmineProbeListener listener : probeListeners) {
            listener.notifyEvent(running.getProbeDef());
        }
    }

    /**
     * Start probes waiting to be started.
     */
    private synchronized void startWaitingProbes() {

        for (JasmineProbe probe : probedefs.values()) {
            if (probe.getStatus() == JasmineProbe.PROBE_TOSTART) {
                logger.debug(probe.getId() + " trying to start");
                JProbe running = new JProbe(probe, this);
                try {
                    running.resume();
                } catch (JasmineProbeException e) {
                    logger.warn("Cannot start probe now: " + probe.getId());
                    continue;
                }
                probes.put(probe.getId(), running);
            }
        }
    }

    private synchronized List<String> getRecursiveIndicatorList(JasmineProbe probe) {
        List<String> ret = new ArrayList<String>();
        for (String iname : probe.getIndicatorList()) {
            ret.add(iname);
            JasmineIndicator indic = indicators.get(iname);
            if (indic != null) {
                JasmineCollectorService jcs = collectorservices.get(indic.getType());
                if (jcs != null) {
                    List<String> full = jcs.getDependantIndicators(indic);
                    ret.addAll(full);
                }
            }
        }
        return ret;
    }
}
