/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: OutputChangeCommand.java 9658 2012-01-24 12:27:04Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;

/**
 * Implements output-change command.
 * @author danesa
 */
@Component(name="OutputChangeCommand")
@Provides
public class OutputChangeCommand extends JasmineProbeCommand implements Command {

        /**
         * command name
         */
        private String name = "output-change";

        /**
         * necessary to be able to print out usage even when wrong options are provided by the user
         */
        private String requiredOptions = "-n outputName -t outputType";

        /**
         * command description
         */
        private String description = "Change an output definition";


        // ----------------------------------------------------------------------------
        // ipojo management
        // ----------------------------------------------------------------------------

        @Requires
        JasmineProbeManager probeManager = null;

        // ----------------------------------------------------------------------------
        // constructors
        // ----------------------------------------------------------------------------

        /**
         * Default Constructor
         */
        public OutputChangeCommand() {
            super();
            setParams(new OutputChangeCommandParams());
        }

        // ----------------------------------------------------------------------------
        // org.apache.felix.shell.Command implementation
        // ----------------------------------------------------------------------------

        /**
         * Returns the name of the command that is implemented by the interface.
         * The command name should not contain whitespace and should also be unique.
         * @return the name of the command.
         */
        public String getName() {
            return name;
        }

        /**
         * Returns a short description of the command; this description should be as short as possible.
         * This information is used when generating the command help information.
         * @return a short description of the command.
         */
        public String getShortDescription() {
            return description;
        }

        public String getRequiredOptions() {
            return requiredOptions;
        }

        public JasmineProbeManager getProbeManager() {
            return probeManager;
        }

        /**
         * Executes the command using the supplied command line, output print stream, and error print stream.
         * @param line the complete command line, including the command name.
         * @param out  the print stream to use for standard output.
         * @param err  the print stream to use for standard error.
         */
        public void execute(String line, PrintStream out, PrintStream err) {
            boolean validOptions = parseOptions(line, out, err);
            if (!validOptions) {
                err.println(getUsage());
                return;
            }

            OutputChangeCommandParams params = ((OutputChangeCommandParams) getParams());

            // get command arguments
            String outputName = params.getName();
            String outputType = params.getType();
            List<String> outputParams = params.getProps();

            // output to change
            JasmineOutput output = new JasmineOutput();
            output.setName(outputName);
            if (outputType != null) {
                output.setType(outputType);
            }
            Map<String, String> properties = new HashMap<String, String>();

            for (String outputParam : outputParams) {
                if (validParam(outputParam)) {
                    int indexEqual = outputParam.indexOf("=");
                    String outputParamName = outputParam.substring(0, indexEqual);
                    String outputParamValue = outputParam.substring(indexEqual + 1, outputParam.length());
                    properties.put(outputParamName, outputParamValue);
                } else {
                    err.println(name + " " + outputName + " failed");
                    // wrong parameter format
                    err.println("Wrong properties format for " + outputParam);
                    err.println("Expected format is paramName=paramValue");
                    err.println(getUsage());
                    params.reset();
                    return;
                }
            }
            output.setProperties(properties);
            params.reset();

            try {
                probeManager.changeOutput(output);
                out.println(name + " " + outputName +  " successfull");
            } catch (JasmineProbeException e) {
                err.println(name + " " + outputName + " failed");
                err.println(e);
                return;
            }
        }

}
