/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: OutputListCommand.java 9709 2012-01-31 13:22:28Z danesa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.Collection;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;

/**
 * Implements output-list command that list the name of all the created outputs.
 * @author danesa
 */
@Component(name="OutputListCommand")
@Provides
public class OutputListCommand extends JasmineProbeCommand implements Command {

        /**
         * command name
         */
        private String name = "output-list";

        /**
         * command description
         */
        private String description = "List all JASMINe Outputs, or show a particular Output";


        // ----------------------------------------------------------------------------
        // ipojo management
        // ----------------------------------------------------------------------------

        @Requires
        JasmineProbeManager probeManager = null;

        // ----------------------------------------------------------------------------
        // constructors
        // ----------------------------------------------------------------------------

        /**
         * Default Constructor
         */
        public OutputListCommand() {
            super();
            setParams(new OutputListCommandParams());
        }

        // ----------------------------------------------------------------------------
        // org.apache.felix.shell.Command implementation
        // ----------------------------------------------------------------------------

        /**
         * Returns the name of the command that is implemented by the interface.
         * The command name should not contain whitespace and should also be unique.
         * @return the name of the command.
         */
        public String getName() {
            return name;
        }

        /**
         * Returns a short description of the command; this description should be as short as possible.
         * This information is used when generating the command help information.
         * @return a short description of the command.
         */
        public String getShortDescription() {
            return description;
        }

        public JasmineProbeManager getProbeManager() {
            return probeManager;
        }

        /**
         * Executes the command using the supplied command line, output print stream, and error print stream.
         * @param line the complete command line, including the command name.
         * @param out  the print stream to use for standard output.
         * @param err  the print stream to use for standard error.
         */
        public void execute(String line, PrintStream out, PrintStream err) {
            boolean validOptions = parseOptions(line, out, err);
            if (!validOptions) {
                err.println(getUsage());
                return;
            }

            OutputListCommandParams params = ((OutputListCommandParams) getParams());

            // get command arguments
            String outputName = params.getName();
            boolean verbose = params.isVerbose();
            String type = params.getType();

            if (outputName == null) {
                listOutputs(type, verbose, out, err);
            } else {
                showOutput(outputName, out, err);
            }

            params.reset();
        }

        private void listOutputs(String type, boolean verbose, PrintStream out, PrintStream err) {
            String[] outputNames = probeManager.listOutputs(type);
            if (outputNames.length == 0) {
                String message1 = "There are no JasmineOutputs";
                String message2 = (type != null ? " of type " + type : "");
                out.println(message1 + message2);
                return;
            }
            for (String outputName : outputNames) {
                try {
                    JasmineOutput output = probeManager.getOutput(outputName);
                    if (verbose) {
                        out.println(output.toLine());
                    } else {
                        out.println(outputName);
                    }
                } catch (JasmineProbeException e) {
                    err.println(e.getMessage());
                }
            }
        }


        private void showOutput(String outputName, PrintStream out, PrintStream err) {
            Collection<JasmineOutput> outputs = probeManager.getOutputs();
            for (JasmineOutput output : outputs) {
                if (outputName.equals(output.getName())) {
                    out.println(output.toString());
                    return;
                }
            }
            err.println("There is no JasmineOutput named " + outputName);
        }
}
