/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.collectors;

import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasminePropertyInfo;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Base class for Collector Service. It allows the management of
 * the collectors of a given type corresponding to all the indicators of that type.
 * It allows to find out the properties that must be defined in order to create
 * an indicator of that type.
 *
 * @author durieuxp
 * @author danesa
 */
public abstract class JCollectorService implements JasmineCollectorService {

    /**
     * Logger.
     */
    protected Log logger = LogFactory.getLog(JCollector.class);

    /**
     * Map of collectors list corresponding to an indicator. The key is an indicator name.
     * For a given indicator, several collector instances may exist because an indicator
     * may be used by several running probes.
     */
    Map<String, List<JCollector>> collectorsList = new HashMap<String, List<JCollector>>();

    /**
     * Info about indicator Properties.
     * TODO Set an ArrayList to be conform to JasmineCollectorInfo (mapping issue ?)
     * warning: the MXBean should be conform ! (ArrayList may cause problem)
     */
    protected List<JasminePropertyInfo> properties;

    /**
     * CollectorInfo unique object
     */
    protected JasmineCollectorInfo collectorInfo;

    /**
     * Get the Collector corresponding to a JasmineIndicator.
     * Create it if necessary.
     * @param indicator the given indicator's definition
     * @param probe the definition of the probe in which the indicator is used
     * @return a Collector instance corresponding to the given indicator
     */
    abstract public JasmineCollector getCollector(JasmineIndicator indicator, JasmineProbe probe) throws JasmineCollectorException;

    /**
     * Remove all the collectors (if name and probeId are both null) - called when the service stopped.
     * Remove all the collectors attached to a given indicator (name not null, probeId null) - called
     * when an indicator is removed, or has changed.
     * Remove the collector corresponding to a given indicator and a given probe.
     * @param name  Indicator name
     * @param probeId Probe identifier
     */
    public synchronized void removeCollectors(String name, String probeId) {
        if (name == null) {
            // remove all the collectors of a a given type (the current service's type)
            logger.debug("Remove all the collectors corresponding to {0} indicators", getType());
            String[] indicatorNames = collectorsList.keySet().toArray(new String[0]);
            for (String indicatorName : indicatorNames) {
                removeCollectors(indicatorName);
            }
            return;
        }
        if (probeId == null) {
            logger.debug(name);
            removeCollectors(name);
        } else {
            logger.debug(name + " in " + probeId);
            removeCollector(name, probeId);
        }
    }
    /**
     * Remove a target. This operation has no effect by on collector services which
     * use implicit targets.
     * @param name the name of the target to remove
     */
    public synchronized void removeTarget(String name) {
    }

    /**
     * Remove all the collectors corresponding to an indicator
     * @param name the indicator name
     */
    private void removeCollectors(String name) {
        logger.debug(name);
        // maybe collectors not yet created
        if (collectorsList.isEmpty()) {
            return;
        }
        List<JCollector> colls = collectorsList.get(name);
        for (JCollector coll : colls) {
            // just in case
            coll.stopPolling();
            // mark it removed
            coll.remove();
        }
        collectorsList.remove(name);
    }

    /**
     * Remove the collector corresponding to an indicator used by a given probe
     * @param name the indicator name
     * @param probeId the probe's identifier
     */
    private void removeCollector(String name, String probeId) {
        List<JCollector> toRemove = new ArrayList<JCollector>();
        List<JCollector> colls = collectorsList.get(name);
        for (JCollector coll : colls) {
            if (coll.probeId == probeId) {
                toRemove.add(coll);
            }
        }
        for (JCollector coll: toRemove) {
            // stop it
            coll.stopPolling();
            // mark it removed
            coll.remove();
            // remove it from the list of collectors corresponding to
            // the indicator with the given name
            colls.remove(coll);
        }
    }

    /**
     * Add a collector instance to the list of collectors corresponding to a
     * given indicator.
     * @param name indicator name
     * @param coll collector instance
     */
    protected synchronized void addCollector(String name, JCollector coll) {
        logger.debug(name);
        if (!collectorsList.containsKey(name)) {
            collectorsList.put(name, new ArrayList<JCollector>());
        }
        collectorsList.get(name).add(coll);
    }

    /**
     * Find a collector corresponding to a given indicator and probe
     * @param name indicator name
     * @param probeId probe id
     * @return return the first collector corresponding to the given indicator and probe
     */
    protected synchronized JCollector getCollector(String name, String probeId) {
        logger.debug(name);
        // get collectors created for the given indicator
        List<JCollector> colls = collectorsList.get(name);
        if (colls == null) {
            return null;
        }
        for (int i = 0; i < colls.size(); i++) {
            JCollector coll = colls.get(i);
            if (probeId.equals(coll.probeId)) {
                // found a collector that is used by the given probe
                return coll;
            }
        }
        return null;
    }

    /**
     * Retrieve the description of all specific properties hidden in JasmineIndicator
     * Return a Map of all properties with their description
     * @deprecated
     * @return  a Map of property descriptions
     */
    public List<JasminePropertyInfo> getPropertiesInfo() {
        return properties;
    }

    /**
     * Get the Information about the Collector
     * @return JasmineIndicatorTypeInfo
     */
    public JasmineCollectorInfo getCollectorInfo() {
        return collectorInfo;
    }


    /**
     * Returns the List of indicators used by this indicator.
     * This default version return an empty list.
     * @param indic JasmineIndicator
     * @return Returns the List of indicators used by this indicator.
     */
    public List<String> getDependantIndicators(JasmineIndicator indic) {
        List<String> ret = new ArrayList<String>();
        return ret;
    }

    /**
     * Returns the List of targets used by this indicator.
     * This default version return an empty list.
     * @param indic JasmineIndicator
     * @return Returns the List of targets used by this indicator.
     */
    public List<String> getDependantTargets(JasmineIndicator indic) {
        List<String> ret = new ArrayList<String>();
        return ret;
    }

    public String getType() {
        return collectorInfo.getType();
    }

}
