/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.manager.internal;

import java.io.*;
import java.math.BigInteger;
import java.util.*;

import javax.management.ObjectName;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;

import org.apache.felix.ipojo.annotations.*;
import org.osgi.framework.ServiceReference;
import org.ow2.jasmine.probe.*;
import org.ow2.jasmine.probe.collector.JasmineAggregateService;
import org.ow2.jasmine.probe.collector.JasmineCollector;
import org.ow2.jasmine.probe.collector.JasmineCollectorException;
import org.ow2.jasmine.probe.collector.JasmineCollectorService;
import org.ow2.jasmine.probe.manager.JProbe;
import org.ow2.jasmine.probe.mbeans.AB;
import org.ow2.jasmine.probe.mbeans.CD;
import org.ow2.jasmine.probe.mbeans.TestOpenMBean;
import org.ow2.jasmine.probe.outer.JasmineOuterService;
import org.ow2.jasmine.probe.probemanager.ProbeManager;
import org.ow2.jasmine.probe.api.generated.*;
import org.ow2.jasmine.probe.probescheduler.SchedulerService;
import org.ow2.jonas.jmx.JmxService;
import org.ow2.jonas.lib.bootstrap.JProp;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implements the main service of the JasmineProbe module exposing the JasmineProbeManager interface
 * to the JasmineProbe module's clients.
 *
 * @author durieuxp
 */
@Component(name = "JasmineProbeManagerService")
@Provides
public class JProbeManager implements JasmineProbeManager, ProbeManager, JProbeManagerMXBean {

    protected static Log logger = LogFactory.getLog(JProbe.class);

    private ProbeConfig probeConfig = null;

    /**
     * JasmineProbe definitions
     */
    private Map<String, JasmineProbe> probedefs = new TreeMap<String, JasmineProbe>();

    /**
     * JasmineOutput definitions
     */
    private Map<String, JasmineOutput> outputs = new TreeMap<String, JasmineOutput>();

    /**
     * JasmineIndicator definitions
     */
    private Map<String, JasmineIndicator> indicators = new TreeMap<String, JasmineIndicator>();

    /**
     * JasmineTarget definitions
     */
    private Map<String, JasmineTarget> targets = new TreeMap<String, JasmineTarget>();

    /**
     *  probe objects
     */
    private Map<String, JProbe> probes = new HashMap<String, JProbe>();

    /**
     * List of the available Outer services.
     */
    private Map<String, JasmineOuterService> outerservices = new HashMap<String, JasmineOuterService>();

    /**
     * List of the available Collector services.
     */
    private Map<String, JasmineCollectorService> collectorservices = new HashMap<String, JasmineCollectorService>();

    /**
     * List of the custom available Aggregate services.
     */
    private Map<String, JasmineAggregateService> aggregateservices = new HashMap<String, JasmineAggregateService>();

    /**
     * Listeners for Probe changes
     */
    private List<JasmineProbeListener> probeListeners = new ArrayList<JasmineProbeListener>();

    /**
     * List of Outputs to be used when not defined in a probe.
     */
    private List<String> defaultOutputList = new ArrayList<String>();

    /**
     * Count used to generate probe ids
     */
    private int probeCount = 0;

    private ObjectName mbeanObjectName = null;

    // -----------------------------------------------------------------------------------
    // ipojo management
    // -----------------------------------------------------------------------------------

    @Validate
    public void start() {
        logger.debug("JasmineProbeManager service activated.");
        try {
            readConfig();

        } catch (Exception e) {
            // Cannot read configuration
            logger.error("Cannot read configuration");
        }

        // register mbean
        try {
            mbeanObjectName = new ObjectName("jasmine:dest=probe-manager");
            jmxService.registerMBean(this, mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot register mbean: " + e);
        }

        // register test mbeans
        registerMBeans();
    }

    private void registerMBeans() {
        try {
            ObjectName mbeanObjectName = ObjectName.getInstance("jasmine:dest=open-manager");

            AB ab_1 = new AB(1, 2);
            AB ab_2 = new AB(3, 4);
            AB ab_3 = new AB(5, 6);

            Long l_1 = new Long(2356);
            Long l_2 = new Long(45678);
            Long l_3 = new Long(1000004);
            Long[] longs = new Long[] {l_1, l_2, l_3};

            CD cd = new CD(longs, ab_3);

            ObjectName[] ons = new ObjectName[] {mbeanObjectName};

            AB[] elems = new AB[] {ab_1, ab_2};

            TestOpenMBean testMbean = new TestOpenMBean(ab_1, ab_2, cd, longs, elems, ons);

            jmxService.registerMBean(testMbean, mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot register mbean: " + e);
        }
    }

    private void unregisterMBeans() {
        try {
            ObjectName mbeanObjectName = ObjectName.getInstance("jasmine:dest=open-manager");
            jmxService.unregisterMBean(mbeanObjectName);
        } catch (Exception e) {
            logger.error("Cannot unregister mbean: " + e);
        }
    }


    @Invalidate
    public void stop() {
        logger.debug("JasmineProbeManager service stopped.");

        // unregister mbean
        jmxService.unregisterMBean(mbeanObjectName);

        // register test mbeans
        unregisterMBeans();
    }

    @Requires
    protected SchedulerService jasmineProbeScheduler = null;

    @Requires
    protected JmxService jmxService = null;

    @Bind(aggregate = true, optional = true)
    protected void bindOuterService(JasmineOuterService jos, ServiceReference sr) {
        String type = (String) sr.getProperty("output.type");
        if (type == null) {
            logger.warn("A service providing implementation for a JASMINe Probe Output type must use 'output.type' property to specify that type");
            return;
        }
        String mess = "A service for " + type + " output";
        JasmineOuterService old = outerservices.get(type);
        if (old != null) {
            logger.warn(mess + " already deployed");
            return;
        }
        outerservices.put(type, jos);
        logger.info(mess + " deployed to JASMINe Probe");

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindOuterService(JasmineOuterService jos, ServiceReference sr) {
        String type = (String) sr.getProperty("output.type");
        if (type == null) {
            logger.warn("Undeploying a service providing implementation for a JASMINe Probe Output that has not defined 'output.type' property");
            return;
        }
        if (outerservices.get(type) == null) {
            return;
        }
        // Get outputs having as type the service type
        String[] outputNames = listOutputs(type);
        for (String outputName : outputNames) {
            for (JasmineProbe probe : probedefs.values()) {
                if (probe.getStatus() == JasmineProbe.PROBE_RUNNING) {
                    // check that this probe does not uses one of the outputs
                    for (String output : probe.getOutputList()) {
                        if (output.equals(outputName)) {
                            logger.warn("Stop probe {0}", probe.getId());
                            try {
                                stopProbe(probe.getId());
                            } catch (JasmineProbeException e) {
                                logger.error(e);
                                //e.printStackTrace();
                            }
                            break;
                        }
                    }
                }
            }
        }
        outerservices.remove(type);
    }

    @Bind(aggregate = true, optional = true)
    protected void bindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        String type = (String) sr.getProperty("indicator.type");
        if (type == null) {
            logger.warn("A service providing implementation for a JASMINe Probe Indicator type must use 'indicator.type' property to specify that type");
            return;
        }
        String mess = "A service for " + type + " indicator";
        JasmineCollectorService old = collectorservices.get(type);
        if (old != null) {
            logger.warn(mess + " already deployed");
            return;
        }
        collectorservices.put(type, jcs);
        logger.info(mess + " deployed to JASMINe Probe");

        // check if some probes can be started
        startWaitingProbes();
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindCollectorService(JasmineCollectorService jcs, ServiceReference sr) {
        String type = (String) sr.getProperty("indicator.type");
        if (type == null) {
            logger.warn("Undeploying a service providing implementation for a JASMINe Probe Indicator that has not defined 'indicator.type' property");
            return;
        }
        if (collectorservices.get(type) == null) {
            return;
        }
        String[] indicatorNames = listIndicators(type);
        for (String indicatorName : indicatorNames) {
            for (JasmineProbe probe : probedefs.values()) {
                if (probe.getStatus() == JasmineProbe.PROBE_RUNNING) {
                    // check that this probe does not uses one of the indicators of the service's type
                    for (String indic : getRecursiveIndicatorList(probe)) {
                        if (indic.equals(indicatorName)) {
                            logger.warn("Stop probe {0}", probe.getId());
                            try {
                                stopProbe(probe.getId());
                            } catch (JasmineProbeException e) {
                                logger.error(e);
                            }
                            break;
                        }
                    }
                }
            }
        }
        collectorservices.remove(type);
    }

    @Bind(aggregate = true, optional = true)
    protected void bindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("A service providing specific implementation for an aggregate JASMINe Probe Indicator must define 'aggregate.type' property to specify its aggregate function");
            return;
        }
        JasmineAggregateService old = aggregateservices.get(type);
        if (old != null) {
            // A service for aggregate function already exists, use it if its not un-deployed.
            // Another approach could be to replace the old function by a new one.
            logger.warn("A service providing implementation for an aggregate JASMINe Probe Indicator already defined for " + type + " function. Please check deployment plan and undeploy old service !");
            return;
        }
        aggregateservices.put(type, jas);
        logger.info("A service providing implementation for an aggregate JASMINe Probe Indicator with {0} function deployed to JasmineProbe", type);
    }

    @Unbind(aggregate = true, optional = true)
    protected void unbindAggregateService(JasmineAggregateService jas, ServiceReference sr) {
        // get aggregate function type
        String type = (String) sr.getProperty("aggregate.type");
        if (type == null) {
            logger.warn("Undeploying a service providing implementation for an aggregate JASMINe Probe Indicator that has not defined 'indicator.type' property");
            return;
        }
        if (aggregateservices.containsKey(type)) {
            aggregateservices.remove(type);
            logger.info("A service providing implementation for an aggregate JASMINe Probe Indicator with {0} function undeployed from JasmineProbe", type);
            logger.info("{0} JasmineAggregateService unbound into JasmineProbe.", type);
        }
    }

    // -----------------------------------------------------------------------------------
    //  internal interface
    // -----------------------------------------------------------------------------------

    /**
     * return the scheduler
     * @return referencee on the SchedulerService
     */
    public SchedulerService getScheduler() {
        return jasmineProbeScheduler;
    }

    /**
     * Get the OuterService for a given type
     */
    public JasmineOuterService getOuterService(String type) {
        return outerservices.get(type);
    }

    /**
     * Get the CollectorService for a given type
     */
    public JasmineCollectorService getCollectorService(String type) {
        return collectorservices.get(type);
    }

    // -----------------------------------------------------------------------------------
    //  JasmineProbeManager implementation
    // -----------------------------------------------------------------------------------

    /**
     * Create a new probe defined by its data
     *
     * @param probe probe description
     * @return the Id to be used to reference this probe later.
     */
    public synchronized String createProbe(JasmineProbe probe) throws JasmineProbeException {
        // Generate an Id if not supplied
        String id = probe.getId();
        logger.debug("creating probe " + id);
        if (id == null) {
            // If possible, take the name of unique indicator
            if (probe.getIndicatorList().size() == 1) {
                String strindic = probe.getIndicatorList().get(0);
                boolean found = false;
                for (JasmineProbe p : probedefs.values()) {
                    if (p.getId().equals(strindic)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    id = strindic;
                }
            }
            if (id == null) {
                id = getNewProbeId();
            }
            probe.setId(id);
        } else {
            // Check Id not already used
            /*
            for (JasmineProbe p : probedefs.values()) {
                if (p.getId().equals(id)) {
                    logger.error("Id already used: " + id);
                    throw new JasmineProbeException("Id already used");
                }
            }*/
            JasmineProbe p = probedefs.get(id);
            if (p != null) {
                throw new JasmineProbeException("Probe id " + id + " already used");
            }
        }

        // Check validity of the JasmineProbe
        if (probe.getIndicatorList() == null || probe.getIndicatorList().isEmpty()) {
             throw new JasmineProbeException("A probe must have at least one indicator");
        }
        if (probe.getOutputList() == null || probe.getOutputList().isEmpty()) {
            probe.setOutputList(defaultOutputList);
        }

        probedefs.put(id, probe);
        return id;
    }

    /**
     * Remove a probe.
     *
     * @param probeId probe identifier.
     */
    public synchronized void removeProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot remove probe with non-existent id " + probeId);
        }

        // Check if probe is running
        JProbe probe = probes.get(probeId);
        if (probe != null) {
            probe.suspend();
            probe.resetIndicators();
            probe.resetOutputs();
            probe.resetTargets();
            probes.remove(probeId);
        }
        logger.debug("removing probe " + probeId);
        probedefs.remove(probeId);
    }

    /**
     * Remove all the managed probes.
     * Stop probe running before.
     *
     * @throws JasmineProbeException
     */
    public synchronized void removeAllProbes() throws JasmineProbeException {
        logger.debug("");

        // must build a list first, to avoid ConcurrentModificationException.
        List<String> namelist = new ArrayList<String>();
        for (String n : probedefs.keySet()) {
            namelist.add(n);
        }
        for (String name : namelist) {
            removeProbe(name);
        }
    }

    /**
     * Get all the Probe definitions in one operation.
     * @return all the probe definitions
     */
    public synchronized List<JasmineProbe> getProbes() {
        logger.debug("");
        List<JasmineProbe> ret = new ArrayList<JasmineProbe>();
        /** Descending order
        for (Iterator<String> keys = ((TreeMap<String, JasmineProbe>) probedefs).descendingKeySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineProbe) probedefs.get(keys.next()));
        }
        */
        /**
         * Iterate over keys (probe ids) in ascending order.
         */
        for (Iterator<String> keys = ((TreeMap<String, JasmineProbe>) probedefs).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineProbe) probedefs.get(keys.next()));
        }
        return ret;
    }

    /**
     * @return all the probe ids
     */
    public synchronized String [] listProbes() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineProbe>) probedefs).keySet().iterator(); keys.hasNext(); ) {
            ret.add(keys.next());
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get a Probe by its name
     * @param probeId probe identifier.
     * @return the Probe definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineProbe getProbe(String probeId) throws JasmineProbeException {
        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot find probe with id " + probeId);
        }
        return def;
     }

    /**
     * Start a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be started.
     */
    public synchronized void startProbe(String probeId) throws JasmineProbeException {
        logger.debug(probeId);

        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with non-existent id " + probeId);
        }

        // Check if already running
        JProbe running = probes.get(probeId);
        if (running != null) {
            switch (running.getStatus()) {
                case JasmineProbe.PROBE_RUNNING:
                    logger.info("Probe already running: " + probeId);
                    return;
                case JasmineProbe.PROBE_STARTED:
                    logger.info("Probe already started: " + probeId);
                    return;
                case JasmineProbe.PROBE_FAILED:
                    logger.info("restarting a failed probe: " + probeId);
                    break;
                case JasmineProbe.PROBE_STOPPED:
                    logger.info("restarting a stopped probe: " + probeId);
                    break;
                default:
                    logger.warn("Bad state for: " + probeId);
                    break;
            }
        } else {
            // Create the JProbe object (a running probe)
            running = new JProbe(def, this);
            probes.put(probeId, running);
        }
        running.resume();
    }

    /**
     * Stop a probe.
     *
     * @param probeId probe identifier.
     * @throws JasmineProbeException the probe could not be stopped.
     */
    public synchronized void stopProbe(String probeId) throws JasmineProbeException {
        logger.debug(probeId);

        // Retrieve the Probe by its Id
        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Cannot start probe with non-existent id " + probeId);
        }

        // Check if already running
        JProbe running = probes.get(probeId);
        if (running == null) {
            throw new JasmineProbeException("This probe is not running: " +  probeId);
        }

        // Stop the probe by halting polling and changing status to JasmineProbe.PROBE_STOPPED
        running.suspend();
    }

    /**
     * Start all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void startAllProbes() throws JasmineProbeException {
        logger.debug("");
        for (JasmineProbe probe : probedefs.values()) {
            try {
                startProbe(probe.getId());
            } catch (Exception e) {
                logger.warn("could not start probe " + probe.getId() + " :" + e);
            }
        }
    }

    /**
     * Stop all the managed probes.
     *
     * @throws JasmineProbeException
     */
    public synchronized void stopAllProbes() throws JasmineProbeException {
        logger.debug("");
        for (JProbe probe : probes.values()) {
            try {
                probe.suspend();
            } catch (Exception e) {
                logger.warn("could not stop probe " + probe.getProbeDef().getId() + " :" + e);
            }
        }
    }

    /**
     * Change a Probe with new parameters
     * All parameters that are not supplied rest unchanged.
     *
     * @param newdef  partial probe description
     */
    public synchronized void changeProbe(JasmineProbe newdef) throws JasmineProbeException {

        // Retrieve the Probe by its Id
        String probeId = newdef.getId();
        logger.debug(probeId);

        JasmineProbe def = probedefs.get(probeId);
        if (def == null) {
            throw new JasmineProbeException("Probe not found: " + probeId);
        }

        // Stop it first if it is running.
        // It will be restarted after changed
        boolean restart = false;
        JProbe running = probes.get(probeId);
        if (running != null) {
            restart = running.suspend();
        }

        // Change period
        if (newdef.getPeriod() > 0) {
            def.setPeriod(newdef.getPeriod());
        }

        // Change output list
        if (newdef.getOutputList().size() > 0) {
            def.setOutputList(newdef.getOutputList());
            if (running != null) {
                running.resetOutputs();
            }
        }

        // Change indicator list
        if (newdef.getIndicatorList().size() > 0) {
            def.setIndicatorList(newdef.getIndicatorList());
            if (running != null) {
                running.resetIndicators();
            }
        }

        // Change target list
        if (newdef.getTargetList().size() > 0) {
            def.setTargetList(newdef.getTargetList());
            if (running != null) {
                running.resetTargets();
            }
        }

        // Restart probe if it was suspended
        if (restart) {
            try {
                running.resume();
            } catch (JasmineProbeException e) {
                logger.warn("Probe {0} changed. It was stopped but could not be restarted: {1}", probeId, e.getMessage());
            }
        }
    }

    /**
     * Set a new value for probe period
     * @param probeId probe identifier.
     * @param period in seconds
     */
    public void changeProbePeriod(String probeId, int period) throws JasmineProbeException {
        // Change value in definition
        JasmineProbe def = probedefs.get(probeId);
        def.setPeriod(period);

        // If probe is running, must change its timer.
        JProbe running = probes.get(probeId);
        if (running != null && running.isRunning()) {
             running.resetTimer();
        }
    }

    /**
     * Register a new Output in the ProbeManager
     * A name is associated to the Output definition.
     * This Output will be usable later in probes.
     *
     * @param newoutput output description
     * @return the Id to be used to reference this output later.
     */
    public String createOutput(JasmineOutput newoutput) throws JasmineProbeException {
        logger.debug("");
        return createOutputInternal(newoutput, true);
    }

    /**
     * Change an output. The output must exist.
     *
     * @param newdef  output changes description
     * @throws JasmineProbeException is thrown if the output does not exist.
     */
    public void changeOutput(JasmineOutput newdef) throws JasmineProbeException {
        // Find the JasmineOutput to be changed
        String name = newdef.getName();
        logger.debug(name);

        JasmineOutput olddef = outputs.get(name);
        if (olddef == null) {
            throw new JasmineProbeException("This output does not exist: " +  name);
        }

        // Stop all running probes using this output.
        List<JasmineProbe> probesUsingOld = probesUsingDef(olddef, true);
        for (JasmineProbe usingOld : probesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            running.suspend();
            running.resetOutputs();
        }

        // remove Outers defined on this Output
        JasmineOuterService jos = outerservices.get(olddef.getType());
        if (jos != null) {
            jos.removeOuters(name);
        }

        // Change type
        if (newdef.getType() != null) {
            olddef.setType(newdef.getType());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            if (newval == null || newval.isEmpty()) {
                olddef.getProperties().remove(key);
            } else {
                // got new value to set
                // If an old value exists, it is replaced by the new value
                olddef.getProperties().put(key, newval);
            }
        }

        // restart probes
        for (JasmineProbe usingOld : probesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            running.resume();
        }
    }

    /**
     * remove an output. The output must exist, otherwise an exception is thrown.
     * The output must not be used by a probe, otherwise an exception is thrown.
     *
     * @param name ident of the output to remove
     * @throws JasmineProbeException is thrown if the output does not exist,
     *                               or if it is used by a running probe.
     */
    public void removeOutput(String name) throws JasmineProbeException {
        logger.debug("");
        JasmineOutput output = outputs.get(name);
        if (output == null) {
            throw new JasmineProbeException("This output does not exist: " +  name);
        }
        if (isDefUsed(output, true)) {
            throw new JasmineProbeException("This output is used and cannot be removed: " +  name);
        }
        JasmineOuterService jos = outerservices.get(output.getType());
        if (jos != null) {
            jos.removeOuters(name);
        }
        outputs.remove(name);
    }

    /**
     * Get the list of all outputs or of outputs having a given type
     * @param type the given type. If null, get all the outputs.
     */
    public synchronized String [] listOutputs(final String type) {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineOutput>) outputs).keySet().iterator(); keys.hasNext(); ) {
            String name = keys.next();
            if (type == null) {
                ret.add(name);
            } else {
                JasmineOutput out = outputs.get(name);
                if (type.equals(out.getType())) {
                    ret.add(name);
                }
            }
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all the Output definitions in one operation.
     * @return all the output definitions
     */
    public synchronized List<JasmineOutput> getOutputs() {
        logger.debug("");
        List<JasmineOutput> ret = new ArrayList<JasmineOutput>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineOutput>) outputs).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineOutput) outputs.get(keys.next()));
        }
        return ret;
    }

    @Override
    public void addDefaultOutput(String name) throws JasmineProbeException {
        // TODO Auto-generated method stub
        if (!outputs.containsKey(name)) {
            throw new JasmineProbeException("This output does not exist: " +  name);
        }
        defaultOutputList.add(name);
    }

    @Override
    public synchronized String[] listDefaultOutputs() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (String id : defaultOutputList) {
            ret.add(id);
        }
        return ret.toArray(new String [0]);
    }

    @Override
    public void removeDefaultOutput(String name) throws JasmineProbeException {
        if (!defaultOutputList.contains(name)) {
            throw new JasmineProbeException("This output is not a default output: " +  name);
        }
    }

    /**
     * Get the list of all outputs of a given type.
     * @param type the given output type
     * @return List of the outputs having that type
     */
    public synchronized List<JasmineOutput> getOutputs(final String type) {
        logger.debug("");
        List<JasmineOutput> ret = new ArrayList<JasmineOutput>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineOutput>) outputs).keySet().iterator(); keys.hasNext(); ) {
            JasmineOutput output = outputs.get(keys.next());
            if (output.getType().equals(type)) {
                ret.add(output);
            }
        }
        return ret;
    }

    /**
     * Get an Output by its name
     * @param id output identifier.
     * @return the Output definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineOutput getOutput(String id) throws JasmineProbeException {
        logger.debug("");
        JasmineOutput def = outputs.get(id);
        if (def == null) {
            throw new JasmineProbeException("Cannot find output with id " + id);
        }
        return def;
     }

    /**
     * get the list of possible properties for Output, depending of its type.
     *
     * @param type type of the Output (console, file, ...)
     * @return List of JasminePropertyInfo
     */
    public List<JasminePropertyInfo> getOutputPropertyInfos(String type) {
        if (type == null || type.length() == 0) {
            logger.error("No type provided");
            return null;
        }
        logger.debug(type);

        // Find the appropriate OuterService
        JasmineOuterService jos = outerservices.get(type);
        if (jos == null) {
            logger.error("No OuterService found for type " + type);
            return null;
        }
        return jos.getPropertiesInfo();
    }

    /**
     * @return the available output types.
     */
    public synchronized Set<String> getOutputTypes() {
        return outerservices.keySet();
    }

    /**
     * Get the list of probes using this output
     * @param name  Output name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    public synchronized Set<String> getOutputCurrentUse(String name) throws JasmineProbeException {
        Set<String> ret = new HashSet<String>();
        for (JasmineProbe probe : probedefs.values()) {
            for (String output : probe.getOutputList()) {
                if (output.equals(name)) {
                    ret.add(probe.getId());
                }
            }
        }
        return ret;
    }

    /**
     * Create a new Indicator
     *
     * @param newindic indicator description
     * @return the Id to be used to reference this indicator later.
     */
    public String createIndicator(JasmineIndicator newindic) throws JasmineProbeException {
        return createIndicatorInternal(newindic, true);
    }

    /**
     * Change an indicator. It must exist.
     *
     * @param newdef indicator description
     * @throws JasmineProbeException is thrown if the indicator does not exist,
     *                               or if it is used by a running probe.
     */
    public void changeIndicator(JasmineIndicator newdef) throws JasmineProbeException {

        // Find the JasmineIndicator to be changed
        String name = newdef.getName();
        logger.debug(name);

        JasmineIndicator olddef = indicators.get(name);
        if (olddef == null) {
            throw new JasmineProbeException("This indicator does not exists: " +  name);
        }

        // Stop all running probes using this indicator.
        List<JasmineProbe> probesUsingOld = probesUsingDef(olddef, true);
        for (JasmineProbe usingOld : probesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            running.suspend();
            running.resetIndicators();
        }

        // Remove all the collectors defined for this indicator
        JasmineCollectorService jcs = collectorservices.get(olddef.getType());
        if (jcs != null) {
            jcs.removeCollectors(name, null);
        }

        // Change type
        if (newdef.getType() != null) {
            olddef.setType(newdef.getType());
        }

        // Change scale
        if (newdef.getScale() != olddef.getScale()) {
            olddef.setScale(newdef.getScale());
        }

        // Change modified sources
        if (! newdef.getSources().isEmpty()) {
            olddef.setSources(newdef.getSources());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            if (newval == null || newval.isEmpty()) {
                olddef.getProperties().remove(key);
            } else {
                // got new value to set
                // If an old value exists, it is replaced by the new value
                olddef.getProperties().put(key, newval);
            }
        }

        // restart probes
        for (JasmineProbe usingOld : probesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            running.resume();
        }
    }


    /**
     * remove an indicator
     *
     * @param name ident of the indicator to remove
     */
    public void removeIndicator(String name) throws JasmineProbeException {
        logger.debug(name);
        JasmineIndicator indic = indicators.get(name);
        if (indic == null) {
            throw new JasmineProbeException("This indicator does not exist: " +  name);
        }
        if (isDefUsed(indic, true)) {
            throw new JasmineProbeException("This indicator is used and cannot be removed: " +  name);
        }

        JasmineCollectorService jcs = collectorservices.get(indic.getType());
        if (jcs != null) {
            jcs.removeCollectors(name, null);
        }
        indicators.remove(name);
    }

    /**
     * Get an Indicator by its name
     * @param name indicator name.
     * @return the definition
     * @throws org.ow2.jasmine.probe.JasmineProbeException
     */
     public synchronized JasmineIndicator getIndicator(String name) throws JasmineProbeException {
        logger.debug(name);
        JasmineIndicator def = indicators.get(name);
        if (def == null) {
            /*
            JasmineProbeException e =  new JasmineProbeException("Cannot find indicator with id " + id);
            e.printStackTrace();
            throw e;*/
            throw new JasmineProbeException("Cannot find indicator " + name);
        }
        return def;
     }

    /**
     * Get all the Indicator definitions in one operation.
     * @return all the indicator definitions
     */
    public synchronized List<JasmineIndicator> getIndicators() {
        logger.debug("");
        List<JasmineIndicator> ret = new ArrayList<JasmineIndicator>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineIndicator>) indicators).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineIndicator) indicators.get(keys.next()));
        }
        return ret;
    }

    /**
     * Get the list of indicators having a given type.
     * @param type the given indicator type
     * @return List of the indicators having that type
     */
    public synchronized List<JasmineIndicator> getIndicators(final String type) {
        logger.debug("");
        List<JasmineIndicator> ret = new ArrayList<JasmineIndicator>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineIndicator>) indicators).keySet().iterator(); keys.hasNext(); ) {
            JasmineIndicator indic = indicators.get(keys.next());
            if (indic.getType().equals(type)) {
                ret.add(indic);
            }
        }
        return ret;
    }

    /**
     * @return all the indicator names
     */
    public synchronized String [] listIndicators(final String type) {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineIndicator>) indicators).keySet().iterator(); keys.hasNext(); ) {
            String name = keys.next();
            if (type == null) {
                ret.add(name);
            } else {
                JasmineIndicator indic = indicators.get(name);
                if (type.equals(indic.getType())) {
                    ret.add(name);
                }
            }
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all infiormation about the type of indicator
     * @param type type of the Indicator (Jmx, Lewys, ...)
     * @return JasmineIndicatorTypeInfo
     */
    public JasmineCollectorInfo getIndicatorTypeInfo(String type) {
        if (type == null || type.length() == 0) {
            logger.error("No type provided");
            return null;
        }
        logger.debug(type);

        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(type);
        if (jcs == null) {
            logger.error("No CollectorService found for type " + type);
            return null;
        }
        return jcs.getCollectorInfo();
    }

    /**
     * @return the available indicator types.
     */
    public synchronized Set<String> getIndicatorTypes() {
        logger.debug("");
        return collectorservices.keySet();
    }

    /**
     * Get the list of probes using this indicator
     * @param name  Indicator name
     * @return list of probe ids
     * @throws JasmineProbeException bad parameter
     */
    public synchronized Set<String> getIndicatorCurrentUse(String name) throws JasmineProbeException {
        Set<String> ret = new HashSet<String>();
        for (JasmineProbe probe : probedefs.values()) {
            for (String indic : getRecursiveIndicatorList(probe)) {
                if (indic.equals(name)) {
                    ret.add(probe.getId());
                }
            }
        }
        return ret;
    }

    /**
     * Create a new target
     *
     * @param newtarget target description
     * @return the Id to be used to reference this target later.
     */
    public synchronized String createTarget(JasmineTarget newtarget) throws JasmineProbeException {
        // name chosen by the user.
        String name = newtarget.getName();
        if (name == null || name.length() == 0) {
            throw new JasmineProbeException("No valid target name");
        }
        logger.debug(name);

        // Check if already known
        JasmineTarget oldtarget = targets.get(name);
        if (oldtarget != null) {
            logger.debug("target already known: " + name);
            if (oldtarget.equals(newtarget)) {
                // If same target already created, just return.
                return name;
            }

            // New definition replaces old definition
            // -if no running probes are using this target
            logger.warn("old definition: " + oldtarget);
            logger.warn("new definition: " + newtarget);
            if (isDefUsed(oldtarget, true)) {
                throw new JasmineProbeException(name + " target cannot be re-created as a running probe is using it");
            }

            // find probes using this target ;
            List<JasmineProbe> probesUsingOld = probesUsingDef(oldtarget, false);
            for (JasmineProbe usingOld : probesUsingOld) {
                JProbe running = probes.get(usingOld.getId());
                if (running != null) {
                    running.resetTarget(oldtarget);
                }
            }

            // remove old definition and replace by the new one.
            targets.remove(name);
        }
        targets.put(name, newtarget);
        return name;
    }

    /**
     * Change a target. It must exist, otherwise an exception is thrown.
     *
     * @param newdef new target description
     * @throws JasmineProbeException is thrown if the target does not exist.
     */
    public void changeTarget(JasmineTarget newdef) throws JasmineProbeException {
        // Find the JasmineTarget to be changed
        String name = newdef.getName();
        logger.debug(name);

        JasmineTarget olddef = targets.get(name);
        if (olddef == null) {
            throw new JasmineProbeException("This target does not exist: " +  name);
        }

        // Stop all running probes using this output.
        List<JasmineProbe> runningProbesUsingOld = probesUsingDef(olddef, true);
        for (JasmineProbe usingOld : runningProbesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            running.suspend();
        }

        // Change type
        if (newdef.getType() != null) {
            olddef.setType(newdef.getType());
        }

        // Change modified properties
        for (String key : newdef.getProperties().keySet()) {
            String newval = newdef.getProperties().get(key);
            if (newval == null || newval.isEmpty()) {
                olddef.getProperties().remove(key);
            } else {
                logger.debug(key+"="+newval);
                // got new value to set
                // If an old value exists, it is replaced by the new value
                olddef.getProperties().put(key, newval);
            }
        }

        // find probes using this target ;
        List<JasmineProbe> probesUsingOld = probesUsingDef(olddef, false);
        for (JasmineProbe usingOld : probesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            if (running != null) {
                // reset targets
                running.resetTargets();
            }
        }

        // restart probes
        for (JasmineProbe usingOld : runningProbesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            running.resume();
        }
    }

    /**
     * remove a target
     *
     * @param name ident of the target to remove
     */
    public void removeTarget(String name) throws JasmineProbeException {
        logger.debug(name);
        JasmineTarget target = targets.get(name);
        if (target == null) {
            throw new JasmineProbeException("This target does not exist: " +  name);
        }
        if (isDefUsed(target, true)) {
            throw new JasmineProbeException("This target is used and cannot be removed: " +  name);
        }
        // find probes using this target ;
        List<JasmineProbe> probesUsingOld = probesUsingDef(target, false);
        for (JasmineProbe usingOld : probesUsingOld) {
            JProbe running = probes.get(usingOld.getId());
            if (running != null) {
                // reset targets
                running.resetTargets();
            }
        }


        targets.remove(name);
    }

    /**
     * @return all the target names
     */
    public synchronized String [] listTargets() {
        logger.debug("");
        List<String> ret = new ArrayList<String>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineTarget>) targets).keySet().iterator(); keys.hasNext(); ) {
            ret.add(keys.next());
        }
        return ret.toArray(new String [0]);
    }

    /**
     * Get all the Target definitions in one operation.
     * @return all the target definitions
     */
    public synchronized List<JasmineTarget> getTargets() {
        logger.debug("");
        List<JasmineTarget> ret = new ArrayList<JasmineTarget>();
        for (Iterator<String> keys = ((TreeMap<String, JasmineTarget>) targets).keySet().iterator(); keys.hasNext(); ) {
            ret.add((JasmineTarget) targets.get(keys.next()));
        }
        return ret;
    }

    /**
     * Get a JasmineTarget by its name
     * @param name
     * @return JasmineTarget definition
     */
    public JasmineTarget getTarget(String name) {
        JasmineTarget target = targets.get(name);
        return target;
    }

    /**
     * get the list of possible properties for Target, depending of its type.
     *
     * @param type type of the Target (jmx, ...)
     * @return List of JasminePropertyInfo
     */
    public List<JasminePropertyInfo> getTargetPropertyInfos(String type) {
        if (type == null || type.length() == 0) {
            logger.error("No type provided");
            return null;
        }
        logger.debug(type);

        // TODO This must be changed when targets may have other type than jmx
        ArrayList<JasminePropertyInfo> ret = new ArrayList<JasminePropertyInfo>();
        ret.add(new JasminePropertyInfo("url", "target jmx URL", true));
        ret.add(new JasminePropertyInfo("user", "user name to access the target", false));
        ret.add(new JasminePropertyInfo("password", "password associated to the user name", false));
        ret.add(new JasminePropertyInfo("protocolProviders", "optional protocol provider", false));
        return ret;
    }

    /**
     * @return the available indicator types.
     */
    public synchronized Set<String> getTargetTypes() {
        logger.debug("");
        // TODO This must be changed when targets may have other type than jmx
        HashSet<String> ret = new HashSet<String>();
        ret.add("jmx");
        return ret;
    }

    /**
     * Register a ProbeListener in order to be notified by probe state changes.
     * Maybe other events will be considered...
     *
     * @param listener object that treats the probe state change
     */
    public synchronized void addProbeListener(JasmineProbeListener listener) {
        logger.debug("");
        probeListeners.add(listener);
    }

    /**
     * Remove a ProbeListener previously registered.
     *
     * @param listener object that treats the probe state change
     */
    public synchronized void removeProbeListener(JasmineProbeListener listener) {
        logger.debug("");
        probeListeners.remove(listener);
    }

    /**
     * Save the current configuration in the specified xml file
     *
     * @param path the xml file, or null if saved in the default configuration file.
     * @throws JasmineProbeException
     */
    public void saveConfig(String path) throws JasmineProbeException {
        logger.debug(path);
        FileOutputStream outputStream = null;
        try {
            outputStream = new FileOutputStream(new File(JProp.getConfDir(), "probe-config.xml"));
        } catch (FileNotFoundException e) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'" + e);
        }
        if (outputStream == null) {
            throw new JasmineProbeException("Cannot write the config file 'probe-config.xml'");
        }
        try {
            saveConfigFile(outputStream);
        } catch (Exception e) {
            logger.warn("could not save config: " + e);
            throw new JasmineProbeException("could not save config: " + e.getMessage());
        }
    }

    /**
     * Load the specified xml configuration file
     * The configuration will be merged with the current one.
     *
     * @param path the xml file.
     * @throws JasmineProbeException
     */
    public void loadConfig(String path) throws JasmineProbeException {
        logger.debug(path);
        try {
            loadConfigFile(new FileInputStream(path));
        } catch (Exception e) {
            logger.warn("Cannot load config file: " + e);
            throw new JasmineProbeException("Cannot load config file: " + e.getMessage());
        }
        // Start probes marked as started in the file
        startWaitingProbes();
    }

    // -----------------------------------------------------------------
    // ProbeManager implementation
    // -----------------------------------------------------------------

    /**
     * Get a Collector by its Indicator name
     * @param probeid
     * @param indicatorName indicator name
     * @return The collector corresponding to an indicator.
     */
    public synchronized JasmineCollector getCollector(String indicatorName, String probeid) throws JasmineProbeException {

        JasmineIndicator indicator = indicators.get(indicatorName);
        if (indicator == null) {
            throw new JasmineProbeException("Cannot return collector for non-existent indicator " + indicatorName);
        }

        // Find the appropriate CollectorService
        JasmineCollectorService jcs = collectorservices.get(indicator.getType());
        if (jcs == null) {
            logger.error("No CollectorService found for {0} having type {1}", indicatorName, indicator.getType());
            throw new JasmineProbeException("No CollectorService found for " + indicatorName);
        }
        // Get the Collector from the CollectorService
        JasmineCollector collector = null;
        try {
            collector = jcs.getCollector(indicator, probedefs.get(probeid));
        } catch (JasmineCollectorException e) {
            logger.error("Cannot get Collector for indicator " + indicatorName + ": " + e);
            throw new JasmineProbeException("Cannot get Collector for indicator " + indicatorName);
        }
        return collector;
    }

    /**
     * @return If AggregateServices exist, return the functions these services implement (the values of aggregate.type properties).
     * Else, return null.
     */
    public synchronized List<String> getAggregateFuntions() {
        if (!aggregateservices.isEmpty()) {
            List<String> functions = new ArrayList<String>();
            Set<String> keySet = aggregateservices.keySet();
            for (String function : keySet) {
                functions.add(function);
            }
            return functions;
        }
        return null;
    }

    /**
     * Return the AggregateService for a given aggregate function
     * @param key the aggregate function that is searched
     * @return the AggregateService having aggregate.type equal to key. Else, return null.
     */
    public synchronized JasmineAggregateService getAggregate(String key) {
        if (aggregateservices.containsKey(key)) {
            return aggregateservices.get(key);
        }
        return null;
    }

    /**
     * @return the current domain name.
     */
    public String getDomainName() {
       return jmxService.getDomainName();
    }

    /**
     * @return the current server name.
     */
    public String getServerName() {
       return jmxService.getJonasServerName();
    }


    // --------------------------------------------------------------------------------------
    // Private methods
    // --------------------------------------------------------------------------------------

    /**
     * Create a new Output
     * @param newoutput
     * @param check
     * @return
     * @throws JasmineProbeException
     */
    private synchronized String createOutputInternal(JasmineOutput newoutput, boolean check) throws JasmineProbeException {
        // Id chosen by the user.
        String name = newoutput.getName();
        if (name == null || name.length() == 0) {
            throw new JasmineProbeException("No valid output name");
        }

        // Check that a type is provided
        String type = newoutput.getType();
        if (type == null) {
            throw new JasmineProbeException("No type provided");
        }

        // Check if already known
        JasmineOutput oldoutput = outputs.get(name);
        if (oldoutput != null) {
             logger.debug(name + " output is already created");
             if (oldoutput.equals(newoutput)) {
                // If same output already created, just return.
                 return name;
             }
        }

        if (check) {
            // Check that the type is supported
            JasmineOuterService jos = outerservices.get(type);
            if (jos == null) {
                throw new TypeNotSupportedException("Output type not supported: " + type);
            }

            // Check that mandatory properties are supplied
            String missinglist = "";
            for (JasminePropertyInfo jpi : jos.getPropertiesInfo()) {
                if (jpi.isRequired()) {
                    boolean found = false;
                    for (String key : newoutput.getProperties().keySet()) {
                        if (key.equals(jpi.getName())) {
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        missinglist += " " + jpi.getName();
                    }
                }
            }
            if (missinglist.length() > 0) {
                logger.warn("Missing mandatory properties:" + missinglist);
                for (String key : newoutput.getProperties().keySet()) {
                    logger.warn("found " + key);
                }
                throw new PropertyMissingException("Missing mandatory properties:" + missinglist);
            }
        }

        // Check if already known
        if (oldoutput != null) {
            // New definition replaces old definition
            // -if no running probes are using this output
            logger.warn("old definition: " + oldoutput);
            logger.warn("new definition: " + newoutput);
            if (isDefUsed(oldoutput, true)) {
                throw new JasmineProbeException(name + " output cannot be re-created as a running probe is using it");
            }

            // find probes using this output ;
            // even if not running, some probes may have reference to an outer corresponding to this indicator
            List<JasmineProbe> probesUsingOld = probesUsingDef(oldoutput, false);
            for (JasmineProbe usingOld : probesUsingOld) {
                JProbe running = probes.get(usingOld.getId());
                if (running != null) {
                    // reset outputs - the corresponding outers are recreated at the next resume
                    running.resetOutputs();
                }
            }

            // remove old definition and replace by the new one.
            outputs.remove(name);
        }

        outputs.put(name, newoutput);
        return name;
    }

    /**
     * Create a new Indicator
     *
     * @param newindic indicator description
     * @param check true if check properties against the Collector (that should be there)
     * @return the Id to be used to reference this indicator later.
     */
    public synchronized String createIndicatorInternal(JasmineIndicator newindic, boolean check) throws JasmineProbeException {

        // Name chosen by the user
        String name = newindic.getName();
        if (name == null || name.length() == 0) {
            // TODO: generate Id if not provided ?
            throw new JasmineProbeException("Automatic indicator name is not supported yet");
        }

        // Check that a type is provided
        String type = newindic.getType();
        if (type == null) {indicators.get(name);
            throw new TypeNotSupportedException("No type provided");
        }

        // Check if already known
        JasmineIndicator oldindic = indicators.get(name);
        if (oldindic != null) {
             logger.debug(name + " indicator is already created");
             if (oldindic.equals(newindic)) {
                // If same indicator already created, just return.
                 return name;
             }
        }

        if (check) {
            // Check that the type is supported
            JasmineCollectorService jcs = collectorservices.get(type);
            if (jcs == null) {
                throw new TypeNotSupportedException("Indicator type not supported: " + type);
            }

            JasmineCollectorInfo info = jcs.getCollectorInfo();

            // Check that mandatory properties are supplied
            String missinglist = "";
            for (JasminePropertyInfo jpi : info.getPropertyInfos()) {
                if (jpi.isRequired()) {
                    boolean found = false;
                    for (String key : newindic.getProperties().keySet()) {
                        if (key.equals(jpi.getName())) {
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        missinglist += " " + jpi.getName();
                    }
                }
            }
            if (missinglist.length() > 0) {
                throw new PropertyMissingException("Missing mandatory properties:" + missinglist);
            }

            // Check that a correct number of indicator sources are supplied
            if (newindic.getSources().size() < info.getSourceMin()) {
                throw new SourceNumberException("Missing source indicator. Minimum number is " + info.getSourceMin());
            }
            if (newindic.getSources().size() > info.getSourceMax() && info.getSourceMax() < 3) {
                throw new SourceNumberException("Too manysource indicators. Maximum number is " + info.getSourceMax());
            }
        }

        if (oldindic != null) {
            // New definition replaces old definition
            // -if no running probes are using this indicator
            logger.warn("old definition: " + oldindic);
            logger.warn("new definition: " + newindic);
            if (isDefUsed(oldindic, true)) {
                throw new JasmineProbeException(name + " indicator cannot be re-created as a running probe is using it");
            }

            // find probes using this indicator ;
            // even if not running, some probes may have reference to a collector corresponding to this indicator
            List<JasmineProbe> probesUsingOld = probesUsingDef(oldindic, false);
            for (JasmineProbe usingOld : probesUsingOld) {
                JProbe running = probes.get(usingOld.getId());
                if (running != null) {
                    // reset collectors - the collectors are recreated at the next resume
                    running.resetIndicators();
                }
            }

            // remove old definition and replace by the new one.
            indicators.remove(name);
        }

        indicators.put(name, newindic);
        return name;
    }

    /**
     * Generate an unique identifier
     */
    private synchronized String getNewProbeId() {
        String ret = "probe-0";
        boolean found = false;
        while (! found) {
            ret = "probe-" + ++probeCount;
            found = true;
            for (JasmineProbe p : probedefs.values()) {
                if (p.getId().equals(ret)) {
                    found = false;
                    break;
                }
            }
        }
        return ret;
    }

    /**
     * Read configuration.
     */
    private void readConfig() throws Exception {
        // Retrieve the probe config file
        String configurationFile = "probe-config.xml";
        ClassLoader loader = ClassLoader.getSystemClassLoader();
        InputStream resource = null;
        try {
            resource = loader.getResourceAsStream(configurationFile);
        } catch (Exception e) {
            logger.error("Cannot find probe-config.xml: " + e);
            throw e;
        }
        if (resource == null) {
            logger.error("Cannot find probe-config.xml");
            return;
        }
        logger.debug("Parsing probe-config.xml");
        loadConfigFile(resource);
    }

    private void saveConfigFile(OutputStream resource) throws Exception {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

            // rebuild probeConfig from current configuration
            probeConfig = new ProbeConfig();

            // outputs
            ArrayList<OutputBaseType> olist = new ArrayList<OutputBaseType>();
            for (JasmineOutput jo : outputs.values()) {
                OutputBaseType output = new OutputBaseType();
                output.setName(jo.getName());
                output.setType(jo.getType());
                for (String key : jo.getProperties().keySet()) {
                    PropType pv =  new PropType();
                    pv.setKey(key);
                    pv.setValue(jo.getProperties().get(key));
                    output.getProperty().add(pv);
                }
                output.setDefault(false);
                for (String on : defaultOutputList) {
                    if (on.equals(jo.getName())) {
                        output.setDefault(true);
                    }
                }
                olist.add(output);
            }
            probeConfig.setOutput(olist);

            // targets
            ArrayList<TargetBaseType> tlist = new ArrayList<TargetBaseType>();
            for (JasmineTarget jt : targets.values()) {
                TargetBaseType target = new TargetBaseType();
                target.setName(jt.getName());
                target.setType(jt.getType());
                for (String key : jt.getProperties().keySet()) {
                    PropType pv =  new PropType();
                    pv.setKey(key);
                    pv.setValue(jt.getProperties().get(key));
                    target.getProperty().add(pv);
                }
                tlist.add(target);
            }
            probeConfig.setTarget(tlist);

            // indicators
            ArrayList<IndicatorBaseType> ilist = new ArrayList<IndicatorBaseType>();
            for (JasmineIndicator ji : indicators.values()) {
                IndicatorBaseType indic = new IndicatorBaseType();
                indic.setName(ji.getName());
                indic.setType(ji.getType());
                //String scale = new Long(ji.getScale()).toString();
                indic.setScale(new Integer(ji.getScale()));
                indic.setSource(ji.getSources());
                for (String key : ji.getProperties().keySet()) {
                    PropType pv =  new PropType();
                    pv.setKey(key);
                    pv.setValue(ji.getProperties().get(key));
                    indic.getProperty().add(pv);
                }
                ilist.add(indic);
            }
            probeConfig.setIndicator(ilist);

            // probedefs
            ArrayList<ProbeType> plist = new ArrayList<ProbeType>();
            for (JasmineProbe jp : probedefs.values()) {
                ProbeType pdata = new ProbeType();
                pdata.setId(jp.getId());
                //String period = new Long(jp.getPeriod()).toString();
                pdata.setPeriod(new Integer(jp.getPeriod()));
                ArrayList<String> onlist = new ArrayList<String>();
                for (String output : jp.getOutputList()) {
                    onlist.add(output);
                }
                pdata.setOutput(onlist);
                ArrayList<String> talist = new ArrayList<String>();
                for (String target : jp.getTargetList()) {
                    talist.add(target);
                }
                pdata.setTarget(talist);
                ArrayList<String> inlist = new ArrayList<String>();
                for (String ind : jp.getIndicatorList()) {
                    inlist.add(ind);
                }
                pdata.setIndicator(inlist);
                switch (jp.getStatus()) {
                case JasmineProbe.PROBE_FAILED:
                case JasmineProbe.PROBE_STOPPED:
                    pdata.setStatus(StatusType.STOPPED);
                    break;
                case JasmineProbe.PROBE_RUNNING:
                case JasmineProbe.PROBE_STARTED:
                    //pdata.setStatus(StatusType.RUNNING);
                    pdata.setStatus(StatusType.STARTED);
                    break;
                }
                plist.add(pdata);
            }
            probeConfig.setProbe(plist);

            // Build the xml file
            marshaller.marshal(probeConfig, resource);
        } catch (Exception e) {
            logger.error("Error while writing probe-config.xml: " + e);
            throw new JasmineProbeException("Error while writing probe-config.xml: " + e);
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    private synchronized void loadConfigFile(InputStream resource) throws JasmineProbeException {
        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ObjectFactory.class);
            Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
            probeConfig = (ProbeConfig) unmarshaller.unmarshal(resource);

            // Create JasmineOutput objects
            for (OutputBaseType output : probeConfig.getOutput()) {
                JasmineOutput jo = new JasmineOutput();
                jo.setName(output.getName());
                jo.setType(output.getType());
                if (output.getDefault() == true) {
                    defaultOutputList.add(output.getName());
                }
                Map<String, String> props = new HashMap<String, String>();
                for (PropType param : output.getProperty()) {
                    props.put(param.getKey(), param.getValue());
                }
                jo.setProperties(props);
                createOutputInternal(jo, false);
            }

            // Create JasmineTarget objects
            for (TargetBaseType target : probeConfig.getTarget()) {
                JasmineTarget jt = new JasmineTarget();
                jt.setName(target.getName());
                Map<String, String> props = new HashMap<String, String>();
                for (PropType param : target.getProperty()) {
                    props.put(param.getKey(), param.getValue());
                }
                jt.setProperties(props);
                createTarget(jt);
            }

            // Create JasmineIndicator objects
            for (IndicatorBaseType indic : probeConfig.getIndicator()) {
                JasmineIndicator ji = new JasmineIndicator();
                ji.setName(indic.getName());
                ji.setType(indic.getType());
                ji.setScale(indic.getScale());
                ji.setSources(indic.getSource());
                Map<String, String> props = new HashMap<String, String>();
                for (PropType param : indic.getProperty()) {
                    props.put(param.getKey(), param.getValue());
                }
                ji.setProperties(props);
                createIndicatorInternal(ji, false);
            }

            // Create JasmineProbe objects
            for (ProbeType probe : probeConfig.getProbe()) {
                JasmineProbe jp = new JasmineProbe();
                jp.setId(probe.getId());
                jp.setPeriod(probe.getPeriod());
                jp.setOutputList(probe.getOutput());
                jp.setIndicatorList(probe.getIndicator());
                jp.setTargetList(probe.getTarget());
                if (probe.getStatus() != null) {
                    StatusType status = probe.getStatus();
                    if (status.equals(StatusType.STARTED)) {
                        jp.setStatus(JasmineProbe.PROBE_TOSTART);
                    }
                }
                createProbe(jp);
            }

        } catch (Exception e) {
            logger.error("Error in probe-config.xml: " + e);
            throw new JasmineProbeException("Error in probe-config.xml: " + e);
        } finally {
            try {
                resource.close();
            } catch (IOException ignore) {
            }
        }
    }

    /**
     * Test if an artifact definition is used at least in one probe instance.
     *
     * @param def the definition to check
     * @param checkrun  If true, check only if used by running probes
     * @return true if def is used in at least 1 probe.
     */
    private synchronized boolean isDefUsed(final Object def, boolean checkrun) {
        for (JasmineProbe probe : probedefs.values()) {
            boolean used = probeUsingDef(probe, def);
            if (!used) {
                continue;
            }
            // the name corresponds to a used artifact
            if (!checkrun
                    || (probe.getStatus() == JasmineProbe.PROBE_RUNNING)
                    || (probe.getStatus() == JasmineProbe.PROBE_STARTED)
                    || (probe.getStatus() == JasmineProbe.PROBE_TOSTART)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Return the list of JasmineProbes corresponding to probes that use a given JASMINe
     * Probe artifact (JasmineIndicator, JasmineOutput or JasmineTarget).
     * @param def a JASMINe Probe artifact
     * @param checkrun if true, only running probes are considered
     * @return null is the def object is not a JASMINe Probe artifact
     */
    private synchronized List<JasmineProbe> probesUsingDef(final Object def, boolean checkrun) {
        List<JasmineProbe> result = new ArrayList<JasmineProbe>();
        for (JasmineProbe probe : probedefs.values()) {
            boolean used = probeUsingDef(probe, def);
            if (!used) {
                continue;
            }
            if (!checkrun
                    || (probe.getStatus() == JasmineProbe.PROBE_RUNNING)
                    || (probe.getStatus() == JasmineProbe.PROBE_STARTED)
                    || (probe.getStatus() == JasmineProbe.PROBE_TOSTART)) {
                result.add(probe);
            }
        }
        return result;
    }

    /**
     * @param probe the probe to check
     * @param artifact the given artifact
     * @return true if the probe's definition contains (directly, or indirectly in recursive
     * definitions) the given artifact's name
     */
    private boolean probeUsingDef(final JasmineProbe probe, final Object artifact) {
        // artifact name
        String name = null;
        boolean targetArtifact = false;
        boolean indicArtifact = false;
        if (artifact instanceof JasmineIndicator) {
            name = ((JasmineIndicator) artifact).getName();
            indicArtifact = true;
            if (probe.getIndicatorList().contains(name)) {
                return true;
            }
        } else if (artifact instanceof JasmineOutput) {
            name = ((JasmineOutput) artifact).getName();
            if (probe.getOutputList().contains(name)) {
                return true;
            }
        } else if (artifact instanceof JasmineTarget) {
            name = ((JasmineTarget) artifact).getName();
            targetArtifact = true;
            if (probe.getTargetList().contains(name)) {
                return true;
            }
        } else {
            return false;
        }
        // Check for indirect use
        boolean used = false;
        if (indicArtifact || targetArtifact) {
            List<String> indicNames = getRecursiveIndicatorList(probe);
            if (indicArtifact && indicNames.contains(name)) {
                used = true;
            } else if (targetArtifact) {
                // Check for an indicator using the given target
                for (String indicName : indicNames) {
                    JasmineIndicator indic = indicators.get(indicName);
                    JasmineCollectorService jcs = collectorservices.get(indic.getType());
                    List<String> usedTargets = jcs.getDependantTargets(indic);
                    if (!usedTargets.isEmpty() && usedTargets.contains(name)) {
                        used = true;
                        break;
                    }
                }
            }
        }
        return used;
    }

    /**
     * Notify a state change for that probe
     *
     * @param running probe
     * @param state   See JasmineProbe.state
     * @param message Error message
     */
    public synchronized void setProbeState(JProbe running, int state, String message) {
        logger.debug("state =" + state);
        running.setStatus(state);
        running.setError(message);
        for (JasmineProbeListener listener : probeListeners) {
            listener.notifyEvent(running.getProbeDef());
        }
    }

    /**
     * Start probes waiting to be started.
     */
    private synchronized void startWaitingProbes() {

        for (JasmineProbe probe : probedefs.values()) {

            if (probe.getStatus() == JasmineProbe.PROBE_TOSTART
                    || probe.getStatus() == JasmineProbe.PROBE_FAILED) {
                logger.debug(probe.getId() + " trying to start");
                JProbe running = new JProbe(probe, this);
                try {
                    running.resume();
                } catch (JasmineProbeException e) {
                    logger.warn("Cannot start probe now: " + probe.getId());
                    continue;
                }
                probes.put(probe.getId(), running);
            }
        }
    }

    private synchronized List<String> getRecursiveIndicatorList(JasmineProbe probe) {
        List<String> ret = new ArrayList<String>();
        for (String iname : probe.getIndicatorList()) {
            ret.add(iname);
            JasmineIndicator indic = indicators.get(iname);
            if (indic != null) {
                JasmineCollectorService jcs = collectorservices.get(indic.getType());
                if (jcs != null) {
                    List<String> full = jcs.getDependantIndicators(indic);
                    ret.addAll(full);
                }
            }
        }
        return ret;
    }

}
