/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */

package org.ow2.jasmine.probe.outers;

import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.ow2.jasmine.probe.JasmineIndicatorValue;
import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbeResult;
import org.ow2.jasmine.probe.JasmineSingleResult;
import org.ow2.jasmine.probe.outer.JasmineOuter;
import org.ow2.jasmine.probe.util.MetaData;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Outer instance.
 * this abstract class should be derived in different forms,
 * depending of the type of output:
 * console, file, mule, etc...
 * These implementations are in separate modules.
 * @author durieuxp
 */
public abstract class JOuter extends Thread implements JasmineOuter {

    /**
     * logger used for outers
     */
    protected Log logger = LogFactory.getLog(JOuter.class);

    /**
     * Queue of JasmineProbeResult elements to be sent
     */
    private Queue<JasmineProbeResult> dataqueue = new ConcurrentLinkedQueue<JasmineProbeResult>();

    /**
     * CSV separator.
     */
    protected static final String SEPARATOR = ";";

    protected JasmineOutput output;

    private boolean valid = false;

    // Monitoring issues for debuging the queue capacity
    // ------------------
    // monitor consumed data (data retrieved from the queue by the outer's thread)
    private int totalRetrievedData = 0;
    private int nbConsumerCycle = 100;
    // monitor produced data (data inserted in the queue by the probes' threads)
    private int totalInsertedData = 0;
    private int nbProducerCycle = 100;

    /**
     * Std format for the date.
     */
    protected SimpleDateFormat dateformat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    abstract public void processData(JasmineProbeResult result);

    public JOuter(JasmineOutput output) {
        super("JasmineOuter-" + output.getName());
        valid = true;
        this.output = output;
    }

    public JasmineOutput getOutput() {
        return output;
    }

    /**
     * Marker allowing to a probe that uses this collector
     * to get rid of it.
     */
    private boolean removed = false;

    /**
     * run method for Thread implementation.
     */
    public void run() {
        while (valid) {
            JasmineProbeResult data = null;
            data = dataqueue.poll();
            if (data != null) {
                if (logger.isDebugEnabled()) {
                    /* DEBUG queue capacity */
                    nbConsumerCycle--;
                    totalRetrievedData++;
                    if (nbConsumerCycle <= 0) {
                        nbConsumerCycle = 100;
                        logger.debug("Monitoring {0} output : Total number of retrieved data = {1}, Data queue size = {2}", output.getName(), totalRetrievedData, dataqueue.size());
                    }
                }
                processData(data);
            }
        }
    }

    /**
     * stop the AbstractOuter
     */
    public synchronized void stopIt() {
        valid = false;
        notify();
    }

    /**
     * Flush the Outer.
     * Default implementation: do nothing.
     */
    public void flushData() {
    }

    /**
     * Compose name of value
     * @param jiv
     * @param jsrName name of the JasmineSingleResult
     * @return Name of the value
     */
    public String composeName(JasmineIndicatorValue jiv, String jsrName) {
        StringBuffer name = new StringBuffer(jiv.getName());
        if (jiv.isMultiValue()) {
            if (jsrName != null && jsrName.length() != 0) {
                name.append(".");
                name.append(jsrName);
            }
        }
        return name.toString();
    }

    /**
     * Find the mbean property or generate one if do not exist.
     * This is for compatibility. Will be removed in 2.0.
     * @param jsr
     * @return a mbean valid objectname
     */
    public String findMbean(JasmineSingleResult jsr) {
        String mbean = null;
        Map<String, String> resultProps = jsr.getProperties();
        if (!resultProps.isEmpty()) {
            for (Iterator<String> propKeys = resultProps.keySet().iterator(); propKeys.hasNext(); ) {
                String key = propKeys.next();
                if (MetaData.MBEAN.equals(key)) {
                    mbean = resultProps.get(key);
                }
            }
        }
        if (mbean == null) {
            // TODO Better to set the indicator type here
            mbean = MetaData.MISC_MBEAN;
        }
        return mbean;
    }

    // ------------------------------------------------------------------------------
    // JasmineOuter implementation
    // ------------------------------------------------------------------------------

    /**
     * Return the name of the indicator corresponding to this Collector
     */
    public String getOutputName() {
        return getOutput().getName();
    }

    /**
     * Publish data on this Output
     * @param result data to publish
     */
    public synchronized void publishData(JasmineProbeResult result) {
        dataqueue.offer(result);
        if (logger.isDebugEnabled()) {
            /* DEBUG queue capacity */
            nbProducerCycle--;
            totalInsertedData++;
            if (nbProducerCycle <= 0) {
                nbProducerCycle = 100;
                logger.debug("Monitoring {0} output : Total number of inserted data = {1}, Data queue size = {2}", output.getName(), totalInsertedData, dataqueue.size());
            }
        }
    }

    /**
     * Construct a String containing properties to be published.
     * @param output contains properties that are already treated
     * @param props properties to treat
     * @param separator used to delimit
     * @return String containing properties to be published.
     */
    protected String outputProps(String output, HashMap<String, String> props, String separator) {
        String result = output;
        for (String key : props.keySet()) {
            // Don't start with separator
            if (result.length() > 0) {
                result += separator;
            }
            result += key + "=" + props.get(key);
        }
        return result;
    }

    /**
     * Mark outer as removed.
     * Called when a JasmineOuterService stops.
     */
    public void remove() {
        logger.debug("Mark removed " + getName());
        removed = true;
    }

    /**
     * @return true if collector marked as removed.
     */
    public boolean isRemoved() {
        return removed;
    }

}
