/**
 * JASMINe
 * Copyright (C) 2011-2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineIndicator;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;

/**
 * Implements indicator-change command.
 * @author danesa
 */
@Component(name="IndicatorChangeCommand")
@Provides
public class IndicatorChangeCommand extends JasmineProbeCommand implements Command {

    /**
     * command name
     */
    private String name = "indicator-change";

    /**
     * necessary to be able to print out usage even when wrong options are provided by the user
     */
    //private String requiredOptions = "-n indicatorName -t indicatorType [-d propKey=propValue]* [-u propKey=propValue]*";
    private String requiredOptions = "-n indicatorName -t indicatorType ";

    /**
     * command description
     */
    private String description = "Change an indicator definition";


    // ----------------------------------------------------------------------------
    // ipojo management
    // ----------------------------------------------------------------------------

    @Requires
    JasmineProbeManager probeManager = null;

    // ----------------------------------------------------------------------------
    // constructors
    // ----------------------------------------------------------------------------

    /**
     * Default Constructor
     */
    public IndicatorChangeCommand() {
        super();
        setParams(new IndicatorChangeCommandParams());
    }

    // ----------------------------------------------------------------------------
    // org.apache.felix.shell.Command implementation
    // ----------------------------------------------------------------------------

    /**
     * Returns the name of the command that is implemented by the interface.
     * The command name should not contain whitespace and should also be unique.
     * @return the name of the command.
     */
    public String getName() {
        return name;
    }

    /**
     * Returns a short description of the command; this description should be as short as possible.
     * This information is used when generating the command help information.
     * @return a short description of the command.
     */
    public String getShortDescription() {
        return description;
    }

    public String getRequiredOptions() {
        return requiredOptions;
    }

    public JasmineProbeManager getProbeManager() {
        return probeManager;
    }

    /**
     * Executes the command using the supplied command line, output print stream, and error print stream.
     * @param line the complete command line, including the command name.
     * @param out  the print stream to use for standard output.
     * @param err  the print stream to use for standard error.
     */
    public void execute(String line, PrintStream out, PrintStream err) {
        boolean validOptions = parseOptions(line, out, err);
        if (!validOptions) {
            err.println(getUsage());
            return;
        }

        IndicatorChangeCommandParams params = ((IndicatorChangeCommandParams) getParams());

        // get command arguments
        String indicatorName = params.getName();
        String indicatorType = params.getType();
        int indicatorScale = params.getScale();
        List<String> indicatorSources = params.getSources();
        List<String> defIndicatorParams = params.getProps();

        // indicator to change
        JasmineIndicator indicator = new JasmineIndicator();
        indicator.setName(indicatorName);
        Map<String, String> properties = new HashMap<String, String>();
        if (indicatorType != null) {
            indicator.setType(indicatorType);
        }
        if (indicator.getScale() != indicatorScale) {
            indicator.setScale(indicatorScale);
        }
        if (indicatorSources != null && !indicatorSources.isEmpty()) {
            indicator.setSources(new ArrayList<String>(indicatorSources));
        }
        for (String indicatorParam : defIndicatorParams) {
            if (validParam(indicatorParam)) {
                int indexEqual = indicatorParam.indexOf("=");
                String indicatorParamName = indicatorParam.substring(0, indexEqual);
                String indicatorParamValue = indicatorParam.substring(indexEqual + 1, indicatorParam.length());
                properties.put(indicatorParamName, indicatorParamValue);
            } else {
                err.println(name + " " + indicatorName + " failed");
                // wrong parameter format
                err.println("Wrong properties format for " + indicatorParam);
                err.println("Expected format is paramName=paramValue");
                err.println(getUsage());
                params.reset();
                return;
            }
        }
        indicator.setProperties(properties);

        params.reset();

        try {
            getProbeManager().changeIndicator(indicator);
            out.println(name + " " + indicatorName +  " successfull");
        } catch (JasmineProbeException e) {
            err.println(name + " " + indicatorName + " failed");
            err.println(e);
            return;
        }

    }

}
