/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.List;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineCollectorInfo;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.JasminePropertyInfo;

/**
 * Implements indicator-info command.
 * @author danesa
 */
@Component(name="IndicatorInfoCommand")
@Provides
public class IndicatorInfoCommand extends JasmineProbeCommand implements Command {

    /**
     * command name
     */
    private String name = "indicator-info";


    /**
     * necessary to be able to print out usage even when wrong options are provided by the user
     */
    private String requiredOptions = "-t indicatorType";

    /**
     * command description
     */
    private String description = "Print information about indicators of a given type";

    // ----------------------------------------------------------------------------
    // ipojo management
    // ----------------------------------------------------------------------------

    @Requires
    JasmineProbeManager probeManager = null;

    // ----------------------------------------------------------------------------
    // constructors
    // ----------------------------------------------------------------------------

    /**
     * Default Constructor
     */
    public IndicatorInfoCommand() {
        super();
        setParams(new IndicatorPropertiesCommandParams());
    }
    @Override
    public String getName() {
        return name;
    }

    @Override
    public JasmineProbeManager getProbeManager() {
        return probeManager;
    }

    public String getShortDescription() {
        return description;
    }

    public String getRequiredOptions() {
        return requiredOptions;
    }

    /**
     * Executes the command using the supplied command line, output print stream, and error print stream.
     * @param line the complete command line, including the command name.
     * @param out  the print stream to use for standard output.
     * @param err  the print stream to use for standard error.
     */
    public void execute(String line, PrintStream out, PrintStream err) {
        boolean validOptions = parseOptions(line, out, err);
        if (!validOptions) {
            err.println(getUsage());
            return;
        }

        IndicatorPropertiesCommandParams params = ((IndicatorPropertiesCommandParams) getParams());

        // get command arguments
        String indicatorType = params.getType();

        params.reset();

        JasmineCollectorInfo info = getProbeManager().getIndicatorTypeInfo(indicatorType);
        List<JasminePropertyInfo> properties = info.getPropertyInfos();
        if (properties == null) {
            out.println("Invalid type " + indicatorType);
            return;
        }
        out.println(info.getShortDesc());
        out.println(info.getLongDesc());
        // Print info about sources only if sources might exist
        if (info.getSourceMax() > 0) {
            out.println(info.getSourceDesc());
        }
        if (! properties.isEmpty()) {
            out.println("The properties flagged with * are mandatory:");
            for (JasminePropertyInfo prop : properties) {
                out.println(prop.toString());
            }
        }
    }



}
