/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineProbe;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

/**
 * Implements probe-change command that creates a probe.
 * @author danesa
 */
@Component(name="ProbeChangeCommand")
@Provides
public class ProbeChangeCommand extends JasmineProbeCommand implements Command  {

    /**
     * command name
     */
    private String name = "probe-change";

    /**
     * necessary to be able to print out usage even when wrong options are provided by the user
     */
    private String requiredOptions = "-n probeName -i indicator -o output -p 10";

    /**
     * command description
     */
    private String description = "change a JASMINe probe";

    // ----------------------------------------------------------------------------
    // ipojo management
    // ----------------------------------------------------------------------------

    @Requires
    JasmineProbeManager probeManager = null;

    // ----------------------------------------------------------------------------
    // constructors
    // ----------------------------------------------------------------------------

    /**
     * Default Constructor
     */
    public ProbeChangeCommand() {
        super();
        setParams(new ProbeChangeCommandParams());
    }


    // ----------------------------------------------------------------------------
    // org.apache.felix.shell.Command implementation
    // ----------------------------------------------------------------------------

    /**
     * Returns the name of the command that is implemented by the interface.
     * The command name should not contain whitespace and should also be unique.
     * @return the name of the command.
     */
    public String getName() {
        return name;
    }

    /**
     * Returns a short description of the command; this description should be as short as possible.
     * This information is used when generating the command help information.
     * @return a short description of the command.
     */
    public String getShortDescription() {
        return description;
    }

    public String getRequiredOptions() {
        return requiredOptions;
    }

    public JasmineProbeManager getProbeManager() {
        return probeManager;
    }

    /**
     * Executes the command using the supplied command line, output print stream, and error print stream.
     * @param line the complete command line, including the command name.
     * @param out  the print stream to use for standard output.
     * @param err  the print stream to use for standard error.
     */
    public void execute(String line, PrintStream out, PrintStream err) {
        boolean validOptions = parseOptions(line, out, err);
        if (!validOptions) {
            err.println(getUsage());
            return;
        }

        ProbeChangeCommandParams params = ((ProbeChangeCommandParams) getParams());

        // get command arguments for probe creation
        String probeName = params.getName();

        JasmineProbe oldProbe = null;
        try {
            oldProbe = probeManager.getProbe(probeName);
        } catch (JasmineProbeException e1) {
            err.println("There is no JasmineProbe named " + probeName);
            params.reset();
            return;
        }

        List<String> indicatorList = null;
        if (!params.getIndicators().isEmpty()) {
            indicatorList = new ArrayList<String>(params.getIndicators());
        } else {
            indicatorList = oldProbe.getIndicatorList();
        }

        List<String> outputList = null;
        if (!params.getOutputs().isEmpty()) {
            outputList = new ArrayList<String>(params.getOutputs());
        } else {
            outputList = oldProbe.getOutputList();
        }

        List<String> targetList = null;
        if (!params.getTargets().isEmpty()) {
            targetList = new ArrayList<String>(params.getTargets());
        } else {
            targetList = oldProbe.getTargetList();
        }

        int period;
        if (params.getPeriod() != -1) {
            period = params.getPeriod();
        } else {
            period = oldProbe.getPeriod();
        }

        // create new JasmineProbe
        JasmineProbe probe = new JasmineProbe();
        probe.setId(probeName);
        probe.setPeriod(period);
        probe.setIndicatorList(indicatorList);
        probe.setOutputList(outputList);
        probe.setTargetList(targetList);

        try {
            out.println(probe.toString());
            probeManager.changeProbe(probe);
        } catch (JasmineProbeException e) {
            err.println(name + " failed");
            err.println(e);
            return;
        }
        out.println(name + " " + probeName +  " successfull");

        // reset params
        params.reset();
    }
}
