/**
 * JASMINe
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.shell;

import java.io.PrintStream;
import java.util.Collection;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.shell.Command;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.JasmineTarget;

/**
 * Implements target-list command that list the name of all the created targets.
 * @author danesa
 */
@Component(name="TargetListCommand")
@Provides
public class TargetListCommand extends JasmineProbeCommand implements Command {

        /**
         * command name
         */
        private String name = "target-list";

        /**
         * command description
         */
        private String description = "List all JASMINe Targets, or show a particular Target";


        // ----------------------------------------------------------------------------
        // ipojo management
        // ----------------------------------------------------------------------------

        @Requires
        JasmineProbeManager probeManager = null;

        // ----------------------------------------------------------------------------
        // constructors
        // ----------------------------------------------------------------------------

        /**
         * Default Constructor
         */
        public TargetListCommand() {
            super();
            setParams( new TargetListCommandParams());
        }

        // ----------------------------------------------------------------------------
        // org.apache.felix.shell.Command implementation
        // ----------------------------------------------------------------------------

        /**
         * Returns the name of the command that is implemented by the interface.
         * The command name should not contain whitespace and should also be unique.
         * @return the name of the command.
         */
        public String getName() {
            return name;
        }

        /**
         * Returns a short description of the command; this description should be as short as possible.
         * This information is used when generating the command help information.
         * @return a short description of the command.
         */
        public String getShortDescription() {
            return description;
        }

        public JasmineProbeManager getProbeManager() {
            return probeManager;
        }

        /**
         * Executes the command using the supplied command line, output print stream, and error print stream.
         * @param line the complete command line, including the command name.
         * @param out  the print stream to use for standard output.
         * @param err  the print stream to use for standard error.
         */
        public void execute(String line, PrintStream out, PrintStream err) {
            boolean validOptions = parseOptions(line, out, err);
            if (!validOptions) {
                err.println(getUsage());
                return;
            }

            TargetListCommandParams params = ((TargetListCommandParams) getParams());

            // get command arguments
            String targetName = params.getName();
            boolean verbose = params.getVerbose();
            if (targetName == null) {
                listTargets(verbose, out);
            } else {
                showTarget(targetName, out, err);
            }

            params.reset();
        }

        private void listTargets(boolean verbose, PrintStream out) {
            Collection<JasmineTarget> targets = probeManager.getTargets();
            if (targets.isEmpty()) {
                out.println("There are no JasmineTargets");
                return;
            }
            for (JasmineTarget target : targets) {
                if (verbose) {
                    out.println(target.toLine());
                } else {
                    out.println(target.getName());
                }
            }
        }

        private void showTarget(String targetName, PrintStream out, PrintStream err) {
            Collection<JasmineTarget> targets = probeManager.getTargets();
            for (JasmineTarget target : targets) {
                if (targetName.equals(target.getName())) {
                    out.println(target.toString());
                    return;
                }
            }
            err.println("There is no JasmineTarget named " + targetName);
        }
}
