/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.rest;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import org.ow2.jasmine.probe.api.generated.ProbeConfType;
import org.ow2.jasmine.probe.api.generated.ProbeType;

/**
 * Root resource class for probes management.
 * Contains methods to treat HTTP requests on "probe" resources.
 * @author danesa
 *
 */
@Path("/probe")
public interface Probes {


    public final String RESOURCE_NAME = "indicator";

    public final String START_ACTION = "start";
    public final String STOP_ACTION = "stop";

    /**
     * Process HTTP GET requests having /probe URI
     * Corresponds to probe-list command
     * @return response containing the existent probe ids.
     */
    @GET
    @Produces(MediaType.APPLICATION_XML)
    Response getProbes(@Context UriInfo ui);

    /**
     * Process HTTP GET requests having /probe/{id} URI
     * Corresponds to probe-list -n id
     * @param probeId
     * @return response
     */
    @GET
    @Path("/{id}")
    @Produces(MediaType.APPLICATION_XML)
    Response getProbe(@PathParam("id") String probeId, @Context UriInfo ui);

    /**
     * Process HTTP DELETE requests having /probe/{id} URI
     * Corresponds to probe-remove id
     * @param probeId
     */
    @DELETE
    @Path("/{id}")
    @Produces(MediaType.APPLICATION_XML)
    Response deleteProbe(@PathParam("id") String probeId);

    /**
     * Process HTTP POST requests having /probe URI and new probe definition in its body.
     * The probeDef may contain the new probe's id. Otherwise, the the id is generated by
     * JasmineProbe.
     * Corresponds to probe-create command.
     * @param probeDef new probe's definition
     * @return status may be CRETED if creation successful (in this case the response contains the new probe's URI
     */
    @POST
    @Consumes(MediaType.APPLICATION_XML)
    @Produces(MediaType.APPLICATION_XML)
    Response createProbe(ProbeType probeDef);

    /**
     * Process HTTP PUT requests having /probe/{id} URI and updated probe configuration in its body.
     * @param probeConf the new configuration for the probe
     * @param probeId the id of the probe to update
     * @return response with state OK or NOT_FOUND or BAD_REQUEST if change failed
     */
    @PUT
    @Path("/{id}")
    @Consumes(MediaType.APPLICATION_XML)
    @Produces(MediaType.APPLICATION_XML)
    Response changeProbe(ProbeConfType probeConf, @PathParam("id") String probeId);

    /**
     * Process HTTP PUT requests having /probe/{id}/period allowing to change a probe's period
     * @param probeId probe to change
     * @param period updated period
     * @return
     */
    @PUT
    @Path("/{id}/period")
    @Produces(MediaType.APPLICATION_XML)
    Response changeProbePeriod(@PathParam("id") String probeId, @QueryParam("p") String period);

    /**
     * @param probeName
     * @param actionName
     */
    @POST
    @Path("/{id}/action/{name}/")
    @Produces(MediaType.APPLICATION_XML)
    Response setState(@PathParam("id") String probeId, @PathParam("name") String actionName);

    /**
     * Process HTTP POST requests having /probe/{id}/data URI
     * Creates a data task for this probe allowing to get collected results from this task.
     * @param probeId
     */
    @POST
    @Path("/{id}/data")
    @Produces(MediaType.APPLICATION_XML)
    Response getDataTask(@PathParam("id") String probeId, @Context UriInfo ui);

    /**
     * Process HTTP DELETE requests having /probe/{id}/data URI
     * Removes the data task created for the probe
     * @param probeId
     */
    @DELETE
    @Path("/{id}/data")
    @Produces(MediaType.APPLICATION_XML)
    Response deleteDataTask(@PathParam("id") String probeId);

}
