/**
 * JASMINe
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id$
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.probe.rest.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;
import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;

import org.ow2.jasmine.probe.JasmineOutput;
import org.ow2.jasmine.probe.JasmineProbeException;
import org.ow2.jasmine.probe.JasmineProbeManager;
import org.ow2.jasmine.probe.api.generated.ErrorType;
import org.ow2.jasmine.probe.api.generated.OutputBaseType;
import org.ow2.jasmine.probe.api.generated.OutputResource;
import org.ow2.jasmine.probe.api.generated.PropType;
import org.ow2.jasmine.probe.rest.Outputs;

public class JOutputs  implements Outputs {

    private JasmineProbeManager jpManager = null;

    public JOutputs(JasmineProbeManager jpManager) {
        this.jpManager = jpManager;
    }

    @Override
    public Response getOutput(String name, UriInfo ui) {

        UriBuilder ub = ui.getAbsolutePathBuilder();
        String outUri = ub.build("").toString();
        String baseUri = outUri.substring(0, outUri.lastIndexOf(name));

        JasmineOutput jout;
        try {
            jout = jpManager.getOutput(name);
        } catch (JasmineProbeException e) {
            ErrorType err = new ErrorType();
            err.setMessage(e.getMessage());
            QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "error");
            JAXBElement<ErrorType> errElem = new JAXBElement<ErrorType>(qname, ErrorType.class, err);
            return Response
            .status(Response.Status.NOT_FOUND)
            .entity(errElem)
            .type(MediaType.APPLICATION_XML)
            .build();
        }

        OutputResource output = new OutputResource();
        output.setName(name);
        output.setType(jout.getType());
        output.setHref(outUri);

        List<PropType> property = new ArrayList<PropType>();
        for (String propKey : jout.getProperties().keySet()) {
            String propValue = jout.getProperties().get(propKey);
            PropType prop = new PropType();
            prop.setValue(propValue);
            prop.setKey(propKey);
            property.add(prop);
        }
        output.setProperty(property);

        QName qname  = new QName("org.ow2.jasmine.probe:probe-config", "output");
        JAXBElement<OutputResource> outResource = new JAXBElement<OutputResource>(qname, OutputResource.class, output);

        return Response
                        .status(Response.Status.OK)
                        .entity(outResource)
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    @Override
    public Response getOutputs(String type,
            boolean verbose, UriInfo ui) {
        List<JAXBElement<OutputResource>> elementsList = new ArrayList<JAXBElement<OutputResource>>();

        List<JasmineOutput> outList = jpManager.getOutputs();
        for (JasmineOutput out : outList) {
            String outType = out.getType();
            if (type == null || type.equals(outType)) {
                String outName = out.getName();
                OutputResource outElement = new OutputResource();
                outElement.setName(out.getName());

                URI outUri = ui.getAbsolutePathBuilder().path(outName).build();
                outElement.setHref(outUri.toString());

                if (verbose) {
                    outElement.setType(outType);
                }

                QName name  = new QName("org.ow2.jasmine.probe:probe-config", "output");
                JAXBElement xmlElement =  new JAXBElement<OutputResource>(name, OutputResource.class, outElement);
                elementsList.add(xmlElement);
            }
        }

        return Response
                        .status(Response.Status.OK)
                        .entity(new GenericEntity<List<JAXBElement<OutputResource>>>(elementsList) {})
                        .type(MediaType.APPLICATION_XML_TYPE)
                        .build();
    }

    @Override
    public Response createOutput(OutputBaseType outputDef) {

        JasmineOutput output = new JasmineOutput();
        output.setName(outputDef.getName());
        output.setType(outputDef.getType());

        Map<String, String> properties = new HashMap<String, String>();

        for (PropType prop : outputDef.getProperty()) {
            properties.put(prop.getKey(), prop.getValue());
        }
        output.setProperties(properties);

        try {
            jpManager.createOutput(output);
            if (outputDef.getDefault()) {
                // this is a default output
                jpManager.addDefaultOutput(outputDef.getName());
            }
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
        }

        return Response.status(Response.Status.OK).build();
    }

    @Override
    public Response changeOutput(OutputBaseType outputDef, String name) {

        try {
            jpManager.getOutput(name);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }

        JasmineOutput output = new JasmineOutput();
        output.setName(name);
        output.setType(outputDef.getType());

        Map<String, String> properties = new HashMap<String, String>();

        for (PropType prop : outputDef.getProperty()) {
            properties.put(prop.getKey(), prop.getValue());
        }
        output.setProperties(properties);

        try {
            jpManager.changeOutput(output);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.BAD_REQUEST);
        }

        return Response.status(Response.Status.OK).build();
    }

    @Override
    public Response deleteOutput(String name) {
        try {
            jpManager.removeOutput(name);
        } catch (JasmineProbeException e) {
            return RestUtil.errorResponse(e, Response.Status.NOT_FOUND);
        }
        return Response.status(Response.Status.OK).build();
    }
}
