/**
 * JASMINe
 * Copyright (C) 2009 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ThreadDumpAction.java 5827 2009-12-28 11:18:06Z eyindanga $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.rules.threadump;

import java.util.regex.Pattern;

import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

public class ThreadDumpAction {

    /**
     * The singleton thread dump action.
     */
    private static ThreadDumpAction threaDumpAction;

    /**
     * Configuration of the thread dump.
     */
    private ThreadDumpConfig threadDumpConfig = null;

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(ThreadDumpAction.class);

    /**
     * Private constructor.
     */
    private ThreadDumpAction() {

    }
    /**
     * Gets the singleton thread dump action.
     * @return the singleton thread dump action
     */
    public static ThreadDumpAction getThreadDumpAction(){
        if(threaDumpAction == null) {
            threaDumpAction = new ThreadDumpAction();
        }
        return threaDumpAction;
    }
    /**
     * @param threadDumpConfig the threadDumpConfig to set
     */
    public void setThreadDumpConfig(ThreadDumpConfig threadDumpConfig) {
        this.threadDumpConfig = threadDumpConfig;
    }
    /**
     * @return the threadDumpConfig
     */
    public ThreadDumpConfig getThreadDumpConfig() {
        return threadDumpConfig;
    }
    /**
     * Executes the action.
     * @throws ThreadDumpException
     */
    public void validate(final String value, final String jxmUrl,
            final String domain, final String serverName) throws ThreadDumpException {
        if (this.getThreadDumpConfig().getJonasJxmUrl() == null) {
            this.getThreadDumpConfig().setJonasJxmUrl(jxmUrl);
        }
        double curValue =
            ThreaDumpUtil.convertCurrentValue(
                    threadDumpConfig.getMemoryThresholdUnit(), value);
        if (ThreaDumpUtil.checkit(curValue,
                threadDumpConfig.getMemoryThreshold())) {
            int repeat = threadDumpConfig.getRepeat();
            if (repeat == -1 || repeat > 0) {
                logger.debug( curValue + " >= " +
                        threadDumpConfig.getMemoryThreshold()
                        + " " + threadDumpConfig.getMemoryThresholdUnit()
                        + " .Generating threadump");
                generateThreadDump(this.getThreadDumpConfig().getJonasJxmUrl(),
                        domain, serverName);
            }else {
                logger.debug( curValue + " >= " +
                        threadDumpConfig.getMemoryThreshold()
                        + " " + threadDumpConfig.getMemoryThresholdUnit()
                        + " But repeat times exceeded.");
            }

        }
    }
    /**
     * Generates thread dump.
     */
    private void generateThreadDump(final String jmxUrl, final String domain,
            final String serverName)
    throws ThreadDumpException {
        try {
            if (threadDumpConfig.getTargetFilter() != null &&
                    Pattern.matches(
                            threadDumpConfig.getTargetFilter(), jmxUrl)) {

                JMXServiceURL  url =
                    new JMXServiceURL(jmxUrl);
                JMXConnector jmxc = JMXConnectorFactory.connect(url, null);
                MBeanServerConnection mbsc = jmxc.getMBeanServerConnection();
                ObjectName mbeanName =
                    new ObjectName(domain + ":j2eeType=J2EEServer,name="
                            + serverName);
                String opName = null;
                String[] params = {""};
                String[] signature = {""};
                if (ThreadDumpEnum.OUTPUT_TYPE_FILE_VALUE.getValue().
                        equalsIgnoreCase(
                                this.threadDumpConfig.getOutputType())){
                    //generate thread dump in a file
                    opName = "printThreadStackDump";
                    params[0] = this.threadDumpConfig.getOutputFilePath();
                    signature[0] = "java.lang.String";
                }else {
                    opName = "logThreadStackDump";
                    params = null;
                    signature = null;
                }
                mbsc.invoke(mbeanName, opName, params, signature);
                int repeat = this.threadDumpConfig.getRepeat();
                if (repeat != -1) {
                    repeat --;
                    this.threadDumpConfig.setRepeat(repeat);
                }
                /**
                 * Set current jmx url.
                 */
                threadDumpConfig.setJonasJxmUrl(jmxUrl);
            }

        } catch (Exception e) {
            throw new ThreadDumpException(e);
        }

    }

}
