/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DummyHost.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.dummy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.agent.main.VirtManagerAgent;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

/**
 * Dummy driver Host MXBean implementation
 */
class DummyHost extends ManagedResource implements HostMXBean {
    static Logger logger = Logger.getLogger(DummyHost.class);

    private List<DummyVirtualMachine> vms = new CopyOnWriteArrayList<DummyVirtualMachine>();

    private static ExecutorService executorService = Executors.newFixedThreadPool(3);

    private Timer perfTimer;

    private String hostName;

    private final ObjectName objectName;

    private int numCPU = 4;

    private int freeMemoryMB = 4 * 1024;

    private DummyServerPool pool;

    public DummyHost(final DummyServerPool pool, final String hostName, final ObjectName objectName) {
        super(objectName);
        this.pool = pool;
        this.hostName = hostName;
        this.objectName = objectName;
    }

    public ServerPoolMXBean getServerPool() {
        return pool;
    }

    private void addVM(final DummyVirtualMachine vm) {
        vms.add(vm);
    }

    private void removeVM(final DummyVirtualMachine vm) {
        vms.remove(vm);
    }

    void postMigrateVM(final DummyVirtualMachine vm, final DummyHost newHost) {
        removeVM(vm);
        newHost.addVM(vm);
        freeMemoryMB += vm.getMemorySizeMB();
        newHost.freeMemoryMB -= vm.getMemorySizeMB();
    }

    public void postDestroyVM(final DummyVirtualMachine vm) {
        removeVM(vm);
        freeMemoryMB += vm.getMemorySizeMB();
    }

    @Override
    public ObjectName getObjectName() {
        return objectName;
    }

    public synchronized void onDestroyVM(final DummyVirtualMachine vm) {
        try {
            AgentCommon.getMBeanServer().unregisterMBean(vm.getObjectName());
        } catch (Exception ex) {
            logger.error("Cannot unregister VM MBean", ex);
        }
        removeVM(vm);
        freeMemoryMB += vm.getMemorySizeMB();
        emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
        logger.info("destroyed VM " + vm.getObjectName());
    }

    private VirtualMachineMXBean doCreateVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        DummyVirtualMachine vm = null;

        // check name uniqueness
        for (DummyVirtualMachine dvm : vms) {
            if (dvm.getNameLabel().equals(vmSpec.getName())) {
                throw new InvalidVMConfigException("VM name already exists");
            }
        }

        try {
            Thread.sleep(500);
            Random random = new Random(System.currentTimeMillis());
            String uuid = Integer.toString(random.nextInt(10000));
            ObjectName vmName = MBeanObjectNamer.makeVirtualMachineName(getServerPool().getPath() + "/" + vmSpec.getName(),
                uuid);
            vm = new DummyVirtualMachine(vmSpec.getName(), uuid, vmName, this);
            addVM(vm);
            vm.setDiskSizeMB(vmSpec.getDiskSizeMB());
            vm.setNumVCPUs(vmSpec.getNumVCPU());
            vm.setMemorySizeMB((int) vmSpec.getMemorySizeMB());
            vm.addUserData("imageID", vmSpec.getVmImageUUID());
            AgentCommon.getMBeanServer().registerMBean(vm, vm.getObjectName());
            logger.info("VM Creation: registered VirtualMachineMBean " + vm.getObjectName());
        } catch (Exception ex) {
            logger.error("Failed to create VM", ex);
            return null;
        }
        emitNotification(NotificationTypes.VM_ADD, "Created", vm.getObjectName());

        freeMemoryMB -= vmSpec.getMemorySizeMB();
        return vm;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        final VMConfigSpec vmSpec2 = new VMConfigSpec(vmSpec);
        if (!sync) {
            executorService.execute(new Runnable() {
                public void run() {
                    try {
                        doCreateVM(vmSpec2, sync);
                    } catch (Exception ex) {
                    }
                };
            });
            return null;
        } else {
            return doCreateVM(vmSpec2, sync);
        }
    }

    HashMap<String, String> hypervisorInfo;

    public Map<String, String> getHypervisorInfo() {
        if (hypervisorInfo == null) {
            hypervisorInfo = new HashMap<String, String>();
            hypervisorInfo.put("name", "DummyHypervisor");
            hypervisorInfo.put("vendor", "");
            hypervisorInfo.put("version", "1.0");
        }
        return hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (cpuInfo == null) {
            cpuInfo = new HashMap<String, String>();
            cpuInfo.put("model", "Core 2 Duo");
            cpuInfo.put("speed", "2330");
            cpuInfo.put("vendor", "Intel");
        }
        return cpuInfo;
    }

    public int getNumCPU() {
        return numCPU;
    }

    public float[] getLoadPerCPU() {
        return new float[] {0, 0};
    }

    public float getCPULoad() {
        float l = vms.size() * 0.25f;
        return Math.min(1, l);
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
        return cpuLoads;
    }

    public long getFreeMemoryMB() {
        return freeMemoryMB;
    }

    public String getHostName() {
        return hostName;
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> vmList = new ArrayList<VirtualMachineMXBean>();
        for (VirtualMachineMXBean vm : vms) {
            vmList.add(vm);
        }
        return vmList;
    }

    public long getTotalMemoryMB() {
        // TODO Auto-generated method stub
        return 4096;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return pool.getVMImageStore();
    }

    public String getUUID() {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.LOG, NotificationTypes.ERROR,
            NotificationTypes.PERF_REPORT}, Notification.class.getName(), "Host event")};
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            currentMonitoringPeriod = periodMillis;
            currentMonitoredMetrics = metricsOfInterest;
            startPerfMonitor();
        }
    }

    private void startPerfMonitor() {
        if (perfTimer == null) {
            logger.info("Host " + hostName + ": starting periodic perf monitor task");
            perfTimer = new Timer();
            perfTimer.schedule(new PerfTimerTask(), 0, currentMonitoringPeriod);
        }
    }

    private void stopPerfMonitor() {
        if (perfTimer != null) {
            logger.info("Host " + hostName + ": stopping periodic perf monitor task");
            perfTimer.cancel();
        }
        perfTimer = null;
    }

    private class PerfTimerTask extends TimerTask {
        @Override
        public void run() {
            HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
            int i = 0;
            for (DummyVirtualMachine vm : vms) {
                cpuLoads.put(vm.getNameLabel(), 0.25f);
                i++;
            }
            emitNotification(NotificationTypes.PERF_REPORT, "CPU load", cpuLoads);
        }
    }
}
