/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareHost.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.vmware.vim.ArrayOfManagedObjectReference;
import com.vmware.vim.DynamicProperty;
import com.vmware.vim.HostCpuInfo;
import com.vmware.vim.HostCpuPackage;
import com.vmware.vim.HostHardwareInfo;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;
import com.vmware.vim.VirtualMachineCloneSpec;
import com.vmware.vim.VirtualMachineConfigSpec;
import com.vmware.vim.VirtualMachineRelocateSpec;

/**
 * VMware driver Host MXBean implementation
 */
class VMwareHost extends ManagedResource implements HostMXBean {
    static Logger logger = Logger.getLogger(VMwareHost.class);

    private VMwareServerPool pool;

    private VMwareServiceConnection connection;

    private String hostName;

    private int numCPU = -1;

    private ManagedObjectReference mor;

    private HashMap<String, VMwareVirtualMachine> vmRefs = new HashMap<String, VMwareVirtualMachine>();

    private VMwarePerfMonitor perfMon = null;

    private HashMap<String, BigInteger> hostMetricValues = null;

    private HashMap<String, HashMap<String, BigInteger>> metricValuesPerVM = null;

    private String[][] counterNames = null;

    private ManagedObjectReference myResourcePool;

    public VMwareHost(final VMwareServerPool pool, final ManagedObjectReference mor, final ObjectName objectName, final String hostName) {
        super(objectName);
        this.pool = pool;
        this.connection = pool.getConnection();
        this.mor = mor;
        this.objectName = objectName;
        this.hostName = hostName;
        // this.PerfMon = new VMWarePerfMonitor(vmUtil.cb, this);
        this.counterNames = new String[][] {new String[] {"cpu", "usage", "average"}, // host
            // & vm
            new String[] {"mem", "usage", "average"}, // host & vm
            new String[] {"mem", "active", "average"}, // host & vm
            new String[] {"mem", "granted", "average"}, // host & vm
            new String[] {"mem", "state", "latest"}, // host only
            new String[] {"disk", "usage", "average"}, // host (& vm
            // ???)
            new String[] {"net", "transmitted", "average"}, // host & vm
            new String[] {"net", "received", "average"}, // host & vm
            new String[] {"sys", "uptime", "latest"} // host & vm
        };
    }

    public ManagedObjectReference getMOR() {
        return mor;
    }

    public ServerPoolMXBean getServerPool() {
        return pool;
    }

    public ObjectName getObjectName() {
        return objectName;
    }

    // HostMBean Interface

    public String getHostName() {
        return hostName;
    }

    HashMap<String, String> hypervisorInfo = null;

    public Map<String, String> getHypervisorInfo() { // TODO
        if (hypervisorInfo == null) {
            hypervisorInfo = new HashMap<String, String>();
            hypervisorInfo.put("name", "ESX");
            hypervisorInfo.put("vendor", "VMware");
            hypervisorInfo.put("version", "3.0.2");
        }
        return hypervisorInfo;
    }

    public int getNumCPU() {
        if (numCPU == -1) {
            getCPUInfo();
        }
        return numCPU;
    }

    HashMap<String, String> cpuInfo = null;

    public Map<String, String> getCPUInfo() { // TODO
        if (cpuInfo == null) {
            cpuInfo = new HashMap<String, String>();
            try {
                DynamicProperty[] cpuArry = getDynamicProarray(this.mor, "hardware");
                HostHardwareInfo hinfo = (HostHardwareInfo) cpuArry[0].getVal();
                HostCpuInfo tinfo = hinfo.getCpuInfo(); // ????
                HostCpuPackage cinfo = hinfo.getCpuPkg(0);

                cpuInfo.put("speed", Long.toString(tinfo.getHz() / 1000000));
                numCPU = tinfo.getNumCpuCores();
                cpuInfo.put("vendor", cinfo.getVendor());
            } catch (Exception ex) {
                logger.debug("Host " + hostName, ex);
            }
        }
        return cpuInfo;
    }

    private VMwareVirtualMachine addVm(final ManagedObjectReference vmRef, final String vmName) {
        ObjectName vmMBeanObjectName = MBeanObjectNamer.makeVirtualMachineName(pool.getPath() + "/" + vmName,
            VMwareVirtualMachine.getUuid(connection, vmRef));
        VMwareVirtualMachine vm = new VMwareVirtualMachine(vmMBeanObjectName, pool, vmRef, this, vmName);
        try {
            AgentCommon.getMBeanServer().registerMBean(vm, vmMBeanObjectName);
        } catch (Exception ex) {
        }
        vmRefs.put(vmName, vm);
        logger.info("Added VirtualMachine " + vmName + " " + vmMBeanObjectName);
        return vm;
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>();

        try {
            ArrayOfManagedObjectReference vms = (ArrayOfManagedObjectReference) getDynamicProarray(mor, "vm")[0].getVal();
            for (ManagedObjectReference vmMor : vms.getManagedObjectReference()) {
                boolean isTemplate = (Boolean) getDynamicProarray(vmMor, "summary.config.template")[0].getVal();
                if (isTemplate) {
                    continue;
                }
                DynamicProperty[] properties = getDynamicProarray(vmMor, "config.name");
                if (properties == null || properties.length < 1) {
                    continue;
                }
                String vmName = (String) properties[0].getVal();
                VMwareVirtualMachine vm = vmRefs.get(vmName);
                if (vm == null) {
                    vm = addVm(vmMor, vmName);
                }

                result.add(vm);
            }
        } catch (Exception ex) {
            logger.error("Failed to retrieve resident VM of host " + hostName, ex);
        }

        return result;
    }

    // synchronized void onCreatedVM(ManagedObjectReference vmRef, String
    // vmName) {
    // VMwareVirtualMachine vm = vmRefs.get(vmName);
    // if (vm != null)
    // emitNotification(NotificationTypes.VM_ADD, "Created",
    // vm.getObjectName());
    // else logger.debug("VmCreatedEvent received for unknown VM " + vmName);
    // }
    //
    // synchronized void onRemovedVM(ManagedObjectReference vmRef, String
    // vmName) {
    // VMwareVirtualMachine vm = vmRefs.get(vmName);
    // if (vm != null) {
    // emitNotification(NotificationTypes.VM_DEL, "Destroyed",
    // vm.getObjectName());
    // logger.debug("Notification: VM removed: " + vmName);
    // } else
    // logger.debug("VmRemovedEvent received for unknown VM");
    // }

    synchronized void onPowerStateChangedVM(final ManagedObjectReference vmRef, final String vmName, final VirtualMachineMXBean.PowerState state,
        final long time) {
        VMwareVirtualMachine vm = vmRefs.get(vmName);
        if (vm != null) {
            vm.onPowerStateChangedVM(state, time);
        }
    }

    public long getTotalMemoryMB() {
        long total = 0;
        try {
            DynamicProperty[] cpuArry = getDynamicProarray(this.mor, "hardware");
            total = ((HostHardwareInfo) cpuArry[0].getVal()).getMemorySize();
        } catch (Exception ex) {
            logger.debug("Host " + hostName, ex);
        }
        return total / (1024 * 1024);
    }

    public long getFreeMemoryMB() {
        if (this.hostMetricValues != null) {
            return (long) this.hostMetricValues.get("mem.granted").longValue();
        } else {
            try {
                int overallMemoryUsage = (Integer) getDynamicProarray(mor, "summary.quickStats.overallMemoryUsage")[0].getVal();
                return getTotalMemoryMB() - overallMemoryUsage;
            } catch (Exception ex) {
                logger.error("Cannot retrieve attribute summary.quickStats.overallMemoryUsage", ex);
                return -1;
            }
        }
    }

    public float[] getLoadPerCPU() {
        logger.info("[VMWareHost.getLoadPerCPU] : Not Implemented");
        return null;
    }

    public float getCPULoad() { // [0,1] interval ?
        if (this.hostMetricValues != null) {
            return (this.hostMetricValues.get("cpu.usage").floatValue() / (float) 10000);
        }
        return 0;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        final VMConfigSpec vmSpec2 = new VMConfigSpec(vmSpec);
        if (!sync) {
            pool.getExecutorService().execute(new Runnable() {
                public void run() {
                    try {
                        doCreateVM(vmSpec2, sync);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return doCreateVM(vmSpec2, sync);
        }
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return pool.getVMImageStore();
    }

    private Timer perfTimer = null;

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        stopPerfMonitor();
        if (periodMillis < 20000) {
            periodMillis = 20000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            currentMonitoringPeriod = periodMillis;
            currentMonitoredMetrics = metricsOfInterest;
            startPerfMonitor();
        }
    }

    public synchronized void startPerfMonitor() {
        if (perfTimer == null) {
            perfTimer = new Timer(true);
            synchronized (connection) {
                perfMon = new VMwarePerfMonitor(connection, this);
            }
            perfMon.configure(counterNames);
            perfTimer.schedule(perfMon, 0, 20000);
        }

    }

    public synchronized void stopPerfMonitor() {
        if (perfTimer != null) {
            perfTimer.cancel();
            this.hostMetricValues = null;
            this.metricValuesPerVM = null;
            for (VMwareVirtualMachine vm : vmRefs.values()) {
                vm.updateMetrics(null);
            }
            perfTimer = null;
        }
    }

    public void updateMetrics(final HashMap<String, BigInteger> hM, final HashMap<String, HashMap<String, BigInteger>> vmM) {
        this.hostMetricValues = hM;
        this.metricValuesPerVM = vmM;

        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        for (VMwareVirtualMachine vm : vmRefs.values()) {
            String vmName = vm.getNameLabel();
            if (vmM.containsKey(vmName)) {
                vm.updateMetrics(vmM.get(vmName));
                cpuLoads.put(vmName, vm.getCPULoad() / getNumCPU());
            }
        }
        emitNotification(NotificationTypes.PERF_REPORT, "CPU load", cpuLoads);
    }

    // TODO revisit this function when VCPU >1
    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        long hostMhz = Long.parseLong(getCPUInfo().get("speed"));

        try {
            for (VMwareVirtualMachine vm : vmRefs.values()) {
                if (vm.getHostMBean() == this && vm.getState() == VirtualMachineMXBean.PowerState.RUNNING
                    && !(Boolean) getDynamicProarray(vm.getRef(), "summary.config.template")[0].getVal()) {

                    int loadMhz = (Integer) getDynamicProarray(vm.getRef(), "summary.quickStats.overallCpuUsage")[0].getVal();
                    float loadPercentage = (float) loadMhz / hostMhz;
                    if (loadPercentage > 1) {
                        loadPercentage = 1;
                    }
                    cpuLoads.put(vm.getNameLabel(), loadPercentage);
                }
            }

        } catch (Exception ex) {
            logger.error("Failed to retrieve CPU loads", ex);
        }
        return cpuLoads;
    }

    public ManagedObjectReference getVCResourcePool() {
        if (myResourcePool == null) {
            DynamicProperty[] dProps;
            try {
                dProps = getDynamicProarray(mor, "parent");
                ManagedObjectReference computeResource = (ManagedObjectReference) dProps[0].getVal();
                dProps = getDynamicProarray(computeResource, "resourcePool");
                myResourcePool = (ManagedObjectReference) dProps[0].getVal();
                logger.debug("Found resourcePool of host " + hostName);
            } catch (Exception e) {
                logger.error("Failed to get Resource Pool of host " + hostName, e);
            }
        }
        return myResourcePool;
    }

    private ManagedObjectReference findVMTemplateMOR(final String name) {
        try {
            ArrayOfManagedObjectReference vms = (ArrayOfManagedObjectReference) getDynamicProarray(mor, "vm")[0].getVal();
            for (ManagedObjectReference vmMor : vms.getManagedObjectReference()) {
                boolean isTemplate = (Boolean) getDynamicProarray(vmMor, "summary.config.template")[0].getVal();
                if (isTemplate) {
                    DynamicProperty[] properties = getDynamicProarray(vmMor, "config.name");
                    if (properties == null || properties.length < 1) {
                        continue;
                    }
                    String vmName = (String) properties[0].getVal();
                    if (vmName.equals(name)) {
                        return vmMor;
                    }
                }
            }
        } catch (Exception ex) {
            logger.error(ex);
        }
        return null;
    }

    private VirtualMachineMXBean doCreateVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB() + ",diskSize="
            + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + ", ID=" + vmSpec.getVmImageUUID() + "]");

        if (pool.getVMImageStore().lookUpByUUID(vmSpec.getVmImageUUID()) == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }

        ManagedObjectReference templateVMRef = findVMTemplateMOR(vmSpec.getVmImageUUID());
        if (templateVMRef == null) {
            logger.debug("Template " + vmSpec.getVmImageUUID() + " not found");
            return null;
        }
        VMwareVirtualMachine newVM;
        ManagedObjectReference vmMor = null;
        ObjectName name = null;

        synchronized (connection) {

            VirtualMachineCloneSpec cloneSpec = new VirtualMachineCloneSpec();
            VirtualMachineRelocateSpec relocSpec = new VirtualMachineRelocateSpec();
            try {
                relocSpec.setPool(getVCResourcePool());
                relocSpec.setDatastore(pool.getDatastoreRef());
                cloneSpec.setLocation(relocSpec);
                cloneSpec.setPowerOn(false);
                cloneSpec.setTemplate(false);
                String clonedName = vmSpec.getName();

                ManagedObjectReference cloneTask = connection.getService().cloneVM_Task(templateVMRef, pool.getVmFolderRef(),
                    clonedName, cloneSpec);
                logger.debug("CloneVMTask launched, waiting for result...");

                String status = connection.waitForTask(cloneTask);

                logger.debug("CloneVMTask, status=" + status);

                if (status.equalsIgnoreCase("failure")) {
                    String msg = "Failure -: Virtual Machine cannot be cloned";
                    logger.error(msg);
                    throw new VMMException(msg);
                }

                if (status.equalsIgnoreCase("success")) {
                    logger.debug("Virtual Machine cloned  successfully.");
                    // TaskInfo = getDynamicProarray(cloneTask,"info");
                    vmMor = (ManagedObjectReference) getDynamicProarray(cloneTask, "info.result")[0].getVal();
                }
            } catch (Exception e) {
                if (e instanceof org.apache.axis.AxisFault) {
                    logger.debug("AxisFault Exception...");
                    org.apache.axis.AxisFault fault = (org.apache.axis.AxisFault) e;
                    org.w3c.dom.Element[] errors = fault.getFaultDetails();
                    for (int i = 0; i < errors.length; i++) {
                        if (errors[i].toString().indexOf("DuplicateName") != -1) {
                            logger.debug("Virtual Machine with the same name already exists");
                            throw new InvalidVMConfigException("Virtual Machine with the same name already exists");
                        } else if (errors[i].toString().indexOf("InvalidArgument") != -1) {
                            logger.debug("Specification is invalid");
                            throw new InvalidVMConfigException();
                        } else if (errors[i].toString().indexOf("InvalidName") != -1) {
                            logger.debug("Virtual Machine name is empty or too long");
                            throw new InvalidVMConfigException("Virtual Machine name is empty or too long");
                        } else if (errors[i].toString().indexOf("RuntimeFault") != -1) {
                            logger.debug(errors[i].toString());
                            throw new VMMException();
                        } else {
                            logger.debug(errors[i].toString());
                            throw new VMMException();
                        }
                    }
                } else {
                    logger.error("CloneVMTask Exception...", e);
                    throw new VMMException(e);
                }
            }
        }

        try {
            name = MBeanObjectNamer.makeVirtualMachineName(pool.getPath() + "/" + vmSpec.getName(), VMwareVirtualMachine
                .getUuid(connection, vmMor));
            newVM = new VMwareVirtualMachine(name, pool, vmMor, this, vmSpec.getName());
            vmRefs.put(vmSpec.getName(), newVM);
            AgentCommon.getMBeanServer().registerMBean(newVM, name);

        } catch (Exception ex) {
            logger.error("Failed to register new VM MBean " + name, ex);
            return null;
        }

        // before starting the VM set the memory and number of CPU parameters
        setResourceCapacity(vmMor, vmSpec);

        newVM.start();
        logger.debug("[Host " + hostName + "] VM started");

        emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return newVM;
    }

    void onVMDestroy(final VMwareVirtualMachine vm) {
        vmRefs.remove(vm.getNameLabel());
        emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
    }

    private void setResourceCapacity(final ManagedObjectReference vmMor, final VMConfigSpec vmSpec) {
        VirtualMachineConfigSpec vmConfigSpec = new VirtualMachineConfigSpec();
        try {
            vmConfigSpec.setMemoryMB(vmSpec.getMemorySizeMB());
            vmConfigSpec.setNumCPUs(vmSpec.getNumVCPU());
            synchronized (connection) {
                ManagedObjectReference tmor = connection.getService().reconfigVM_Task(vmMor, vmConfigSpec);

                String status = connection.waitForTask(tmor);

                if (status.equalsIgnoreCase("failure")) {
                    String msg = "Failure -: Virtual Machine cannot be cloned";
                    logger.error(msg);
                } else {
                    logger.debug("VM " + vmSpec.getName() + ": Memory capacity set to " + vmSpec.getMemorySizeMB()
                        + "MB, number of CPU set to " + vmSpec.getNumVCPU());
                }
            }
        } catch (Exception e) {
            logger.error("Failed to set VM memory and numCPU", e);
        }
    }

    private DynamicProperty[] getDynamicProarray(final ManagedObjectReference MOR, final String pName) throws Exception {
        ObjectContent[] objContent = connection.getObjectProperties(null, MOR, new String[] {pName});
        ObjectContent contentObj = objContent[0];
        DynamicProperty[] objArr = contentObj.getPropSet();
        return objArr;
    }

    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

}
