/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareServerPool.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

import com.vmware.vim.ArrayOfManagedObjectReference;
import com.vmware.vim.DynamicProperty;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;

/**
 * VMware driver ServerPool MXBean implementation
 */
class VMwareServerPool extends ServerPool {
    static protected Logger logger = Logger.getLogger(VMwareServerPool.class);

    static private int KEEP_ALIVE_PERIOD_MS = 2 * 60 * 1000;

    private static final int THREADPOOL_SIZE = 20;

    private static ExecutorService executorService = Executors.newFixedThreadPool(THREADPOOL_SIZE);

    private VMwareServiceConnection connection;

    private HashMap<String, VMwareHost> hostRefs = new HashMap<String, VMwareHost>();

    private String mountPoint;

    private ManagedObjectReference mountedFolderRef, vmFolderRef, datastoreRef;

    private VMwareVMTemplateDataStore imageStore;

    public VMwareServerPool(final String name, final ObjectName objectName, final Properties props) throws VMMException {
        super(name, objectName);
        String hostName = props.getProperty("virtualCenterHostName");
        mountPoint = props.getProperty("mountpoint");
        String userName = props.getProperty("user");
        String password = props.getProperty("password");
        String datastoreName = props.getProperty("datastore");

        try {
            connection = new VMwareServiceConnection(hostName, userName, password);
            logger.debug("Connecting to Virtual Center...");
            connection.connect();
            logger.debug("Done.");

            mountedFolderRef = connection.getService().findByInventoryPath(connection.getServiceContent().getSearchIndex(),
                mountPoint);
            if (mountedFolderRef == null) {
                String msg = "VMware Driver initialization error: mountpoint " + mountPoint + " not found ";
                logger.error(msg);
                throw new VMMException(msg);
            }

            // Find the virtual machine folder for this folder.
            vmFolderRef = (ManagedObjectReference) getDynamicProarray(mountedFolderRef, "vmFolder")[0].getVal();
            if (vmFolderRef == null) {
                String msg = "VMware Driver initialization error: virtual machine Folder is not found";
                logger.error(msg);
                throw new VMMException(msg);
            }

            // find the datastore
            ArrayOfManagedObjectReference dataStoreRefs = (ArrayOfManagedObjectReference) getDynamicProarray(mountedFolderRef,
                "datastore")[0].getVal();
            if (dataStoreRefs != null) {
                for (int i = 0; i < dataStoreRefs.getManagedObjectReference().length; i++) {
                    String dsName = (String) getDynamicProarray(dataStoreRefs.getManagedObjectReference(i), "info.name")[0]
                        .getVal();
                    if (dsName.equals(datastoreName)) {
                        datastoreRef = dataStoreRefs.getManagedObjectReference(i);
                        break;
                    }
                }
            }
            if (datastoreRef == null) {
                String msg = "VMware Driver initialization error: Datastore " + datastoreName + " cannot be found";
                logger.error(msg);
                throw new VMMException(msg);
            }

            String vmImageStoreName = mountPoint + "-VMIStore"; // TODO
            imageStore = new VMwareVMTemplateDataStore(MBeanObjectNamer.makeVMImageStoreName(vmImageStoreName),
                vmImageStoreName, datastoreRef);
            AgentCommon.getMBeanServer().registerMBean(imageStore, imageStore.getObjectName());

        } catch (Exception ex) {
            String msg = "VMware Driver initialization error: " + ex.getMessage();
            logger.error(msg, ex);
            throw new VMMException(msg);
        }

        new VMwareEventCollector(this, new VMwareServiceConnection(hostName, userName, password), mountPoint).start();

        executorService.execute(new Runnable() {
            public void run() {
                while (true) {
                    try {
                        Thread.sleep(KEEP_ALIVE_PERIOD_MS);
                        synchronized (connection) {
                            Calendar time = connection.getService().currentTime(connection.getServiceInstanceRef());
                        }

                    } catch (Exception ex) {
                        logger.error("error", ex);
                    }
                }
            }
        });
    }

    public ExecutorService getExecutorService() {
        return executorService;
    }

    public VMwareServiceConnection getConnection() {
        return connection;
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return imageStore;
    }

    public String getMountPoint() {
        return mountPoint;
    }

    VMwareHost getHostByName(final String name) {
        return hostRefs.get(name);
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) {
        VMwareHost host = hostRefs.get(hostName);
        if (host != null) {
            return host;
        }

        ManagedObjectReference mor = null;
        try {
            synchronized (connection) {
                mor = connection.getDecendentMoRef(null, "HostSystem", hostName);
            }
        } catch (Exception ex) {
            logger.error("Unable to find HostSystem managed object " + hostName + " in VMware VC inventory", ex);
            return null;
        }

        if (mor == null) {
            logger.error("Host " + hostName + " not found");
            return null;
        }

        ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(getPath() + "/" + hostName, hostName);
        host = new VMwareHost(this, mor, mbeanObjectName, hostName);
        host.getResidentVMs();

        try {

            AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
        } catch (Exception ex) {
            logger.error("Unable to register MBean " + mbeanObjectName, ex);
            return null;
        }
        hostRefs.put(hostName, host);
        // logger.info("Added host "+hostName+" "+host.getObjectName());
        addHost(host);
        return host;
    }

    public ManagedObjectReference getMountedFolderRef() {
        return mountedFolderRef;
    }

    public ManagedObjectReference getVmFolderRef() {
        return vmFolderRef;
    }

    public ManagedObjectReference getDatastoreRef() {
        return datastoreRef;
    }

    private DynamicProperty[] getDynamicProarray(final ManagedObjectReference MOR, final String pName) throws Exception {
        ObjectContent[] objContent;
        objContent = connection.getObjectProperties(null, MOR, new String[] {pName});
        ObjectContent contentObj = objContent[0];
        DynamicProperty[] objArr = contentObj.getPropSet();
        return objArr;
    }

    private class VMwareVMTemplateDataStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
        private String name;

        private ArrayList<VirtualMachineImageMXBean> imageList = new ArrayList<VirtualMachineImageMXBean>();

        private ManagedObjectReference datastoreRef;

        public VMwareVMTemplateDataStore(final ObjectName objectName, final String name, final ManagedObjectReference datastoreRef) {
            super(objectName);
            this.name = name;
            this.datastoreRef = datastoreRef;
        }

        public long getCapacityMB() {
            try {
                long capacityInBytes = (Long) getDynamicProarray(datastoreRef, "summary.capacity")[0].getVal();
                return capacityInBytes / 1024;
            } catch (Exception ex) {
                logger.error("Failed to get datastore capacity", ex);
                return -1;
            }
        }

        public long getFreeSpaceMB() {
            try {
                long freeSpaceInBytes = (Long) getDynamicProarray(datastoreRef, "summary.freeSpace")[0].getVal();
                return freeSpaceInBytes / 1024;
            } catch (Exception ex) {
                logger.error("Failed to get datastore free space", ex);
                return -1;
            }
        }

        public String getName() {
            return name;
        }

        public List<VirtualMachineImageMXBean> listVMImageTemplates() {
            if (imageList.size() == 0) {
                try {
                    ArrayList<ManagedObjectReference> vms = connection.getDecendentMoRefs(
                        getMountedFolderRef(), "VirtualMachine");

                    for (ManagedObjectReference vmMor : vms) {
                        boolean isTemplate = (Boolean) getDynamicProarray(vmMor, "summary.config.template")[0].getVal();
                        if (isTemplate) {
                            DynamicProperty[] properties = getDynamicProarray(vmMor, "config.name");
                            if (properties == null || properties.length < 1) {
                                continue;
                            }
                            String vmName = (String) properties[0].getVal();
                            String uuid = (String) getDynamicProarray(vmMor, "config.uuid")[0].getVal();
                            ObjectName on = MBeanObjectNamer.makeVMImageName(uuid);
                            VMwareVirtualMachineImage image = new VMwareVirtualMachineImage(on, vmName, uuid);
                            logger.debug("New VMware template: " + image);
                            AgentCommon.getMBeanServer().registerMBean(image, on);
                            imageList.add(image);
                        }

                    }

                } catch (Exception ex) {
                    logger.error("Failed to list VM templates", ex);
                }
            }
            return imageList;
        }

        public VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
            for (VirtualMachineImageMXBean image : imageList) {
                if (image.getUUID().equals(uuid)) {
                    return image;
                }
            }
            return null;
        }

        public void removeVMImageTemplate(final VirtualMachineImageMXBean vmImage) throws VMMException {
            String vmImageID = vmImage.getUUID();
            try {
                ManagedObjectReference vmMOR = connection.getDecendentMoRef(null, "VirtualMachine",
                    vmImageID);
                // asynchronous op, forget about result ?
                connection.getService().destroy_Task(vmMOR);
            } catch (Exception ex) {
                logger.error("Failed to retrieve VM reference " + vmImageID, ex);
            }
        }

    }

    private static class VMwareVirtualMachineImage extends ManagedResource implements VirtualMachineImageMXBean {
        private String name, uuid;

        public VMwareVirtualMachineImage(final ObjectName objectName, final String name, final String uuid) {
            super(objectName);
            this.name = name;
            this.uuid = uuid;
        }

        public String getName() {
            return name;
        }

        // TODO we ignore the UUID and use the name as a unique
        public String getUUID() {
            return name;
        }

        public String getMetaData() {
            return "";
        }

        @Override
        public String toString() {
            return "VMwareTemplate(name=" + getName() + ",uuid=" + getUUID() + ")";
        }
    }

}
