/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenHost.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.io.BufferedReader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.agent.main.VirtManagerAgent;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.HostCpu;
import com.xensource.xenapi.VM;

/**
 * Xen driver Host MXBean implementation
 */
public class XenHost extends ManagedResource implements HostMXBean, NotificationEmitter {
    static Logger logger = Logger.getLogger(XenHost.class);

    static String XEN_CONFIG_HOME = "/etc/xen/auto";

    // the thread pool size determines the maximum number of
    // simultaneous VM creation on the same host
    private static final int MAX_PARALELL_VM_CREATION_PER_HOST = 3;

    // one extra thread for the event reader
    private static final int THREADPOOL_SIZE = 1 + MAX_PARALELL_VM_CREATION_PER_HOST;

    private static ExecutorService executorService = Executors.newFixedThreadPool(THREADPOOL_SIZE);

    private List<XenVirtualMachine> vmList = new CopyOnWriteArrayList<XenVirtualMachine>();

    private XenServerPool serverPool;

    private Connection connection;

    private Timer perfTimer;

    private com.xensource.xenapi.Host host;

    private String hostName;

    private int numCPU = -1;

    private float aggregatedCPULoad = -1;

    private long cpuFrequencyMhz = -1;

    private String sshPassword;

    public static class XenAPIParams {
        private final String xenAPIPort;

        private final String xenAPIUser;

        private final String xenAPIPassword;

        public XenAPIParams(final String xenAPIPort, final String xenAPIUser, final String xenAPIPassword) {
            this.xenAPIPort = xenAPIPort;
            this.xenAPIUser = xenAPIUser;
            this.xenAPIPassword = xenAPIPassword;
        }

        public String getXenAPIPort() {
            return xenAPIPort;
        }

        public String getXenAPIUser() {
            return xenAPIUser;
        }

        public String getXenAPIPassword() {
            return xenAPIPassword;
        }

    }

    public static XenHost newHost(final XenServerPool resourcePool, final ObjectName objectName, final String hostName,
        final XenAPIParams xenAPIParams, final String sshPassword) {

        try {
            String xenAPIURL = "http://" + hostName + ":" + xenAPIParams.getXenAPIPort();
            Connection connection = new Connection(xenAPIURL, xenAPIParams.getXenAPIUser(), xenAPIParams.getXenAPIPassword());
            Connection connection2 = new Connection(xenAPIURL, xenAPIParams.getXenAPIUser(), xenAPIParams.getXenAPIPassword());
            XenHost xenHost = new XenHost(resourcePool, connection, connection2, objectName, hostName, sshPassword);
            return xenHost;
        } catch (Exception ex) {
            return null;
        }

    }

    private XenHost(final XenServerPool serverPool, final Connection connection, final Connection connection2,
        final ObjectName objectName, final String hostName, final String sshPassword) {
        super(objectName);
        this.serverPool = serverPool;
        this.connection = connection;
        this.hostName = hostName;
        this.sshPassword = sshPassword;

        try {
            Set<com.xensource.xenapi.Host> hosts = com.xensource.xenapi.Host.getAll(connection);
            host = hosts.iterator().next();
            populate();
        } catch (Exception ex) {
            logger.debug("Host " + hostName, ex);
        }

        // executorService.execute(new Runnable() {
        // public void run() {
        // try {
        // Set<String> eventClasses = new HashSet<String>();
        // Event.register(connection2, eventClasses);
        // while (true) {
        // Set<Event.Record> events = Event.next(connection2);
        // for (Event.Record ev : events) {
        // if (ev.clazz.equals("VM")) {
        // XenVirtualMachine xenVM = null;
        // for (XenVirtualMachine vm : vmList)
        // if (vm.getUuid().equals(ev.objUuid)) {
        // xenVM = vm;
        // }
        // if (xenVM != null) {
        // if (ev.operation == Types.EventOperation.MOD)
        // xenVM.onVMStateChanged();
        // else if (ev.operation == Types.EventOperation.ADD) {
        //
        // } else if (ev.operation == Types.EventOperation.DEL) {
        //
        // }
        // }
        // }
        // }
        // }
        // } catch (Exception ex) {
        // logger.debug("Host " + XenHost.this.hostName, ex);
        // }
        // }
        // });

    }

    Connection getXenAPIConnection() {
        return connection;
    }

    private void addVM(final XenVirtualMachine vm) {
        vmList.add(vm);
    }

    private void removeVM(final XenVirtualMachine vm) {
        vmList.remove(vm);
    }

    public ServerPoolMXBean getServerPool() {
        return serverPool;
    }

    public void postMigrateVM(final XenVirtualMachine vm, final XenHost newHost) {
        removeVM(vm);
        newHost.addVM(vm);
    }

    public void onVMDestroy(final XenVirtualMachine vm) {
        removeVM(vm);
        emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            currentMonitoringPeriod = periodMillis;
            currentMonitoredMetrics = metricsOfInterest;
            startPerfMonitor();
        }
    }

    String getIPAddress(String macAddress) {
        macAddress = macAddress.toUpperCase();

        logger.debug("Determining IP address from MAC address " + macAddress + " ...");

        RemoteExec.Result result = null;
        String command = serverPool.getVMMDomain0HomeDir() + "/bin/getIPfromMAC.sh " + macAddress;

        logger.debug("Launching command: " + command);

        try {
            result = RemoteExec.commandAsRoot(hostName, sshPassword, command);
        } catch (RemoteExec.SshException ex) {
            logger.debug("SSH failure", ex);
            return null;
        }
        if (result.exitCode != 0) {
            logger.error("Cannot get IP from MAC " + result.output);
            return null;
        }

        String ipAddress = result.output;
        if (ipAddress != null && ipAddress.length() > 0 && ipAddress.endsWith("\n")) {
            ipAddress = ipAddress.substring(0, ipAddress.length() - 1);
        }
        logger.debug("Mac-to-IP " + macAddress + " -> " + ipAddress);
        return ipAddress;
    }

    private synchronized void startPerfMonitor() {
        if (perfTimer == null) {
            perfTimer = new Timer();
            perfTimer.schedule(new PerfTimerTask(), 0, currentMonitoringPeriod);
            logger.info("Host " + hostName + ": started periodic performance reporting task");
        }
    }

    private synchronized void stopPerfMonitor() {
        if (perfTimer != null) {
            perfTimer.cancel();
            logger.info("Host " + hostName + ": stopped periodic performance reporting task");
        }
        perfTimer = null;
    }

    @Override
    public ObjectName getObjectName() {
        return objectName;
    }

    HashMap<String, String> hypervisorInfo;

    // TODO
    public Map<String, String> getHypervisorInfo() {
        if (hypervisorInfo == null) {
            hypervisorInfo = new HashMap<String, String>();
            hypervisorInfo.put("vendor", "");
            hypervisorInfo.put("name", "Xen");
            hypervisorInfo.put("version", "3.2");
        }
        return hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (cpuInfo == null) {
            cpuInfo = new HashMap<String, String>();
            synchronized (connection) {
                try {
                    HostCpu[] cpus = host.getHostCPUs(connection).toArray(new HostCpu[0]);
                    numCPU = cpus.length;
                    HostCpu.Record rec = cpus[0].getRecord(connection);
                    cpuInfo.put("model", rec.modelname);
                    cpuFrequencyMhz = rec.speed;
                    cpuInfo.put("speed", Long.toString(rec.speed));
                    cpuInfo.put("vendor", rec.vendor);
                    // cpuInfo.put("features",rec.features);
                    // cpuInfo.put("flags",rec.flags);
                } catch (Exception ex) {
                    logger.debug("Host " + hostName, ex);
                }
            }
        }
        return cpuInfo;
    }

    public long getCPUFrequencyMhz() {
        if (cpuFrequencyMhz == -1) {
            getCPUInfo();
        }
        return cpuFrequencyMhz;

    }

    public int getNumCPU() {
        if (numCPU == -1) {
            getCPUInfo();
        }
        return numCPU;
    }

    public String getHostName() {
        return hostName;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return serverPool.getVMImageStore();
    }

    public long getFreeMemoryMB() {
        RemoteExec.Result result = null;
        String command = "xm info | grep ^free_memory | awk '{print $3}'";
        try {
            result = RemoteExec.commandAsRoot(hostName, sshPassword, command);
        } catch (RemoteExec.SshException ex) {
            logger.error("SSH exception: ", ex);
        }
        if (result.exitCode != 0) {
            logger.error("getFreeMemoryMB: SSH " + command + " failed");
            return 0;
        }
        if (result.output.endsWith("\n")) {
            result.output = result.output.substring(0, result.output.length() - 1);
        }
        long freeMem = Long.parseLong(result.output);

        for (VirtualMachineMXBean vm : vmList) {
            XenVirtualMachine xvm = (XenVirtualMachine) vm;
            if (xvm.getDomID() == 0) {
                freeMem += xvm.getMemorySizeMB() - (196 + 350); // XXX
            }
        }

        return freeMem;
    }

    private void populate() {
        try {
            synchronized (connection) {
                Set<VM> vms = VM.getAll(connection);
                for (VM vm : vms) {
                    ObjectName name;
                    if (vm.getDomid(connection) == 0) {
                        // for Domain 0, we use the host UUID
                        name = MBeanObjectNamer.makeVirtualMachineName(
                            serverPool.getPath() + "/" + vm.getNameLabel(connection), host.getUuid(connection));
                    } else {
                        name = MBeanObjectNamer.makeVirtualMachineName(
                            serverPool.getPath() + "/" + vm.getNameLabel(connection), vm.getUuid(connection));
                    }

                    XenVirtualMachine xenVM = new XenVirtualMachine(name, this, connection, vm, null);
                    addVM(xenVM);
                    AgentCommon.getMBeanServer().registerMBean(xenVM, name);
                    logger.info("Added XenVirtualMachineMBean " + name);
                }
            }
        } catch (Exception ex) {
            logger.debug("Host " + hostName, ex);
        }
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>(vmList);
        return result;
    }

    private VirtualMachineMXBean doCreateVM(final VMConfigSpec vmSpec, final boolean sync) throws InvalidVMConfigException, VMMException {
        logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB() + ",diskSize="
            + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + "]");

        if (serverPool.getVMImageStore().lookUpByUUID(vmSpec.getVmImageUUID()) == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }
        // XXX we make the vm label unique by appending the current time
        String vmName = vmSpec.getName() + "-" + System.currentTimeMillis();

        String command = serverPool.getVMMDomain0HomeDir() + "/bin/createVM " + vmName + " " + vmSpec.getDiskSizeMB() + " "
            + vmSpec.getVmImageUUID() + " " + vmSpec.getMemorySizeMB() + " " + vmSpec.getNumVCPU() + " "
            + UUID.randomUUID().toString() + " 2>&1";
        logger.debug("launching command " + command);
        RemoteExec.Result result;
        try {
            result = RemoteExec.commandAsRoot(hostName, sshPassword, command);
        } catch (RemoteExec.SshException ex) {
            logger.error("command failed: ", ex);
            throw new VMMException("VM creation failed: connection failure", ex);
        }
        if (result.exitCode != 0) {
            logger.error("Command " + command + " failed: " + result.output);
            // TODO parse error message
            throw new VMMException("VM creation failed: " + result.output);
        }
        VM newVM = null;
        String vmUUID = "";
        synchronized (connection) {
            try {
                newVM = VM.getByNameLabel(connection, vmName).iterator().next();
                vmUUID = newVM.getUuid(connection);
            } catch (Exception ex) {
                logger.error("Failed to get VM uuid: ", ex);
            }
        }
        ObjectName name = null;
        XenVirtualMachine xenVM = null;
        try {
            name = MBeanObjectNamer.makeVirtualMachineName(serverPool.getPath() + "/" + vmName, vmUUID);
            HashMap<String, String> map = new HashMap<String, String>();
            map.put("imageID", vmSpec.getVmImageUUID());
            xenVM = new XenVirtualMachine(name, this, connection, newVM, map);
            addVM(xenVM);

            String mac = xenVM.getMacAddress();

            try {
                command = "sed -i \"s/\\[' '\\]/\\['mac=" + mac + "'\\]/g\" " + XEN_CONFIG_HOME + "/" + xenVM.getNameLabel()
                    + ".cfg";
                result = RemoteExec.commandAsRoot(hostName, sshPassword, command);
                if (result.exitCode != 0) {
                    logger.error("Command " + command + " failed: " + result.output);
                }
            } catch (Exception e) {
            }

            AgentCommon.getMBeanServer().registerMBean(xenVM, name);
        } catch (Exception ex) {
            logger.debug("Host " + hostName, ex);
            throw new VMMException(ex);
        }

        emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return xenVM;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpec, final boolean sync) throws InsufficientResourcesException,
        InvalidVMConfigException, VMMException {
        final VMConfigSpec vmSpec2 = new VMConfigSpec(vmSpec);
        if (!sync) {
            executorService.execute(new Runnable() {
                public void run() {
                    try {
                        doCreateVM(vmSpec2, sync);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return doCreateVM(vmSpec2, sync);
        }
    }

    public long getTotalMemoryMB() {
        synchronized (connection) {
            try {
                return host.getMetrics(connection).getMemoryTotal(connection) / (1024 * 1024);
            } catch (Exception ex) {
                logger.debug("Host " + hostName, ex);
            }
        }
        return 0;
    }

    public float[] getLoadPerCPU() {
        float[] result = new float[0];
        synchronized (connection) {
            try {
                Set<HostCpu> cpus = host.getHostCPUs(connection);
                result = new float[cpus.size()];
                int i = 0;
                for (HostCpu cpu : cpus) {
                    result[i++] = (float) (cpu.getUtilisation(connection) * 100);
                }
            } catch (Exception ex) {
                logger.debug("Host " + hostName, ex);
            }
        }
        return result;
    }

    public float getCPULoad() {
        return aggregatedCPULoad;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

    RemoteExec.Result remoteExec(final String command) throws Exception {
        RemoteExec.Result result = null;
        result = RemoteExec.commandAsRoot(hostName, sshPassword, command);
        return result;
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        RemoteExec.Result result = null;
        String command = "xentop -b -i 2 -d 1";
        try {
            result = RemoteExec.commandAsRoot(XenHost.this.hostName, sshPassword, command);
        } catch (RemoteExec.SshException ex) {
            return cpuLoads;
        }

        try {
            BufferedReader reader = new BufferedReader(new StringReader(result.output));
            String s;
            int matched = 0;
            while (true) {
                s = reader.readLine();
                if (s == null) {
                    break;
                }
                if (s.indexOf("NAME") != -1) {
                    matched++;
                    if (matched == 2) {
                        break;
                    }
                }
            }
            // skip header line
            // reader.readLine();
            float acc = 0;
            while ((s = reader.readLine()) != null) {
                StringTokenizer st = new StringTokenizer(s);
                String vmLabel = st.nextToken();
                st.nextToken(); // state
                st.nextToken(); // CPU (sec)
                String cpuLoadString = st.nextToken();
                st.nextToken(); // MEM(k)
                st.nextToken(); // MEM (%)
                String t = st.nextToken(); // MAXMEM (k)
                if (t.equals("no")) {
                    st.nextToken(); // limit
                }
                st.nextToken(); // MAXMEM (%)
                String vcpusString = st.nextToken();
                float cpuLoad = Float.parseFloat(cpuLoadString) / 100;
                cpuLoad /= getNumCPU(); // Integer.parseInt(vcpusString);
                cpuLoads.put(vmLabel, cpuLoad);
            }
        } catch (Exception ex) {
            logger.debug("Host " + hostName, ex);
        }
        return cpuLoads;
    }

    private long lastCPULoadSampleTime = 0;

    private class PerfTimerTask extends TimerTask {
        @Override
        public void run() {
            HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
            RemoteExec.Result result = null;
            // two iterations of xentop, we read the second one...
            String command = "xentop -b -i 2 -d 1";
            try {
                result = RemoteExec.commandAsRoot(XenHost.this.hostName, XenHost.this.sshPassword, command);
            } catch (RemoteExec.SshException ex) {
                logger.error("Failed to invoke xentop", ex);
                return;
            }
            long now = System.currentTimeMillis();
            try {
                BufferedReader reader = new BufferedReader(new StringReader(result.output));
                String s;
                int matched = 0;
                while (true) {
                    s = reader.readLine();
                    if (s == null) {
                        break;
                    }
                    if (s.indexOf("NAME") != -1) {
                        matched++;
                        if (matched == 2) {
                            break;
                        }
                    }
                }
                // skip header line
                // reader.readLine();
                float acc = 0;
                boolean sampleCPU = (now - lastCPULoadSampleTime) > 20 * 1000;
                while ((s = reader.readLine()) != null) {
                    StringTokenizer st = new StringTokenizer(s);
                    String vmLabel = st.nextToken();
                    st.nextToken(); // state
                    st.nextToken(); // CPU (sec)
                    String cpuLoadString = st.nextToken();
                    st.nextToken(); // MEM(k)
                    st.nextToken(); // MEM (%)
                    String t = st.nextToken(); // MAXMEM (k)
                    if (t.equals("no")) {
                        st.nextToken(); // limit
                    }
                    st.nextToken(); // MAXMEM (%)
                    String vcpusString = st.nextToken();
                    float cpuLoad = Float.parseFloat(cpuLoadString) / 100;
                    cpuLoad /= getNumCPU();

                    cpuLoads.put(vmLabel, cpuLoad);

                    if (sampleCPU) {
                        for (VirtualMachineMXBean vm : vmList) {
                            XenVirtualMachine xvm = (XenVirtualMachine) vm;
                            if (xvm.getNameLabel().equals(vmLabel)) {
                                xvm.updateCPUConsumption(now, cpuLoad);
                            }
                        }
                    }

                    acc += cpuLoad;
                }
                aggregatedCPULoad = acc;
                if (sampleCPU) {
                    lastCPULoadSampleTime = now;
                }
            } catch (Exception ex) {
                logger.debug("Host " + hostName, ex);
            }
            emitNotification(NotificationTypes.PERF_REPORT, "CPU load", cpuLoads);
        }
    }
}
