/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenServerPool.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.Map;
import java.util.Properties;

import javax.management.ObjectName;

import org.ow2.jasmine.vmm.agent.domain.ServerPool;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;

/**
 * The XenServerPool is the Xen-driver-specific implementation of the
 * ServerPoolMXBean interface.
 * <p>
 * The Xen driver resorts to a combination of SSH commands and XenAPI
 * connections to control each Xen host of a XenServerPool. The XenServerPool
 * accepts the following properties:
 * <ul>
 * <li>sshDefaultRootPassword : default root ssh password
 * <li>vmmDomain0HomeDir : directory where the driver host-side shell scripts
 * are installed
 * <li>xenapiDefaultUser : XenAPI default login
 * <li>xenapiDefaultPassword : XenAPI default password
 * <li>xenapiDefaulPort : XenAPI default port
 * </ul>
 * These properties will be used to connect to each host member of the
 * XenServerPool.
 */
public class XenServerPool extends ServerPool {
    private String xenAPIPort;

    private String xenAPIUser;

    private String xenAPIPassword;

    private VirtualMachineImageStoreMXBean imageStore;

    private String vmmDomain0HomeDir;

    private String vmImageStoreDir;

    private String sshPassword;

    public XenServerPool(final String name, final ObjectName objectName, final Properties props) {
        super(name, objectName);
        xenAPIPort = props.getProperty("xenapiDefaulPort", "9363");
        xenAPIUser = props.getProperty("xenapiDefaultUser", "root");
        xenAPIPassword = props.getProperty("xenapiDefaultPassword");
        vmmDomain0HomeDir = props.getProperty("vmmDomain0HomeDir", "/opt/vmm");
        vmImageStoreDir = props.getProperty("vmImageStoreDir", "/var/xen/images/templates");
        sshPassword = props.getProperty("sshDefaultRootPassword");
    }

    @Override
    public HostMXBean newHost(final String hostName, final Map<String, String> props) {
        try {
            for (HostMXBean host : hosts) {
                if (host.getHostName().equals(hostName)) {
                    return host;
                }
            }
            ObjectName mbeanObjectName = MBeanObjectNamer.makeHostName(getPath() + "/" + hostName, hostName);
            XenHost host = XenHost.newHost(this, mbeanObjectName, hostName, new XenHost.XenAPIParams(xenAPIPort, xenAPIUser,
                xenAPIPassword), sshPassword);
            if (host != null) {
                AgentCommon.getMBeanServer().registerMBean(host, mbeanObjectName);
                hosts.add(host);
                logger.info("In domain " + name + ": added Host MBean " + mbeanObjectName);
                if (imageStore == null) {
                    ObjectName defaultDataStoreName = MBeanObjectNamer.makeVMImageStoreName(hostName + ".default");
                    imageStore = new XenVMImageStore(defaultDataStoreName, vmImageStoreDir, hostName, vmmDomain0HomeDir,
                        sshPassword);
                    AgentCommon.getMBeanServer().registerMBean(imageStore, defaultDataStoreName);
                }
            }

            return host;
        } catch (Exception ex) {
            logger.error("Failed to add XenHost", ex);
            return null;
        }
    }

    @Override
    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return imageStore;
    }

    public String getVMMDomain0HomeDir() {
        return vmmDomain0HomeDir;
    }

}
