/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenVMImageStore.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.management.ObjectName;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec.SshException;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Xen driver VirtualMachineImageStore MXBean implementation This store is
 * implemented as a directory containing for each VM template:
 * <ul>
 * <li>a XML metadata file
 * <li>the image file (raw filesystem image)
 * </ul>
 */
public class XenVMImageStore extends ManagedResource implements VirtualMachineImageStoreMXBean {
    static Logger logger = Logger.getLogger(XenVMImageStore.class);

    private String rootDirectory;

    private String storeServer;

    private String sshPassword;

    private String vmmDomain0HomeDir;

    private ArrayList<XenVirtualMachineImage> vmList = new ArrayList<XenVirtualMachineImage>();

    public XenVMImageStore(final ObjectName objectName, final String rootDirectory, final String storeServer, final String vmmDomain0HomeDir,
        final String sshPassword) throws VMMException {
        super(objectName);
        this.storeServer = storeServer;
        this.rootDirectory = rootDirectory;
        this.sshPassword = sshPassword;
        this.vmmDomain0HomeDir = vmmDomain0HomeDir;
        sync();
    }

    public void removeVMImageTemplate(final VirtualMachineImageMXBean vmImage) throws VMMException {
        String vmImageID = vmImage.getUUID();
        try {
            String command = vmmDomain0HomeDir + "/bin/deleteVMTemplate " + vmImageID;
            logger.debug("Removing VM image: " + command);
            RemoteExec.Result result = RemoteExec.commandAsRoot(storeServer, sshPassword, command);
            if (result.exitCode != 0) {
                logger.error(result.error);
                throw new VMMException(result.error);
            }
        } catch (Exception ex) {
            throw new VMMException(ex.getMessage());
        }
    }

    synchronized void sync() throws VMMException {
        RemoteExec.Result result;
        try {
            String command = "cd " + rootDirectory + " ; ls -1 *.xml";
            logger.debug("Listing VM images: " + command);
            result = RemoteExec.commandAsRoot(storeServer, sshPassword, command);
        } catch (SshException ex) {
            throw new VMMException(ex.getMessage());
        }
        if (result.exitCode != 0) {
            logger.error(result.error);
            throw new VMMException(result.error);
        }
        String imageFileNames[] = result.output.split("\\s+");
        // add and register new images
        for (String fileName : imageFileNames) {
            String uuid = fileName.substring(0, fileName.length() - 4);
            if (lookUpByUUID(uuid) == null) {
                String command = "cd " + rootDirectory + " ; cat " + fileName;
                try {
                    result = RemoteExec.commandAsRoot(storeServer, sshPassword, command);
                } catch (RemoteExec.SshException ex) {
                    logger.error("SSH failure", ex);
                    continue;
                }
                if (result.exitCode != 0) {
                    continue;
                }
                XenVirtualMachineImage image = XenVirtualMachineImage.newVMImage(MBeanObjectNamer.makeVMImageName(uuid),
                    result.output);
                if (image == null) {
                    continue;
                }
                try {
                    AgentCommon.getMBeanServer().registerMBean(image, image.getObjectName());
                    logger.info("Added VMImage " + image.name + " (" + image.description + ")");
                } catch (Exception ex) {
                    logger.error("Failed to register VMImage MBean", ex);
                    continue;
                }
                vmList.add(image);
            }
        }
        // get rid of deleted images
        for (XenVirtualMachineImage image : vmList) {
            boolean match = false;
            for (String fileName : imageFileNames) {
                String uuid = fileName.substring(0, fileName.length() - 4);
                if (image.getUUID().equals(uuid)) {
                    match = true;
                    break;
                }
            }
            if (!match) {
                try {
                    AgentCommon.getMBeanServer().unregisterMBean(MBeanObjectNamer.makeVMImageName(image.getName()));
                } catch (Exception ex) {
                    logger.error("Failed to unregister VMImage MBean", ex);
                }
                vmList.remove(image);
            }
        }
    }

    void newTemplateFromVMImage(final String vmName, final String vmImageUUID, final String name, final String description) throws VMMException {
        // copy the VM disk to the template directory
        RemoteExec.Result result = null;
        try {
            String command = vmmDomain0HomeDir + "/bin/makeNewVMTemplate " + vmName + " " + vmImageUUID + " \"" + name
                + "\" \"" + description + "\"";
            logger.debug("Creating new VM image from VM " + vmName + "...\n");
            result = RemoteExec.commandAsRoot(storeServer, sshPassword, command);
        } catch (Exception ex) {
            logger.error("SSH failure with host " + storeServer, ex);
        }
        if (result.exitCode != 0) {
            logger.error(result.error);
            throw new VMMException(result.error);
        }
        sync();
    }

    public VirtualMachineImageMXBean lookUpByName(final String name) {
        for (XenVirtualMachineImage image : vmList) {
            if (image.getName().equals(name)) {
                return image;
            }
        }
        return null;
    }

    public VirtualMachineImageMXBean lookUpByUUID(final String uuid) {
        for (XenVirtualMachineImage image : vmList) {
            if (image.getUUID().equals(uuid)) {
                return image;
            }
        }
        return null;
    }

    public List<VirtualMachineImageMXBean> listVMImageTemplates() {
        ArrayList<VirtualMachineImageMXBean> result = new ArrayList<VirtualMachineImageMXBean>(vmList);
        return result;
    }

    public long getCapacityMB() {
        return Utils.getDiskCapacityMB(storeServer, sshPassword, rootDirectory);
    }

    public long getFreeSpaceMB() {
        return Utils.getDiskFreeSpaceMB(storeServer, sshPassword, rootDirectory);
    }

    public String getName() {
        return "DiskImageStore(" + rootDirectory + ")";
    }

    private static class XenVirtualMachineImage extends ManagedResource implements VirtualMachineImageMXBean {
        private String name, uuid, metadata, description, imageFile;

        public static XenVirtualMachineImage newVMImage(final ObjectName objectName, final String metadata) {
            InputStream stream = new ByteArrayInputStream(metadata.getBytes());
            XenVirtualMachineImage vmi = new XenVirtualMachineImage(objectName);
            VMIXMLHandler handler = new VMIXMLHandler(vmi);
            try {
                SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
                parser.parse(stream, handler);
            } catch (Exception ex) {
                return null;
            }
            vmi.metadata = new String(metadata);
            return vmi;
        }

        private XenVirtualMachineImage(final ObjectName objectName) {
            super(objectName);
        }

        public String getMetaData() {
            return metadata;
        }

        public String getName() {
            return name;
        }

        public String getDescription() {
            return description;
        }

        public String getUUID() {
            return uuid;
        }

        public String getImageFile() {
            return imageFile;
        }

        static class VMIXMLHandler extends DefaultHandler {
            XenVirtualMachineImage vmi;

            String currentText;

            VMIXMLHandler(final XenVirtualMachineImage vmi) {
                this.vmi = vmi;
            }

            @Override
            public void startElement(final String uri, final String localName, final String qName, final Attributes attrs) {
                if (qName.equals("virtualMachineImage")) {
                    String name = attrs.getValue("name");
                    String uuid = attrs.getValue("uuid");
                    vmi.name = name;
                    vmi.uuid = uuid;
                }
            }

            @Override
            public void characters(final char[] ch, final int start, final int length) {
                currentText = "";
                for (int i = start; i < start + length; i++) {
                    currentText = currentText + ch[i];
                }
            }

            @Override
            public void endElement(final java.lang.String uri, final java.lang.String localName, final java.lang.String qName)
                throws SAXException {
                if (qName.equals("description")) {
                    vmi.description = currentText;
                } else if (qName.equals("imageFile")) {
                    vmi.imageFile = currentText;
                }
            }

        }

    }

}
