/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenVirtualMachine.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.main.VirtManagerAgent;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.Types;
import com.xensource.xenapi.VIF;
import com.xensource.xenapi.VM;

/**
 * Xen driver Virtual Machine MXBean implementation
 */
public class XenVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(XenVirtualMachine.class);

    private XenHost host;

    private Connection connection;

    private VM vm;

    private String uuid;

    private String name;

    private int domID;

    private String ipAddress;

    private String macAddress;

    private String imageID;

    private Date startTime;

    private float cpuConsumption = 0;

    private PowerState cachedPowerState;

    public XenVirtualMachine(final ObjectName objectName, final XenHost host, final Connection connection, final VM vm, final Map<String, String> userData) {
        super(objectName);
        try {
            this.host = host;
            this.connection = connection;
            this.vm = vm;
            synchronized (connection) {
                uuid = vm.getUuid(connection);
                name = vm.getNameLabel(connection);

                domID = vm.getDomid(connection).intValue();

                if (userData != null) {
                    for (String key : userData.keySet()) {
                        addUserData(key, userData.get(key));
                    }
                    imageID = userData.get("imageID");
                }
                if (imageID == null) {
                    imageID = getUserData("imageID");
                }

            }
        } catch (Exception ex) {
            logger.error("Failedto init VM", ex);
        }
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) {
        XenHost xenHost = null;
        for (HostMXBean h : host.getServerPool().getManagedHosts()) {
            if (h.getObjectName().equals(targetHost.getObjectName())) {
                xenHost = (XenHost) h;
                break;
            }
        }
        return xenHost != null && xenHost != host && getMemorySizeMB() < xenHost.getFreeMemoryMB();
    }

    public String getUuid() {
        return uuid;
    }

    public HostMXBean getHostMBean() {
        return host;
    }

    public long getDomID() {
        return domID;
    }

    public Date getStartTime() {
        if (startTime == null) {
            try {
                startTime = vm.getMetrics(connection).getStartTime(connection);
                Calendar calendar = new GregorianCalendar();
                startTime = new Date(startTime.getTime() + calendar.get(Calendar.DST_OFFSET)
                    + calendar.get(Calendar.ZONE_OFFSET));
            } catch (Exception ex) {
                logger.error("Failed to getStartTime of VM " + name, ex);
            }
        }
        return startTime;
    }

    public long getUpTimeSeconds() {
        return (System.currentTimeMillis() - getStartTime().getTime()) / 1000;
    }

    private long lastCPULoadTime = 0;

    private float lastCPULoad;

    public void updateCPUConsumption(final long time, final float newCPULoad) {
        if (lastCPULoadTime > 0) {
            float val = (time - lastCPULoadTime) * (newCPULoad + lastCPULoad) / (2 * 1000 * 60);
            val *= host.getNumCPU() * ((XenHost) host).getCPUFrequencyMhz() / 1000.0f;
            cpuConsumption += val;
        }
        lastCPULoad = newCPULoad;
        lastCPULoadTime = time;
    }

    public float getCPULoad() {
        // TODO only works if perfmonitor is running
        return lastCPULoad;
    }

    public float[] getLoadPerVCPU() {
        float vals[] = new float[8];
        for (int i = 0; i < vals.length; i++) {
            vals[i] = -1;
        }
        synchronized (connection) {
            try {
                Map<Long, Double> map = vm.getMetrics(connection).getVCPUsUtilisation(connection);
                for (Map.Entry<Long, Double> entry : map.entrySet()) {
                    vals[entry.getKey().intValue()] = (float) (entry.getValue() * 100);
                }
            } catch (Exception ex) {
                logger.error("Failed to get VM metrics", ex);
            }
        }
        return vals;
    }

    public long getMemorySizeMB() {
        synchronized (connection) {
            try {
                // return
                // vm.getMetrics(connection).getMemoryActual(connection).intValue
                // ()/(1024*1024);
                return vm.getMemoryDynamicMax(connection) / (1024 * 1024);
            } catch (Exception ex) {
                logger.error("Failed to get VM MemoryDynamicMax", ex);
            }
        }
        return 0;
    }

    public void setMemorySizeMB(final long size) {
        synchronized (connection) {
            try {
                vm.setMemoryDynamicMax(connection, size * 1014 * 1024);
                vm.setMemoryDynamicMin(connection, size * 1024 * 1024);
            } catch (Exception ex) {
                logger.error("Failed to set VM MemoryDynamicMin/max", ex);
            }
        }
    }

    public long getMemoryUsedMB() {
        synchronized (connection) {
            try {
                return vm.getMetrics(connection).getMemoryActual(connection).intValue() / (1024 * 1024);
            } catch (Exception ex) {
                logger.error("Failed to get memory metrics form VM " + name, ex);
            }
        }
        return 0;
    }

    public String getNameLabel() {
        return name;
    }

    public void addUserData(final String key, final String value) {
        synchronized (connection) {
            try {
                RemoteExec.Result result = null;
                String command = "xenstore-write /vm/" + uuid + "/" + key + " " + value + " 2>&1";
                logger.debug("VM " + name + " addUserData: " + command);
                result = host.remoteExec(command);
                if (result.exitCode != 0) {
                    logger.debug("VM " + name + " addUserData failed: " + result.output);
                    logger.error(command + " returns " + result.exitCode);
                }
            } catch (Exception ex) {
                logger.error("Failed to set add user data to VM " + name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getUserData(java.lang.String
     * )
     */
    public String getUserData(final String key) {
        // TODO this method has a major drawback: it fails
        // in case of live migration
        synchronized (connection) {
            try {
                RemoteExec.Result result = null;
                String command = "xenstore-read /vm/" + uuid + "/" + key;
                result = host.remoteExec(command);
                if (result.exitCode != 0) {
                    // logger.info("VM user data: no key " + key + " on VM " +
                    // name);
                    return "";
                }
                return result.output;
            } catch (Exception ex) {
                logger.error("VM getUserData failure: ", ex);
                return "";
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getNumVCPUs()
     */
    public int getNumVCPUs() {
        synchronized (connection) {
            try {
                return vm.getMetrics(connection).getVCPUsNumber(connection).intValue();
            } catch (Exception ex) {
                logger.error("Failed to get VM metrics", ex);
            }
        }
        return -1;
    }

    public String getPinnedVCPUs() {
        // TODO Auto-generated method stub
        return null;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingCap()
     */
    public int getSchedulingCap() {
        synchronized (connection) {
            try {
                Map<String, String> vcpuParams = vm.getVCPUsParams(connection);
                if (vcpuParams.get("cap") != null) {
                    return Integer.parseInt(vcpuParams.get("cap"));
                }
            } catch (Exception ex) {
                logger.error("Failed to get VM VCPU params", ex);
            }
        }
        return 0;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingWeight()
     */
    public int getSchedulingWeight() {
        synchronized (connection) {
            try {
                Map<String, String> vcpuParams = vm.getVCPUsParams(connection);
                if (vcpuParams.get("weight") != null) {
                    return Integer.parseInt(vcpuParams.get("weight"));
                }
            } catch (Exception ex) {
                logger.error("Failed to get VM VCPU params", ex);
            }
        }
        return 0;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getState()
     */
    public PowerState getState() {
        synchronized (connection) {
            try {
                Types.VmPowerState state = vm.getPowerState(connection);
                if (state == Types.VmPowerState.HALTED) {
                    return (cachedPowerState = PowerState.HALTED);
                } else if (state == Types.VmPowerState.RUNNING) {
                    return (cachedPowerState = PowerState.RUNNING);
                } else if (state == Types.VmPowerState.PAUSED) {
                    return (cachedPowerState = PowerState.PAUSED);
                } else if (state == Types.VmPowerState.SUSPENDED) {
                    return (cachedPowerState = PowerState.SUSPENDED);
                }
                return (cachedPowerState = PowerState.UNKNOWN);
            } catch (Exception ex) {
                // logger.error("Failed to get power state", ex);
            }
        }
        return (cachedPowerState = PowerState.UNKNOWN);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getGuestIpAddress()
     */
    public String getGuestIpAddress() {
        try {
            if (getState() != PowerState.RUNNING) {
                return "unknown";
            }
        } catch (Exception ex) {
        }
        if (ipAddress == null) {
            // wait until the guest OS has booted to acquire its IP address
            int nbAttempts = 10;
            while (nbAttempts-- > 0) {
                ipAddress = host.getIPAddress(getMacAddress());
                if (ipAddress != null && !ipAddress.equals("")) {
                    break;
                }
                try {
                    TimeUnit.SECONDS.sleep(10);
                } catch (InterruptedException ex) {
                }
            }
        }
        return ipAddress == null ? "unknown" : ipAddress;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getMacAddress()
     */
    public String getMacAddress() {
        try {
            if (getState() != PowerState.RUNNING) {
                return "";
            }
        } catch (Exception ex) {
        }
        if (macAddress == null) {
            synchronized (connection) {
                try {
                    Set<VIF> vifs = vm.getVIFs(connection);
                    for (VIF vif : vifs) {
                        String vifName = vif.getDevice(connection);
                        if (vifName.equals("eth0")) {
                            macAddress = vif.getMAC(connection);
                            logger.debug("VM " + name + " Mac address=" + macAddress);
                            break;
                        }
                    }
                } catch (Exception ex) {
                    logger.error("Failed to MAC address of VM " + name, ex);
                }
            }
        }
        if (macAddress == null) {
            logger.error("Unable to get MAC address of VM " + name);
        }
        return macAddress;
    }

    public void setMemory(final int memory) {
        // TODO
        throw new UnsupportedOperationException();
    }

    public void setNumVCPUs(final int numVCPUs) {
        // TODO
        throw new UnsupportedOperationException();
    }

    public void setPinnedVCPUs(final String pinnedVCPUs) {
        // TODO
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingCap(int)
     */
    public void setSchedulingCap(final int schedulingCap) {
        synchronized (connection) {
            try {
                long domID = vm.getDomid(connection);
                RemoteExec.Result result = null;
                String command = "xm sched-credit -d " + domID + " -c " + schedulingCap;
                result = host.remoteExec(command);
                if (result.exitCode != 0) {
                    logger.error(command + " returns " + result.exitCode);
                }
            } catch (Exception ex) {
                logger.error("Failed to set sched cap of VM " + name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingWeight(int)
     */
    public void setSchedulingWeight(final int schedulingWeight) {
        synchronized (connection) {
            try {
                long domID = vm.getDomid(connection);
                RemoteExec.Result result = null;
                String command = "xm sched-credit -d " + domID + " -w " + schedulingWeight;
                result = host.remoteExec(command);
                if (result.exitCode != 0) {
                    logger.error(command + " returns " + result.exitCode);
                }
            } catch (Exception ex) {
                logger.error("Failed to set sched weigth of VM " + name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#suspend()
     */
    public void suspend() {
        try {
            RemoteExec.Result result = null;
            String command = (((XenServerPool) host.getServerPool()).getVMMDomain0HomeDir()) + "/bin/suspendVM "
                + getNameLabel();
            logger.debug("Launching command " + command);
            result = host.remoteExec(command);
            if (result.exitCode != 0) {
                logger.error(command + " failed: " + result.error);
                return;
            }
            logger.debug(result.output);
            onVMStateChanged();
        } catch (Exception ex) {
            logger.error("ssh failure", ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#migrate(org.ow2.jasmine.
     * vmm.api.HostMXBean, boolean)
     */
    public void migrate(HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {
        // TODO cold migration
        if (!live) {
            throw new IllegalOperationException("Not implemented");
        }
        String targetHostName = targetHost.getHostName();
        logger.info("Attempting live migration of VM " + name + " to host " + targetHostName);

        boolean foundHostInServerPool = false;

        for (HostMXBean h : host.getServerPool().getManagedHosts()) {
            if (h.getHostName().equals(targetHost.getHostName())) {
                foundHostInServerPool = true;
                targetHost = h;
                break;
            }
        }

        if (!foundHostInServerPool) {
            throw new IllegalOperationException("Source and target hosts belong to different server pools");
        }

        XenHost targetXenHost = (XenHost) targetHost; // XXXXX
        String imageID = getUserData("imageID"); // XXX

        logger.info("VM " + name + " live migration to host " + targetHostName + "...");
        emitNotification(NotificationTypes.VM_MIGRATION_START, targetHostName, null);

        String command = ((XenServerPool) host.getServerPool()).getVMMDomain0HomeDir() + "/bin/migrateVM " + name + " "
            + targetHostName;
        logger.info("Launching command " + command);
        RemoteExec.Result result = null;
        try {
            result = host.remoteExec(command);
        } catch (Exception ex) {
            logger.error("migrate: ", ex);
            emitNotification(NotificationTypes.VM_MIGRATION_ABORT, targetHostName, null);
            throw new VMMException("Internal error");
        }
        if (result.exitCode != 0) {
            emitNotification(NotificationTypes.VM_MIGRATION_ABORT, targetHostName, null);
            throw new VMMException("Command \"" + command + "\" failed: " + result.output);
        }

        logger.info("VM " + name + " live migration done");

        host.postMigrateVM(this, targetXenHost);
        host = targetXenHost;
        addUserData("imageID", imageID);
        connection = targetXenHost.getXenAPIConnection();
        synchronized (connection) {
            int attempts = 3;
            while (true) {
                try {
                    vm = VM.getByUuid(connection, uuid);
                    break;
                } catch (Exception ex) {
                    attempts -= 1;
                    if (attempts > 0) {
                        try {
                            Thread.sleep(1000);
                        } catch (Exception e) {
                        }
                        continue;
                    } else {
                        logger.error("Cannot get vm XenAPI handle after migration: ", ex);
                        break;
                    }
                }
            }
        }

        emitNotification(NotificationTypes.VM_MIGRATION, targetHostName, uuid);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#shutdown()
     */
    public void shutdown() {
        try {
            RemoteExec.Result result = null;
            String command = (((XenServerPool) host.getServerPool()).getVMMDomain0HomeDir()) + "/bin/shutdownVM "
                + getNameLabel();
            logger.debug("Launching command " + command);
            result = host.remoteExec(command);
            if (result.exitCode != 0) {
                logger.error(command + " failed: " + result.error);
                return;
            }
            logger.debug(result.output);
            onVMStateChanged();
        } catch (Exception ex) {
            logger.error("ssh failure", ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#start()
     */
    public void start() {
        try {
            RemoteExec.Result result = null;
            String command = (((XenServerPool) host.getServerPool()).getVMMDomain0HomeDir()) + "/bin/startVM " + getNameLabel();
            logger.debug("Launching command " + command);
            result = host.remoteExec(command);
            if (result.exitCode != 0) {
                logger.error(command + " failed: " + result.error);
                return;
            }
            logger.debug(result.output);
            onVMStateChanged();
        } catch (Exception ex) {
            logger.error("ssh failure", ex);
        }
    }

    void onVMStateChanged() {
        PowerState state = getState();
        // XXX we ignore the pause state
        if (state != PowerState.PAUSED) {
            emitNotification(NotificationTypes.VM_STATE_CHANGE, state.toString(), null);
            logger.debug("VM " + getNameLabel() + " new power state: " + state.toString());
        }
    }

    /**
     * Restart a VM based on its configuration stored on its host if the VM is
     * managed by Xend, this function is not needed
     */
    void restart() {
        logger.debug("Restarting VM...");
        String command = "xm create " + XenHost.XEN_CONFIG_HOME + "/" + name + ".cfg";
        RemoteExec.Result result = null;
        try {
            logger.debug("Launching command " + command);
            result = host.remoteExec(command);
            // must reset the XenAPI vm handle
            vm = VM.getByUuid(connection, uuid);
        } catch (Exception ex) {
            logger.error("Failed to get UUID of VM " + name, ex);
        }
        if (result.exitCode != 0) {
            logger.error("Failed to restart VM: " + result.error);
        } else {
            logger.debug("Restarting VM done.");
        }
        // clear powerState so that getState fetch real state from Xen server
        cachedPowerState = PowerState.UNKNOWN;

    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#reboot()
     */
    public void reboot() {
        synchronized (connection) {
            try {
                vm.cleanReboot(connection);
                logger.info("VM " + name + " rebooted");
            } catch (Exception ex) {
                logger.error("Failed to reboot VM " + name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#resume()
     */
    public void resume() {
        try {
            RemoteExec.Result result = null;
            String command = (((XenServerPool) host.getServerPool()).getVMMDomain0HomeDir()) + "/bin/resumeVM "
                + getNameLabel();
            logger.debug("Launching command " + command);
            result = host.remoteExec(command);
            if (result.exitCode != 0) {
                logger.error(command + " failed: " + result.error);
                return;
            }
            logger.debug(result.output);
            onVMStateChanged();
        } catch (Exception ex) {
            logger.error("ssh failure", ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#destroy()
     */
    public void destroy() {
        try {
            RemoteExec.Result result = null;
            String command = (((XenServerPool) host.getServerPool()).getVMMDomain0HomeDir()) + "/bin/destroyVM "
                + getNameLabel();
            logger.debug("Launching command " + command);
            result = host.remoteExec(command);
            if (result.exitCode != 0) {
                logger.error(command + " failed: " + result.error);
            } else {
                logger.debug(result.output);
                logger.info("VM " + name + " destroyed");
            }
        } catch (Exception ex) {
            logger.error("ssh failure", ex);
        }
        try {
            host.onVMDestroy(this);
            VirtManagerAgent.getMBeanServer().unregisterMBean(objectName);
        } catch (Exception ex) {
            logger.error("Failed to un register VM MBean", ex);
        }

    }

    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION_ABORT, NotificationTypes.VM_MIGRATION_START, NotificationTypes.VM_MIGRATION,
            NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(), "VM event")};
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#makeTemplate(java.lang.String
     * , java.lang.String, java.lang.String)
     */
    public void makeTemplate(final String vmImageID, final String name, final String description) throws InsufficientResourcesException,
        IllegalOperationException, VMMException {
        // shutdown VM to make sure disk image is a proper state
        shutdown();

        try {
            ((XenVMImageStore) host.getServerPool().getVMImageStore()).newTemplateFromVMImage(getNameLabel(), vmImageID, name,
                description);
        } finally {
            start();
        }

    }
}
