/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DomainBuilder.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.domain;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Stack;

import javax.management.ObjectName;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.driver.Driver;
import org.ow2.jasmine.vmm.agent.driver.DriverFactory;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * The class DomainBuilder constructs Domain objects from XML specifications.
 * TODO check the syntax of the XML file with a schema
 */
public final class DomainBuilder {
    static Logger logger = Logger.getLogger(DomainBuilder.class);

    static class DomainXMLHandler extends DefaultHandler {
        Stack<Domain> domainStack = new Stack<Domain>();

        Domain rootDomain, currentDomain;

        ServerPool currentServerPool;

        StringBuffer currentPath = new StringBuffer();

        @Override
        public void startElement(final String uri, final String localName, final String qName, final Attributes attrs) {
            if (qName.equals("domain")) {
                String name = attrs.getValue("name");
                if (name == null) {
                    logger.error("Missing domain name, skipping domain definition");
                    currentDomain = null;
                    return;
                }
                currentPath.append("/" + name);

                ObjectName domainObjectName = MBeanObjectNamer.makeDomainName(currentPath.toString());
                Domain domain = new Domain(name, domainObjectName);
                try {
                    AgentCommon.getMBeanServer().registerMBean(domain, domainObjectName);
                } catch (Exception ex) {
                    logger.error("Skipping domain " + name, ex);
                    currentDomain = null;
                    return;
                }
                logger.info("Added Domain " + domainObjectName);
                if (domainStack.isEmpty()) {
                    rootDomain = domain;
                } else {
                    domainStack.peek().addSubDomain(domain);
                }
                domainStack.push(domain);
                currentDomain = domain;
            } else if (qName.equals("serverPool")) {
                if (currentDomain != null) {
                    String name = attrs.getValue("name");
                    String driverName = attrs.getValue("driver");
                    if (name == null) {
                        logger.error("skipped server pool because of missing name");
                        return;
                    }
                    if (driverName == null) {
                        logger.error("skipped server pool because of missing driver name");
                        return;
                    }
                    ObjectName serverPoolObjectName = null;
                    String policyId = attrs.getValue("policy");
                    String propFileName = attrs.getValue("properties");
                    Properties props = new Properties();
                    if (propFileName != null) {
                        InputStream stream;
                        try {
                            stream = getClass().getClassLoader().getResourceAsStream(propFileName);
                            if (stream != null) {
                                props.load(stream);
                            } else {
                                logger.error("Cannot load property file " + propFileName);
                                currentServerPool = null;
                                return;
                            }
                        } catch (IOException ex) {
                            logger.error("Cannot load property file " + propFileName, ex);
                            currentServerPool = null;
                            return;
                        }
                    }

                    serverPoolObjectName = MBeanObjectNamer.makeServerPoolName(currentPath.toString() + "/" + name);
                    Driver driver = DriverFactory.getInstance().newDriver(driverName);
                    if (driver != null) {
                        currentServerPool = driver.newServerPool(name, serverPoolObjectName, props);
                        currentServerPool.setProvisioningPolicy(policyId);
                        try {
                            AgentCommon.getMBeanServer().registerMBean(currentServerPool, serverPoolObjectName);
                        } catch (Exception ex) {
                            logger.error("Skipping ServerPool " + name, ex);
                            currentServerPool = null;
                            return;
                        }
                        logger.info("Added ServerPool " + serverPoolObjectName);
                        currentDomain.addServerPool(currentServerPool);
                    } else {
                        logger.error("Driver " + driverName + " not found");
                        currentServerPool = null;
                    }
                }
            } else if (qName.equals("host")) {
                String hostName = attrs.getValue("name");
                if (hostName == null) {
                    logger.error("Missing host name, skipping host definition");
                    return;
                }
                if (currentServerPool != null) {
                    currentServerPool.newHost(hostName, attributes2Map(attrs));
                }
            }
        }

        @Override
        public void endElement(final java.lang.String uri, final java.lang.String localName, final java.lang.String qName) throws SAXException {
            if (qName.equals("domain")) {
                if (currentDomain != null) {
                    domainStack.pop();
                    currentPath.delete(currentPath.lastIndexOf("/"), currentPath.length());
                }
                if (domainStack.isEmpty()) {
                    currentDomain = null;
                } else {
                    currentDomain = domainStack.peek();
                }
            } else if (qName.equals("serverPool")) {
                currentServerPool = null;
            }
        }

    }

    static private Map<String, String> attributes2Map(final Attributes attrs) {
        Map<String, String> map = new HashMap<String, String>();
        for (int i = 0; i < attrs.getLength(); i++) {
            String name = attrs.getLocalName(i);
            String val = attrs.getValue(i);
            map.put(name, val);
        }
        return map;
    }

    /**
     * Constructs a new Domain object from an XML description
     *
     * @param file the file containing of the XML description
     * @return the domain object or null if the domain could not be created.
     */
    public static Domain createManagedResourcesFromXML(final File file) {
        try {
            DomainXMLHandler handler = new DomainXMLHandler();
            SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
            logger.info("Reading resource description file " + file.getAbsolutePath());
            parser.parse(new FileInputStream(file), handler);
            return handler.rootDomain;
        } catch (Exception ex) {
            logger.error("While parsing resource description file:", ex);
            return null;
        }
    }

}
