/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwarePerfMonitor.java 3177 2009-03-20 13:30:34Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.math.BigInteger;
import java.util.HashMap;
import java.util.TimerTask;

import org.apache.log4j.Logger;

import com.vmware.vim.ArrayOfPerfCounterInfo;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.PerfCompositeMetric;
import com.vmware.vim.PerfCounterInfo;
import com.vmware.vim.PerfEntityMetric;
import com.vmware.vim.PerfEntityMetricBase;
import com.vmware.vim.PerfInterval;
import com.vmware.vim.PerfMetricId;
import com.vmware.vim.PerfMetricIntSeries;
import com.vmware.vim.PerfQuerySpec;
import com.vmware.vim.PerfSampleInfo;

/**
 * VMwarePerfMonitor is a periodic task which collect performance metrics of all
 * VMs on a given host and of the host itself
 */
class VMwarePerfMonitor extends TimerTask {
    static private Logger logger = Logger.getLogger(VMwarePerfMonitor.class);

    private VMwareServiceConnection connection;

    private PerfQuerySpec qSpec = null;

    private PerfMetricId[] midList = null;

    private ManagedObjectReference perfManager = null;

    private HashMap<String, String> counterMap = null;

    private VMwareHost host = null;

    public VMwarePerfMonitor(final VMwareServiceConnection connection, final VMwareHost host) {
        this.host = host;
        this.connection = connection;
    }

    public void configure(final String[][] counterNames) {
        try {
            counterMap = new HashMap<String, String>();
            int nbCounters = counterNames.length;

            // getMetaData and Refs...
            if (perfManager == null) {
                perfManager = connection.getServiceContent().getPerfManager();
            }
            ArrayOfPerfCounterInfo cArray = (ArrayOfPerfCounterInfo) connection.getObjectProperty(perfManager, "perfCounter");
            PerfCounterInfo[] counterInfoList = cArray.getPerfCounterInfo();
            midList = new PerfMetricId[nbCounters];
            for (int i = 0; i < midList.length; i++) {
                PerfCounterInfo counterInfo = getCounterInfo(counterInfoList, counterNames[i][0], counterNames[i][1],
                    counterNames[i][2]);
                midList[i] = new PerfMetricId(null, null, counterInfo.getKey(), "");
            }
        } catch (Exception ex) {
            logger.error("Failed to configure VMwarePerfMonitor", ex);
        }
        PerfInterval interval = new PerfInterval();
        interval.setSamplingPeriod(5);
        interval.setLength(3600);
        // com.vmware.vim.UpdatePerfInterval(perfManager, interval); ??

        qSpec = new PerfQuerySpec();
        qSpec.setMetricId(midList);
        qSpec.setIntervalId(new Integer(20)); // sampling period cannot be lower
                                              // than 20 seconds
        qSpec.setMaxSample(1);
        qSpec.setEntity(host.getMOR());
    }

    @Override
    public void run() {
        HashMap<String, BigInteger> hostMetricValues = new HashMap<String, BigInteger>();
        HashMap<String, HashMap<String, BigInteger>> vmMetricValues = new HashMap<String, HashMap<String, BigInteger>>();
        long now = System.currentTimeMillis();
        try {
            logger.debug("Sending periodic perfQuery");
            PerfCompositeMetric cMetric = null;
            synchronized (connection) {
                cMetric = connection.getService().queryPerfComposite(perfManager, qSpec);
            }
            if (cMetric == null) {
                return;
            }
            PerfEntityMetric eMetric = (PerfEntityMetric) cMetric.getEntity();
            PerfMetricIntSeries iSeries = (PerfMetricIntSeries) eMetric.getValue(0);
            PerfEntityMetricBase[] childEntityMetric = cMetric.getChildEntity();
            PerfSampleInfo[] rootInfo = eMetric.getSampleInfo();
            int numSamples = eMetric.getSampleInfo().length;
            // Main result - host level
            for (int i = 0; i < eMetric.getValue().length; i++) {
                PerfMetricIntSeries rawSeries = (PerfMetricIntSeries) eMetric.getValue(i);
                String tName = counterMap.get("" + rawSeries.getId().getCounterId());
                long val = rawSeries.getValue(0);
                hostMetricValues.put(tName, BigInteger.valueOf(val));
            }
            // results for child (aka running VMs)
            if (childEntityMetric != null) {
                for (int i = 0; i < childEntityMetric.length; i++) {
                    PerfEntityMetric cStats = (PerfEntityMetric) childEntityMetric[i];
                    String eName = (String) connection.getObjectProperty(cStats.getEntity(), "name");
                    int numChildSamples = cStats.getSampleInfo().length;
                    HashMap<String, BigInteger> vmResult = new HashMap<String, BigInteger>();
                    vmMetricValues.put(eName, vmResult);
                    for (int k = 0; k < cStats.getValue().length; k++) {
                        PerfMetricIntSeries cSeries = (PerfMetricIntSeries) cStats.getValue(k);
                        String tName = counterMap.get("" + cSeries.getId().getCounterId());
                        long val = cSeries.getValue(cSeries.getValue().length - 1);
                        vmResult.put(tName, BigInteger.valueOf(val));
                    }
                }
            }
            host.updateMetrics(hostMetricValues, vmMetricValues);
        } catch (Exception ex) {
            logger.error("Failed to query perf metrics", ex);
        }
    }

    private PerfCounterInfo getCounterInfo(final PerfCounterInfo[] cInfo, final String groupName, final String counterName, final String rollup) {
        String fullName = new String(groupName + "." + counterName);
        for (int i = 0; i < cInfo.length; i++) {
            PerfCounterInfo current = cInfo[i];
            if (current.getGroupInfo().getKey().equals(groupName) && current.getNameInfo().getKey().equals(counterName)
                && current.getRollupType().getValue().equals(rollup)) {
                // logger.debug("Using counter # " + current.getKey() + " [for "
                // + fullName + "]");
                counterMap.put("" + current.getKey(), fullName);
                return current;
            }
        }
        logger.debug("Warning - unable to find correct PerfCounterInfo for " + groupName + "." + counterName + " - " + rollup);
        return null;
    }

}
