/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenVirtualMachine.java 3229 2009-04-03 13:07:56Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.IllegalOperationException;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.Types;
import com.xensource.xenapi.VIF;
import com.xensource.xenapi.VM;

/**
 * Xen driver Virtual Machine MXBean implementation
 */
public class XenVirtualMachine extends ManagedResource implements VirtualMachineMXBean {
    static Logger logger = Logger.getLogger(XenVirtualMachine.class);

    private XenHost host;

    private Connection connection;

    private VM vm;

    private String uuid;

    private String name;

    private int domID;

    private String ipAddress;

    private String macAddress;

    private String imageID;

    private Date startTime;

    private float cpuConsumption = 0;

    private PowerState cachedPowerState;

    public XenVirtualMachine(final ObjectName objectName, final XenHost host, final Connection connection, final VM vm,
        final Map<String, String> userData) {
        super(objectName);
        try {
            this.host = host;
            this.connection = connection;
            this.vm = vm;
            synchronized (connection) {
                this.uuid = vm.getUuid(connection);
                this.name = vm.getNameLabel(connection);

                this.domID = vm.getDomid(connection).intValue();

                if (userData != null) {
                    for (String key : userData.keySet()) {
                        this.addUserData(key, userData.get(key));
                    }
                    this.imageID = userData.get("imageID");
                }
                if (this.imageID == null) {
                    this.imageID = this.getUserData("imageID");
                }

            }
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("Failedto init VM", ex);
        }
    }

    public boolean canLiveMigrateToHost(final HostMXBean targetHost) {
        XenHost xenHost = null;
        for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
            if (h.getObjectName().equals(targetHost.getObjectName())) {
                xenHost = (XenHost) h;
                break;
            }
        }
        return xenHost != null && xenHost != this.host && this.getMemorySizeMB() < xenHost.getFreeMemoryMB();
    }

    public String getUuid() {
        return this.uuid;
    }

    public HostMXBean getHostMBean() {
        return this.host;
    }

    public long getDomID() {
        return this.domID;
    }

    public Date getStartTime() {
        if (this.startTime == null) {
            try {
                this.startTime = this.vm.getMetrics(this.connection).getStartTime(this.connection);
                Calendar calendar = new GregorianCalendar();
                this.startTime = new Date(this.startTime.getTime() + calendar.get(Calendar.DST_OFFSET)
                    + calendar.get(Calendar.ZONE_OFFSET));
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to getStartTime of VM " + this.name, ex);
            }
        }
        return this.startTime;
    }

    public long getUpTimeSeconds() {
        return (System.currentTimeMillis() - this.getStartTime().getTime()) / 1000;
    }

    private long lastCPULoadTime = 0;

    private float lastCPULoad;

    public void updateCPUConsumption(final long time, final float newCPULoad) {
        if (this.lastCPULoadTime > 0) {
            float val = (time - this.lastCPULoadTime) * (newCPULoad + this.lastCPULoad) / (2 * 1000 * 60);
            val *= this.host.getNumCPU() * (this.host).getCPUFrequencyMhz() / 1000.0f;
            this.cpuConsumption += val;
        }
        this.lastCPULoad = newCPULoad;
        this.lastCPULoadTime = time;
    }

    public float getCPULoad() {
        // TODO only works if perfmonitor is running
        return this.lastCPULoad;
    }

    public float[] getLoadPerVCPU() {
        float vals[] = new float[8];
        for (int i = 0; i < vals.length; i++) {
            vals[i] = -1;
        }
        synchronized (this.connection) {
            try {
                Map<Long, Double> map = this.vm.getMetrics(this.connection).getVCPUsUtilisation(this.connection);
                for (Map.Entry<Long, Double> entry : map.entrySet()) {
                    vals[entry.getKey().intValue()] = (float) (entry.getValue() * 100);
                }
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to get VM metrics", ex);
            }
        }
        return vals;
    }

    public long getMemorySizeMB() {
        synchronized (this.connection) {
            try {
                // return
                // vm.getMetrics(connection).getMemoryActual(connection).intValue
                // ()/(1024*1024);
                return this.vm.getMemoryDynamicMax(this.connection) / (1024 * 1024);
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to get VM MemoryDynamicMax", ex);
            }
        }
        return 0;
    }

    public void setMemorySizeMB(final long size) {
        synchronized (this.connection) {
            try {
                this.vm.setMemoryDynamicMax(this.connection, size * 1014 * 1024);
                this.vm.setMemoryDynamicMin(this.connection, size * 1024 * 1024);
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to set VM MemoryDynamicMin/max", ex);
            }
        }
    }

    public long getMemoryUsedMB() {
        synchronized (this.connection) {
            try {
                return this.vm.getMetrics(this.connection).getMemoryActual(this.connection).intValue() / (1024 * 1024);
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to get memory metrics form VM " + this.name, ex);
            }
        }
        return 0;
    }

    public String getNameLabel() {
        return this.name;
    }

    public void addUserData(final String key, final String value) {
        synchronized (this.connection) {
            try {
                RemoteExec.Result result = null;
                String command = "xenstore-write /vm/" + this.uuid + "/" + key + " " + value + " 2>&1";
                XenVirtualMachine.logger.debug("VM " + this.name + " addUserData: " + command);
                result = RemoteExec.commandAsRoot(this.host.getHostName(), this.host.getSshAuthInfo(), command);
                if (result.exitCode != 0) {
                    XenVirtualMachine.logger.debug("VM " + this.name + " addUserData failed: " + result.output);
                    XenVirtualMachine.logger.error(command + " returns " + result.exitCode);
                }
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to set add user data to VM " + this.name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getUserData(java.lang.String
     * )
     */
    public String getUserData(final String key) {
        // TODO this method has a major drawback: it fails
        // in case of live migration
        synchronized (this.connection) {
            try {
                RemoteExec.Result result = null;
                String command = "xenstore-read /vm/" + this.uuid + "/" + key;
                result = RemoteExec.commandAsRoot(this.host.getHostName(), this.host.getSshAuthInfo(), command);
                if (result.exitCode != 0) {
                    // logger.info("VM user data: no key " + key + " on VM " +
                    // name);
                    return "";
                }
                return result.output;
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("VM getUserData failure: ", ex);
                return "";
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getNumVCPUs()
     */
    public int getNumVCPUs() {
        synchronized (this.connection) {
            try {
                return this.vm.getMetrics(this.connection).getVCPUsNumber(this.connection).intValue();
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to get VM metrics", ex);
            }
        }
        return -1;
    }

    public String getPinnedVCPUs() {
        // TODO Auto-generated method stub
        return null;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingCap()
     */
    public int getSchedulingCap() {
        synchronized (this.connection) {
            try {
                Map<String, String> vcpuParams = this.vm.getVCPUsParams(this.connection);
                if (vcpuParams.get("cap") != null) {
                    return Integer.parseInt(vcpuParams.get("cap"));
                }
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to get VM VCPU params", ex);
            }
        }
        return 0;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getSchedulingWeight()
     */
    public int getSchedulingWeight() {
        synchronized (this.connection) {
            try {
                Map<String, String> vcpuParams = this.vm.getVCPUsParams(this.connection);
                if (vcpuParams.get("weight") != null) {
                    return Integer.parseInt(vcpuParams.get("weight"));
                }
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to get VM VCPU params", ex);
            }
        }
        return 0;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getState()
     */
    public PowerState getState() {
        synchronized (this.connection) {
            try {
                Types.VmPowerState state = this.vm.getPowerState(this.connection);
                if (state == Types.VmPowerState.HALTED) {
                    return (this.cachedPowerState = PowerState.HALTED);
                } else if (state == Types.VmPowerState.RUNNING) {
                    return (this.cachedPowerState = PowerState.RUNNING);
                } else if (state == Types.VmPowerState.PAUSED) {
                    return (this.cachedPowerState = PowerState.PAUSED);
                } else if (state == Types.VmPowerState.SUSPENDED) {
                    return (this.cachedPowerState = PowerState.SUSPENDED);
                }
                return (this.cachedPowerState = PowerState.UNKNOWN);
            } catch (Exception ex) {
                // logger.error("Failed to get power state", ex);
            }
        }
        return (this.cachedPowerState = PowerState.UNKNOWN);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getGuestIpAddress()
     */
    public String getGuestIpAddress() {
        try {
            if (this.getState() != PowerState.RUNNING) {
                return "unknown";
            }
        } catch (Exception ex) {
        }
        if (this.ipAddress == null) {
            // wait until the guest OS has booted to acquire its IP address
            int nbAttempts = 10;
            while (nbAttempts-- > 0) {
                this.ipAddress = this.host.getIPAddress(this.getMacAddress());
                if (this.ipAddress != null && !this.ipAddress.equals("")) {
                    break;
                }
                try {
                    TimeUnit.SECONDS.sleep(10);
                } catch (InterruptedException ex) {
                }
            }
        }
        return this.ipAddress == null ? "unknown" : this.ipAddress;
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#getMacAddress()
     */
    public String getMacAddress() {
        try {
            if (this.getState() != PowerState.RUNNING) {
                return "";
            }
        } catch (Exception ex) {
        }
        if (this.macAddress == null) {
            synchronized (this.connection) {
                try {
                    Set<VIF> vifs = this.vm.getVIFs(this.connection);
                    for (VIF vif : vifs) {
                        String vifName = vif.getDevice(this.connection);
                        if (vifName.equals("eth0")) {
                            this.macAddress = vif.getMAC(this.connection);
                            XenVirtualMachine.logger.debug("VM " + this.name + " Mac address=" + this.macAddress);
                            break;
                        }
                    }
                } catch (Exception ex) {
                    XenVirtualMachine.logger.error("Failed to MAC address of VM " + this.name, ex);
                }
            }
        }
        if (this.macAddress == null) {
            XenVirtualMachine.logger.error("Unable to get MAC address of VM " + this.name);
        }
        return this.macAddress;
    }

    public void setMemory(final int memory) {
        // TODO
        throw new UnsupportedOperationException();
    }

    public void setNumVCPUs(final int numVCPUs) {
        // TODO
        throw new UnsupportedOperationException();
    }

    public void setPinnedVCPUs(final String pinnedVCPUs) {
        // TODO
        throw new UnsupportedOperationException();
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingCap(int)
     */
    public void setSchedulingCap(final int schedulingCap) {
        synchronized (this.connection) {
            try {
                long domID = this.vm.getDomid(this.connection);
                RemoteExec.Result result = null;
                String command = "xm sched-credit -d " + domID + " -c " + schedulingCap;
                result = RemoteExec.commandAsRoot(this.host.getHostName(), this.host.getSshAuthInfo(), command);
                if (result.exitCode != 0) {
                    XenVirtualMachine.logger.error(command + " returns " + result.exitCode);
                }
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to set sched cap of VM " + this.name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#setSchedulingWeight(int)
     */
    public void setSchedulingWeight(final int schedulingWeight) {
        synchronized (this.connection) {
            try {
                long domID = this.vm.getDomid(this.connection);
                RemoteExec.Result result = null;
                String command = "xm sched-credit -d " + domID + " -w " + schedulingWeight;
                result = RemoteExec.commandAsRoot(this.host.getHostName(), this.host.getSshAuthInfo(), command);
                if (result.exitCode != 0) {
                    XenVirtualMachine.logger.error(command + " returns " + result.exitCode);
                }
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to set sched weigth of VM " + this.name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#suspend()
     */
    public void suspend() {
        try {
            RemoteExec.Result result = XenSSH.suspendVM(this.host, this.getNameLabel());
            if (result.exitCode != 0) {
                XenVirtualMachine.logger.error("Command suspendVM failed: " + result.error);
                return;
            }
            XenVirtualMachine.logger.debug(result.output);
            this.onVMStateChanged();
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("ssh failure", ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#migrate(org.ow2.jasmine.
     * vmm.api.HostMXBean, boolean)
     */
    public void migrate(HostMXBean targetHost, final boolean live) throws IllegalOperationException, VMMException {
        // TODO cold migration
        if (!live) {
            throw new IllegalOperationException("Not implemented");
        }
        String targetHostName = targetHost.getHostName();
        XenVirtualMachine.logger.info("Attempting live migration of VM " + this.name + " to host " + targetHostName);

        boolean foundHostInServerPool = false;

        for (HostMXBean h : this.host.getServerPool().getManagedHosts()) {
            if (h.getHostName().equals(targetHost.getHostName())) {
                foundHostInServerPool = true;
                targetHost = h;
                break;
            }
        }

        if (!foundHostInServerPool) {
            throw new IllegalOperationException("Source and target hosts belong to different server pools");
        }

        XenHost targetXenHost = (XenHost) targetHost; // XXXXX
        String imageID = this.getUserData("imageID"); // XXX

        XenVirtualMachine.logger.info("VM " + this.name + " live migration to host " + targetHostName + "...");
        this.emitNotification(NotificationTypes.VM_MIGRATION_START, targetHostName, null);

        RemoteExec.Result result;
        try {
            result = XenSSH.migrateVM(this.host, this.name, targetHostName);
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("migrate: ", ex);
            this.emitNotification(NotificationTypes.VM_MIGRATION_ABORT, targetHostName, null);
            throw new VMMException("MigrateVM: SSH connection error", ex);
        }
        if (result.exitCode != 0) {
            this.emitNotification(NotificationTypes.VM_MIGRATION_ABORT, targetHostName, null);
            throw new VMMException("Command migrateVM failed: " + result.error);
        }

        try {
            result = XenSSH.postMigrateVM(this.host, this.name, targetHostName);
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("PostMigrateVM: ", ex);
        }
        if (result.exitCode != 0) {
            XenVirtualMachine.logger.error("PostMigrateVM: " + result.error);
        }

        XenVirtualMachine.logger.info("VM " + this.name + " live migration done");

        this.host.postMigrateVM(this, targetXenHost);
        this.host = targetXenHost;
        this.addUserData("imageID", imageID);
        this.connection = targetXenHost.getXenAPIConnection();
        synchronized (this.connection) {
            int attempts = 3;
            while (true) {
                try {
                    this.vm = VM.getByUuid(this.connection, this.uuid);
                    break;
                } catch (Exception ex) {
                    attempts -= 1;
                    if (attempts > 0) {
                        try {
                            Thread.sleep(1000);
                        } catch (Exception e) {
                        }
                        continue;
                    } else {
                        XenVirtualMachine.logger.error("Cannot get VM XenAPI handle after migration: ", ex);
                        break;
                    }
                }
            }
        }

        this.emitNotification(NotificationTypes.VM_MIGRATION, targetHostName, this.uuid);
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#shutdown()
     */
    public void shutdown() {
        try {
            RemoteExec.Result result = XenSSH.shutdownVM(this.host, this.getNameLabel());
            if (result.exitCode != 0) {
                XenVirtualMachine.logger.error("Command shutdownVM failed: " + result.error);
                return;
            }
            XenVirtualMachine.logger.debug(result.output);
            this.onVMStateChanged();
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("ssh failure", ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#start()
     */
    public void start() {
        try {
            RemoteExec.Result result = XenSSH.startVM(this.host, this.getNameLabel());
            if (result.exitCode != 0) {
                XenVirtualMachine.logger.error("Command start failed: " + result.error);
                return;
            }
            XenVirtualMachine.logger.debug(result.output);
            this.onVMStateChanged();
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("ssh failure", ex);
        }
    }

    void onVMStateChanged() {
        PowerState state = this.getState();
        // XXX we ignore the pause state
        if (state != PowerState.PAUSED) {
            this.emitNotification(NotificationTypes.VM_STATE_CHANGE, state.toString(), null);
            XenVirtualMachine.logger.debug("VM " + this.getNameLabel() + " new power state: " + state.toString());
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#reboot()
     */
    public void reboot() {
        synchronized (this.connection) {
            try {
                this.vm.cleanReboot(this.connection);
                XenVirtualMachine.logger.info("VM " + this.name + " rebooted");
            } catch (Exception ex) {
                XenVirtualMachine.logger.error("Failed to reboot VM " + this.name, ex);
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#resume()
     */
    public void resume() {
        try {
            RemoteExec.Result result = XenSSH.resumeVM(this.host, this.getNameLabel());
            if (result.exitCode != 0) {
                XenVirtualMachine.logger.error("ResumeVM failed: " + result.error);
                return;
            }
            XenVirtualMachine.logger.debug(result.output);
            this.onVMStateChanged();
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("ssh failure", ex);
        }
    }

    /*
     * (non-Javadoc)
     * @see org.ow2.jasmine.vmm.api.VirtualMachineMXBean#destroy()
     */
    public void destroy() {
        try {
            RemoteExec.Result result = XenSSH.destroyVM(this.host, this.getNameLabel());
            if (result.exitCode != 0) {
                XenVirtualMachine.logger.error("DestroyVM failed: " + result.error);
            } else {
                XenVirtualMachine.logger.debug(result.output);
                XenVirtualMachine.logger.info("VM " + this.name + " destroyed");
            }
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("ssh failure", ex);
        }
        try {
            this.host.onVMDestroy(this);
            AgentCommon.getMBeanServer().unregisterMBean(this.objectName);
        } catch (Exception ex) {
            XenVirtualMachine.logger.error("Failed to unregister VM MBean", ex);
        }

    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_MIGRATION_ABORT, NotificationTypes.VM_MIGRATION_START, NotificationTypes.VM_MIGRATION,
            NotificationTypes.LOG, NotificationTypes.ERROR}, Notification.class.getName(), "VM event")};
    }

    /*
     * (non-Javadoc)
     * @see
     * org.ow2.jasmine.vmm.api.VirtualMachineMXBean#makeTemplate(java.lang.String
     * , java.lang.String, java.lang.String)
     */
    public void makeTemplate(final String vmImageID, final String name, final String description)
        throws InsufficientResourcesException, IllegalOperationException, VMMException {
        // shutdown VM to make sure disk image is a proper state
        this.shutdown();

        try {
            ((XenVMImageStore) this.host.getServerPool().getVMImageStore()).newTemplateFromVMImage(this.getNameLabel(),
                vmImageID, name, description);
        } finally {
            this.start();
        }

    }
}
