/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareEventCollector.java 3236 2009-04-03 13:21:32Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.util.Calendar;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.vmware.vim.Event;
import com.vmware.vim.EventFilterSpec;
import com.vmware.vim.EventFilterSpecByEntity;
import com.vmware.vim.EventFilterSpecByTime;
import com.vmware.vim.EventFilterSpecRecursionOption;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.VmBeingHotMigratedEvent;
import com.vmware.vim.VmConnectedEvent;
import com.vmware.vim.VmCreatedEvent;
import com.vmware.vim.VmEvent;
import com.vmware.vim.VmMigratedEvent;
import com.vmware.vim.VmPoweredOffEvent;
import com.vmware.vim.VmPoweredOnEvent;
import com.vmware.vim.VmRemovedEvent;
import com.vmware.vim.VmSuspendedEvent;

/**
 * The VMwareEventCollector class collects event on a periodic basis and
 * notifies VMwareHosts accordingly. Events of interest include all VM lifecycle
 * events
 */
class VMwareEventCollector {
    static Logger logger = Logger.getLogger(VMwareEventCollector.class);

    private static int PERIOD_MS = 15 * 1000;

    private VMwareServiceConnection connection;

    private VMwareServerPool pool;

    private String mountPointPath;

    public VMwareEventCollector(final VMwareServerPool pool, final VMwareServiceConnection connection,
        final String mountPointPath) {
        this.pool = pool;
        this.connection = connection;
        this.mountPointPath = mountPointPath;
    }

    private ManagedObjectReference createCollector() throws Exception {
        this.connection.connect();
        ManagedObjectReference meRef = this.connection.getService().findByInventoryPath(
            this.connection.getServiceContent().getSearchIndex(), this.mountPointPath);

        EventFilterSpecByEntity entitySpec = new EventFilterSpecByEntity();
        entitySpec.setEntity(meRef);

        entitySpec.setRecursion(EventFilterSpecRecursionOption.all);
        final EventFilterSpec eFSpec = new EventFilterSpec();
        eFSpec.setEntity(entitySpec);
        EventFilterSpecByTime filterSpecByTime = new EventFilterSpecByTime();
        filterSpecByTime.setBeginTime(Calendar.getInstance());
        eFSpec.setTime(filterSpecByTime);
        return this.connection.getService().createCollectorForEvents(this.connection.getServiceContent().getEventManager(),
            eFSpec);
    }

    public void start() {
        try {
            this.pool.getExecutorService().execute(new Runnable() {
                public void run() {
                    ManagedObjectReference eHistoryCollect = null;
                    final EventFilterSpec eFSpec = new EventFilterSpec();
                    try {
                        VMwareEventCollector.this.connection.connect();
                        ManagedObjectReference meRef = VMwareEventCollector.this.connection.getService().findByInventoryPath(
                            VMwareEventCollector.this.connection.getServiceContent().getSearchIndex(),
                            VMwareEventCollector.this.mountPointPath);

                        EventFilterSpecByEntity entitySpec = new EventFilterSpecByEntity();
                        entitySpec.setEntity(meRef);

                        entitySpec.setRecursion(EventFilterSpecRecursionOption.all);

                        eFSpec.setEntity(entitySpec);
                        EventFilterSpecByTime filterSpecByTime = new EventFilterSpecByTime();
                        filterSpecByTime.setBeginTime(Calendar.getInstance());
                        eFSpec.setTime(filterSpecByTime);
                    } catch (Exception ex) {
                        VMwareEventCollector.logger.error("Failed to start event collector", ex);
                        return;
                    }

                    while (true) {
                        try {
                            if (eHistoryCollect == null) {
                                eHistoryCollect = VMwareEventCollector.this.createCollector();
                            }
                            ((org.apache.axis.client.Stub) VMwareEventCollector.this.connection.getService())
                                .setTimeout(60000 * 60 * 10);

                            Event[] eventCollection = VMwareEventCollector.this.connection.getService().readNextEvents(
                                eHistoryCollect, 100);

                            // Event[] eventCollection =
                            // connection.getService().queryEvents(
                            // connection.getServiceContent().getEventManager(),
                            // eFSpec);

                            if (eventCollection != null) {
                                for (Event event : eventCollection) {
                                    long eventTime = event.getCreatedTime().getTimeInMillis();
                                    if (!(event instanceof VmEvent)) {
                                        continue;
                                    }
                                    if (((VmEvent) event).isTemplate()) {
                                        VMwareEventCollector.logger.debug("Received event for VM template "
                                            + event.getVm().getName());
                                        ((VMwareServerPool.VMwareVMTemplateDataStore) VMwareEventCollector.this.pool
                                            .getVMImageStore()).sync();
                                        continue;
                                    }
                                    if (event instanceof VmPoweredOffEvent) {
                                        VMwareHost host = VMwareEventCollector.this.pool.getHostByName(event.getHost()
                                            .getName());
                                        if (host != null) {
                                            host.onPowerStateChangedVM(event.getVm().getVm(), event.getVm().getName(),
                                                VirtualMachineMXBean.PowerState.HALTED, eventTime);
                                        }
                                    } else if (event instanceof VmPoweredOnEvent) {
                                        VMwareHost host = VMwareEventCollector.this.pool.getHostByName(event.getHost()
                                            .getName());
                                        if (host != null) {
                                            host.onPowerStateChangedVM(event.getVm().getVm(), event.getVm().getName(),
                                                VirtualMachineMXBean.PowerState.RUNNING, eventTime);
                                        }
                                    } else if (event instanceof VmSuspendedEvent) {
                                        VMwareHost host = VMwareEventCollector.this.pool.getHostByName(event.getHost()
                                            .getName());
                                        if (host != null) {
                                            host.onPowerStateChangedVM(event.getVm().getVm(), event.getVm().getName(),
                                                VirtualMachineMXBean.PowerState.SUSPENDED, eventTime);
                                        }
                                    } else if (event instanceof VmCreatedEvent) {
                                        VMwareHost host = VMwareEventCollector.this.pool.getHostByName(event.getHost()
                                            .getName());
                                        VMwareEventCollector.logger.debug("CreatedEvent " + event.getVm().getName());
                                        // if (host != null)
                                        // host.onCreatedVM(event.getVm().getVm(),
                                        // event.getVm().getName());
                                    } else if (event instanceof VmConnectedEvent) {
                                        VMwareHost host = VMwareEventCollector.this.pool.getHostByName(event.getHost()
                                            .getName());
                                        VMwareEventCollector.logger.debug("ConnectedEvent " + event.getVm().getName());
                                        ((VMwareServerPool.VMwareVMTemplateDataStore) VMwareEventCollector.this.pool
                                            .getVMImageStore()).sync();
                                        // if (host != null)
                                        // host.onCreatedVM(event.getVm().getVm(),
                                        // event.getVm().getName());
                                    } else if (event instanceof VmRemovedEvent) {
                                        VMwareHost host = VMwareEventCollector.this.pool.getHostByName(event.getHost()
                                            .getName());
                                        // if (host != null)
                                        // host.onRemovedVM(event.getVm().getVm(),
                                        // event.getVm().getName());
                                    } else if (event instanceof VmBeingHotMigratedEvent) {
                                        VmBeingHotMigratedEvent beingMigratedEvent = (VmBeingHotMigratedEvent) event;
                                        // TODO
                                    } else if (event instanceof VmMigratedEvent) {
                                        VmMigratedEvent migratedEvent = (VmMigratedEvent) event;
                                        // TODO
                                    }

                                }
                            }
                            Thread.sleep(VMwareEventCollector.PERIOD_MS);
                        } catch (Exception ex) {
                            VMwareEventCollector.logger.error("In VMware event collector: ", ex);
                        }
                    }
                }
            });

        } catch (Exception ex) {
            VMwareEventCollector.logger.error("Failed to initialize event collector", ex);
        }

    }
}
