/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMwareServiceConnection.java 3233 2009-04-03 13:17:04Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.vmware;

import java.lang.reflect.Method;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.vmware.vim.DynamicProperty;
import com.vmware.vim.LocalizedMethodFault;
import com.vmware.vim.ManagedObjectReference;
import com.vmware.vim.ObjectContent;
import com.vmware.vim.ObjectSpec;
import com.vmware.vim.ObjectUpdate;
import com.vmware.vim.ObjectUpdateKind;
import com.vmware.vim.PropertyChange;
import com.vmware.vim.PropertyChangeOp;
import com.vmware.vim.PropertyFilterSpec;
import com.vmware.vim.PropertyFilterUpdate;
import com.vmware.vim.PropertySpec;
import com.vmware.vim.RuntimeFault;
import com.vmware.vim.SelectionSpec;
import com.vmware.vim.ServiceContent;
import com.vmware.vim.TaskInfo;
import com.vmware.vim.TaskInfoState;
import com.vmware.vim.TraversalSpec;
import com.vmware.vim.UpdateSet;
import com.vmware.vim.VimPortType;
import com.vmware.vim.VimServiceLocator;

/**
 * the VMwareServiceConnection class represents an active connection to a VMware
 * virtual center management server and provides various utility methods
 */
class VMwareServiceConnection {
    private VimServiceLocator locator;

    private String hostName, username, password;

    private VimPortType service;

    private ServiceContent sic;

    private ManagedObjectReference serviceInstanceRef;

    private boolean isLeased;

    public VMwareServiceConnection(final String hostName, final String username, final String password) {
        this.username = username;
        this.password = password;
        this.hostName = hostName;
    }

    public synchronized void connect() throws Exception {
        this.locator = new VimServiceLocator();
        this.locator.setMaintainSession(true);
        this.service = this.locator.getVimPort(new URL("https://" + this.hostName + "/sdk"));
        this.serviceInstanceRef = new ManagedObjectReference();
        this.serviceInstanceRef.setType("ServiceInstance");
        this.serviceInstanceRef.set_value("ServiceInstance");
        this.sic = this.service.retrieveServiceContent(this.serviceInstanceRef);
        this.service.login(this.sic.getSessionManager(), this.username, this.password, null);
    }

    synchronized boolean lease() {
        if (!this.isLeased) {
            this.isLeased = true;
            return true;
        }
        return false;
    }

    public synchronized void release() {
        this.isLeased = false;
    }

    public synchronized void close() throws Exception {
        if (this.service != null) {
            this.service.logout(this.sic.getSessionManager());
            this.service = null;
            this.sic = null;
        }
    }

    public VimPortType getService() {
        return this.service;
    }

    public ServiceContent getServiceContent() {
        return this.sic;
    }

    public ManagedObjectReference getServiceInstanceRef() {
        return this.serviceInstanceRef;
    }

    public synchronized ObjectContent[] getObjectProperties(final ManagedObjectReference collector,
        final ManagedObjectReference mobj, final String[] properties) throws Exception {
        if (mobj == null) {
            return null;
        }
        ManagedObjectReference usecoll = collector;
        if (usecoll == null) {
            usecoll = this.sic.getPropertyCollector();
        }
        PropertyFilterSpec spec = new PropertyFilterSpec();
        spec.setPropSet(new PropertySpec[] {new PropertySpec()});
        spec.getPropSet(0).setAll(new Boolean(properties == null || properties.length == 0));
        spec.getPropSet(0).setType(mobj.getType());
        spec.getPropSet(0).setPathSet(properties);
        spec.setObjectSet(new ObjectSpec[] {new ObjectSpec()});
        spec.getObjectSet(0).setObj(mobj);
        spec.getObjectSet(0).setSkip(Boolean.FALSE);
        return this.service.retrieveProperties(usecoll, new PropertyFilterSpec[] {spec});
    }

    public synchronized ArrayList getDecendentMoRefs(final ManagedObjectReference root, final String type) throws Exception {
        ArrayList mors = this.getDecendentMoRefs(root, type, null);
        return mors;
    }

    public synchronized ArrayList getDecendentMoRefs(final ManagedObjectReference root, final String type,
        final String[][] filter) throws Exception {
        String[][] typeinfo = new String[][] {new String[] {type, "name"},};
        ObjectContent[] ocary = this.getContentsRecursively(null, root, typeinfo, true);
        ArrayList refs = new ArrayList();
        if (ocary == null || ocary.length == 0) {
            return refs;
        }
        for (ObjectContent element : ocary) {
            refs.add(element.getObj());
        }
        if (filter != null) {
            ArrayList filtermors = this.filterMOR(refs, filter);
            return filtermors;
        } else {
            return refs;
        }
    }

    private ArrayList filterMOR(final ArrayList mors, final String[][] filter) throws Exception {
        ArrayList filteredmors = new ArrayList();
        for (int i = 0; i < mors.size(); i++) {
            boolean flag = true;
            String guest = null;
            for (int k = 0; k < filter.length; k++) {
                String prop = filter[k][0];
                String reqVal = filter[k][1];
                String value = this.getProp(((ManagedObjectReference) mors.get(i)), prop);
                if (reqVal == null) {
                    continue;

                }
                if (value == null && reqVal == null) {
                    continue;
                }
                if (value == null && reqVal != null) {
                    flag = false;
                    k = filter.length + 1;
                } else if (value.equalsIgnoreCase(reqVal)) {
                } else {
                    flag = false;
                    k = filter.length + 1;
                }
            }
            if (flag) {
                filteredmors.add(mors.get(i));
            }
        }
        return filteredmors;
    }

    private String getProp(final ManagedObjectReference obj, final String prop) {
        String propVal = null;
        try {
            propVal = (String) this.getDynamicProperty(obj, prop);
        } catch (Exception e) {
        }
        return propVal;
    }

    public Object getDynamicProperty(final ManagedObjectReference mor, final String propertyName) throws Exception {
        ObjectContent[] objContent = this.getObjectProperties(null, mor, new String[] {propertyName});

        Object propertyValue = null;
        if (objContent != null) {
            DynamicProperty[] dynamicProperty = objContent[0].getPropSet();
            if (dynamicProperty != null) {
                /*
                 * Check the dynamic propery for ArrayOfXXX object
                 */
                Object dynamicPropertyVal = dynamicProperty[0].getVal();
                String dynamicPropertyName = dynamicPropertyVal.getClass().getName();
                if (dynamicPropertyName.indexOf("ArrayOf") != -1) {
                    String methodName = dynamicPropertyName.substring(dynamicPropertyName.indexOf("ArrayOf")
                        + "ArrayOf".length(), dynamicPropertyName.length());
                    /*
                     * If object is ArrayOfXXX object, then get the XXX[] by
                     * invoking getXXX() on the object. For Ex:
                     * ArrayOfManagedObjectReference.getManagedObjectReference()
                     * returns ManagedObjectReference[] array.
                     */
                    if (this.methodExists(dynamicPropertyVal, "get" + methodName, null)) {
                        methodName = "get" + methodName;
                    } else {
                        /*
                         * Construct methodName for ArrayOf primitive types Ex:
                         * For ArrayOfInt, methodName is get_int
                         */
                        methodName = "get_" + methodName.toLowerCase();
                    }
                    Method getMorMethod = dynamicPropertyVal.getClass().getDeclaredMethod(methodName, (Class[]) null);
                    propertyValue = getMorMethod.invoke(dynamicPropertyVal, (Object[]) null);
                } else if (dynamicPropertyVal.getClass().isArray()) {
                    /*
                     * Handle the case of an unwrapped array being deserialized.
                     */
                    propertyValue = dynamicPropertyVal;
                } else {
                    propertyValue = dynamicPropertyVal;
                }
            }
        }
        return propertyValue;
    }

    boolean methodExists(final Object obj, final String methodName, final Class[] parameterTypes) {
        boolean exists = false;
        try {
            Method method = obj.getClass().getMethod(methodName, parameterTypes);
            if (method != null) {
                exists = true;
            }
        } catch (Exception e) {
        }
        return exists;
    }

    public synchronized ManagedObjectReference getDecendentMoRef(final ManagedObjectReference root, final String type,
        final String name) throws Exception {
        if (name == null || name.length() == 0) {
            return null;
        }

        String[][] typeinfo = new String[][] {new String[] {type, "name",},};

        ObjectContent[] ocary = this.getContentsRecursively(null, root, typeinfo, true);

        if (ocary == null || ocary.length == 0) {
            return null;
        }

        ObjectContent oc = null;
        ManagedObjectReference mor = null;
        DynamicProperty[] propary = null;
        String propval = null;
        boolean found = false;
        for (int oci = 0; oci < ocary.length && !found; oci++) {
            oc = ocary[oci];
            mor = oc.getObj();
            propary = oc.getPropSet();

            propval = null;
            if (type == null || this.typeIsA(type, mor.getType())) {
                if (propary.length > 0) {
                    propval = (String) propary[0].getVal();
                }
                found = propval != null && name.equals(propval);
            }
        }
        if (!found) {
            mor = null;
        }
        return mor;
    }

    public synchronized ObjectContent[] getContentsRecursively(final ManagedObjectReference collector,
        final ManagedObjectReference root, final String[][] typeinfo, final boolean recurse) throws Exception {
        if (typeinfo == null || typeinfo.length == 0) {
            return null;
        }
        ManagedObjectReference usecoll = collector;
        if (usecoll == null) {
            usecoll = this.sic.getPropertyCollector();
        }
        ManagedObjectReference useroot = root;
        if (useroot == null) {
            useroot = this.sic.getRootFolder();
        }
        SelectionSpec[] selectionSpecs = null;
        if (recurse) {
            selectionSpecs = this.buildFullTraversal();
        }
        PropertySpec[] propspecary = this.buildPropertySpecArray(typeinfo);
        PropertyFilterSpec spec = new PropertyFilterSpec(null, null, propspecary, new ObjectSpec[] {new ObjectSpec(null, null,
            useroot, Boolean.FALSE, selectionSpecs)});
        ObjectContent[] retoc = this.service.retrieveProperties(usecoll, new PropertyFilterSpec[] {spec});
        return retoc;
    }

    public synchronized SelectionSpec[] buildFullTraversal() {

        // Recurse through all ResourcePools
        TraversalSpec rpToRp = new TraversalSpec(null, null, null, "ResourcePool", "resourcePool", Boolean.FALSE,
            new SelectionSpec[] {new SelectionSpec(null, null, "rpToRp"), new SelectionSpec(null, null, "rpToVm")});
        rpToRp.setName("rpToRp");

        // Recurse through all ResourcePools
        TraversalSpec rpToVm = new TraversalSpec(null, null, null, "ResourcePool", "vm", Boolean.FALSE, new SelectionSpec[] {});
        rpToVm.setName("rpToVm");

        // Traversal through ResourcePool branch
        TraversalSpec crToRp = new TraversalSpec(null, null, null, "ComputeResource", "resourcePool", Boolean.FALSE,
            new SelectionSpec[] {new SelectionSpec(null, null, "rpToRp"), new SelectionSpec(null, null, "rpToVm")});
        crToRp.setName("crToRp");

        // Traversal through host branch
        TraversalSpec crToH = new TraversalSpec(null, null, null, "ComputeResource", "host", Boolean.FALSE,
            new SelectionSpec[] {});
        crToH.setName("crToH");
        // Traversal through hostFolder branch
        TraversalSpec dcToHf = new TraversalSpec(null, null, null, "Datacenter", "hostFolder", Boolean.FALSE,
            new SelectionSpec[] {new SelectionSpec(null, null, "visitFolders")});
        dcToHf.setName("dcToHf");

        // Traversal through vmFolder branch
        TraversalSpec dcToVmf = new TraversalSpec(null, null, null, "Datacenter", "vmFolder", Boolean.FALSE,
            new SelectionSpec[] {new SelectionSpec(null, null, "visitFolders")});
        dcToVmf.setName("dcToVmf");

        // Recurse through all Hosts
        TraversalSpec HToVm = new TraversalSpec(null, null, null, "HostSystem", "vm", Boolean.FALSE,
            new SelectionSpec[] {new SelectionSpec(null, null, "visitFolders")});
        HToVm.setName("HToVm");

        // Recurse thriugh the folders
        TraversalSpec visitFolders = new TraversalSpec(null, null, null, "Folder", "childEntity", Boolean.FALSE,
            new SelectionSpec[] {new SelectionSpec(null, null, "visitFolders"), new SelectionSpec(null, null, "dcToHf"),
                new SelectionSpec(null, null, "dcToVmf"), new SelectionSpec(null, null, "crToH"),
                new SelectionSpec(null, null, "crToRp"), new SelectionSpec(null, null, "HToVm"),
                new SelectionSpec(null, null, "rpToVm"),});
        visitFolders.setName("visitFolders");
        return new SelectionSpec[] {visitFolders, dcToVmf, dcToHf, crToH, crToRp, rpToRp, HToVm, rpToVm};
    }

    public synchronized PropertySpec[] buildPropertySpecArray(final String[][] typeinfo) {
        // Eliminate duplicates
        HashMap tInfo = new HashMap();
        for (int ti = 0; ti < typeinfo.length; ++ti) {
            Set props = (Set) tInfo.get(typeinfo[ti][0]);
            if (props == null) {
                props = new HashSet();
                tInfo.put(typeinfo[ti][0], props);
            }
            boolean typeSkipped = false;
            for (int pi = 0; pi < typeinfo[ti].length; ++pi) {
                String prop = typeinfo[ti][pi];
                if (typeSkipped) {
                    props.add(prop);
                } else {
                    typeSkipped = true;
                }
            }
        }

        // Create PropertySpecs
        ArrayList pSpecs = new ArrayList();
        for (Iterator ki = tInfo.keySet().iterator(); ki.hasNext();) {
            String type = (String) ki.next();
            PropertySpec pSpec = new PropertySpec();
            Set props = (Set) tInfo.get(type);
            pSpec.setType(type);
            pSpec.setAll(props.isEmpty() ? Boolean.TRUE : Boolean.FALSE);
            pSpec.setPathSet(new String[props.size()]);
            int index = 0;
            for (Iterator pi = props.iterator(); pi.hasNext();) {
                String prop = (String) pi.next();
                pSpec.setPathSet(index++, prop);
            }
            pSpecs.add(pSpec);
        }

        return (PropertySpec[]) pSpecs.toArray(new PropertySpec[0]);
    }

    static String[] meTree = {"ManagedEntity", "ComputeResource", "ClusterComputeResource", "Datacenter", "Folder",
        "HostSystem", "ResourcePool", "VirtualMachine"};

    static String[] crTree = {"ComputeResource", "ClusterComputeResource"};

    static String[] hcTree = {"HistoryCollector", "EventHistoryCollector", "TaskHistoryCollector"};

    boolean typeIsA(final String searchType, final String foundType) {
        if (searchType.equals(foundType)) {
            return true;
        } else if (searchType.equals("ManagedEntity")) {
            for (int i = 0; i < VMwareServiceConnection.meTree.length; ++i) {
                if (VMwareServiceConnection.meTree[i].equals(foundType)) {
                    return true;
                }
            }
        } else if (searchType.equals("ComputeResource")) {
            for (int i = 0; i < VMwareServiceConnection.crTree.length; ++i) {
                if (VMwareServiceConnection.crTree[i].equals(foundType)) {
                    return true;
                }
            }
        } else if (searchType.equals("HistoryCollector")) {
            for (int i = 0; i < VMwareServiceConnection.hcTree.length; ++i) {
                if (VMwareServiceConnection.hcTree[i].equals(foundType)) {
                    return true;
                }
            }
        }
        return false;
    }

    public String waitForTask(final ManagedObjectReference taskmor) throws Exception {
        Object[] result = this.waitForValues(taskmor, new String[] {"info.state", "info.error"}, new String[] {"state"},
            new Object[][] {new Object[] {TaskInfoState.success, TaskInfoState.error}});
        if (result[0].equals(TaskInfoState.success)) {
            return "success";
        } else {
            TaskInfo tinfo = (TaskInfo) this.getDynamicProperty(taskmor, "info");
            LocalizedMethodFault fault = tinfo.getError();
            String error = "Error Occured";
            if (fault != null) {
                error = fault.getFault().getFaultReason();
                System.out.println("Fault " + fault.getFault().getFaultCode());
                System.out.println("Message " + fault.getLocalizedMessage());
            }
            return error;
        }
    }

    public Object[] waitForValues(final ManagedObjectReference objmor, final String[] filterProps, final String[] endWaitProps,
        final Object[][] expectedVals) throws Exception {
        // version string is initially null
        String version = "";
        Object[] endVals = new Object[endWaitProps.length];
        Object[] filterVals = new Object[filterProps.length];

        PropertyFilterSpec spec = new PropertyFilterSpec();
        spec.setObjectSet(new ObjectSpec[] {new ObjectSpec()});
        spec.getObjectSet(0).setObj(objmor);

        spec.setPropSet(new PropertySpec[] {new PropertySpec()});
        spec.getPropSet(0).setPathSet(filterProps);
        spec.getPropSet(0).setType(objmor.getType());

        spec.getObjectSet(0).setSelectSet(null);
        spec.getObjectSet(0).setSkip(Boolean.FALSE);

        ManagedObjectReference filterSpecRef = this.service.createFilter(this.sic.getPropertyCollector(), spec, true);

        boolean reached = false;

        UpdateSet updateset = null;
        PropertyFilterUpdate[] filtupary = null;
        PropertyFilterUpdate filtup = null;
        ObjectUpdate[] objupary = null;
        ObjectUpdate objup = null;
        PropertyChange[] propchgary = null;
        PropertyChange propchg = null;
        while (!reached) {
            boolean retry = true;
            while (retry) {
                try {
                    updateset = this.service.waitForUpdates(this.sic.getPropertyCollector(), version);
                    retry = false;
                } catch (Exception e) {
                    if (e instanceof org.apache.axis.AxisFault) {
                        org.apache.axis.AxisFault fault = (org.apache.axis.AxisFault) e;
                        org.w3c.dom.Element[] errors = fault.getFaultDetails();
                        String faultString = fault.getFaultString();
                        if (faultString.indexOf("java.net.SocketTimeoutException") != -1) {
                            System.out.println("Retrying2........");
                            retry = true;
                        } else {
                            throw e;
                        }
                    }
                }
            }
            version = updateset.getVersion();
            if (updateset == null || updateset.getFilterSet() == null) {
                continue;
            }
            // Make this code more general purpose when PropCol changes later.
            filtupary = updateset.getFilterSet();
            filtup = null;
            for (PropertyFilterUpdate element : filtupary) {
                filtup = element;
                objupary = filtup.getObjectSet();
                objup = null;
                propchgary = null;
                for (ObjectUpdate element2 : objupary) {
                    objup = element2;
                    // TODO: Handle all "kind"s of updates.
                    if (objup.getKind() == ObjectUpdateKind.modify || objup.getKind() == ObjectUpdateKind.enter
                        || objup.getKind() == ObjectUpdateKind.leave) {
                        propchgary = objup.getChangeSet();
                        for (PropertyChange element3 : propchgary) {
                            propchg = element3;
                            this.updateValues(endWaitProps, endVals, propchg);
                            this.updateValues(filterProps, filterVals, propchg);
                        }
                    }
                }
            }
            Object expctdval = null;
            // Check if the expected values have been reached and exit the loop
            // if done.
            // Also exit the WaitForUpdates loop if this is the case.
            for (int chgi = 0; chgi < endVals.length && !reached; chgi++) {
                for (int vali = 0; vali < expectedVals[chgi].length && !reached; vali++) {
                    expctdval = expectedVals[chgi][vali];
                    reached = expctdval.equals(endVals[chgi]) || reached;
                }
            }
        }

        // Destroy the filter when we are done.
        this.service.destroyPropertyFilter(filterSpecRef);

        return filterVals;
    }

    protected void updateValues(final String[] props, final Object[] vals, final PropertyChange propchg) {
        for (int findi = 0; findi < props.length; findi++) {
            if (propchg.getName().lastIndexOf(props[findi]) >= 0) {
                if (propchg.getOp() == PropertyChangeOp.remove) {
                    vals[findi] = "";
                } else {
                    vals[findi] = propchg.getVal();
                    // System.out.println("Changed value : " +
                    // propchg.getVal());
                }
            }
        }
    }

    Object getObjectProperty(final ManagedObjectReference moRef, final String propertyName) throws RuntimeFault,
        RemoteException {
        return this.getProperties(moRef, new String[] {propertyName})[0];
    }

    Object[] getProperties(final ManagedObjectReference moRef, final String[] properties) throws RuntimeFault, RemoteException {
        // PropertySpec specifies what properties to
        // retrieve and from type of Managed Object
        PropertySpec pSpec = new PropertySpec();
        pSpec.setType(moRef.getType());
        pSpec.setPathSet(properties);

        // ObjectSpec specifies the starting object and
        // any TraversalSpecs used to specify other objects
        // for consideration
        ObjectSpec oSpec = new ObjectSpec();
        oSpec.setObj(moRef);

        // PropertyFilterSpec is used to hold the ObjectSpec and
        // PropertySpec for the call
        PropertyFilterSpec pfSpec = new PropertyFilterSpec();
        pfSpec.setPropSet(new PropertySpec[] {pSpec});
        pfSpec.setObjectSet(new ObjectSpec[] {oSpec});

        // retrieveProperties() returns the properties
        // selected from the PropertyFilterSpec
        ObjectContent[] ocs = this.service.retrieveProperties(this.sic.getPropertyCollector(),
            new PropertyFilterSpec[] {pfSpec});

        // Return value, one object for each property specified
        Object[] ret = new Object[properties.length];

        if (ocs != null) {
            for (int i = 0; i < ocs.length; ++i) {
                ObjectContent oc = ocs[i];
                DynamicProperty[] dps = oc.getPropSet();
                if (dps != null) {
                    for (int j = 0; j < dps.length; ++j) {
                        DynamicProperty dp = dps[j];
                        // find property path index
                        for (int p = 0; p < ret.length; ++p) {
                            if (properties[p].equals(dp.getName())) {
                                ret[p] = dp.getVal();
                            }
                        }
                    }
                }
            }
        }
        return ret;
    }

}
