/**
 * JASMINe VMMapi: JASMINe Virtual Machine Management API
 * Copyright (C) 2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XenHost.java 3294 2009-04-07 10:47:06Z dangtran $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.vmm.agent.driver.xen;

import java.io.BufferedReader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.management.MBeanNotificationInfo;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.ow2.jasmine.vmm.agent.domain.ManagedResource;
import org.ow2.jasmine.vmm.agent.driver.util.RemoteExec;
import org.ow2.jasmine.vmm.agent.driver.xen.XenVMImageStore.XenVirtualMachineImage;
import org.ow2.jasmine.vmm.agent.jmx.MBeanObjectNamer;
import org.ow2.jasmine.vmm.agent.main.AgentCommon;
import org.ow2.jasmine.vmm.api.HostMXBean;
import org.ow2.jasmine.vmm.api.InsufficientResourcesException;
import org.ow2.jasmine.vmm.api.InvalidVMConfigException;
import org.ow2.jasmine.vmm.api.NotificationTypes;
import org.ow2.jasmine.vmm.api.ServerPoolMXBean;
import org.ow2.jasmine.vmm.api.VMConfigSpec;
import org.ow2.jasmine.vmm.api.VMMException;
import org.ow2.jasmine.vmm.api.VirtualMachineImageStoreMXBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean;

import com.xensource.xenapi.Connection;
import com.xensource.xenapi.HostCpu;
import com.xensource.xenapi.VM;

/**
 * Xen driver Host MXBean implementation
 */
public class XenHost extends ManagedResource implements HostMXBean, NotificationEmitter {
    static Logger logger = Logger.getLogger(XenHost.class);

    static String XEN_CONFIG_HOME = "/etc/xen/auto";

    // the thread pool size determines the maximum number of
    // simultaneous VM creation on the same host
    private static final int MAX_PARALELL_VM_CREATION_PER_HOST = 3;

    // one extra thread for the event reader
    private static final int THREADPOOL_SIZE = 1 + XenHost.MAX_PARALELL_VM_CREATION_PER_HOST;

    private static ExecutorService executorService = Executors.newFixedThreadPool(XenHost.THREADPOOL_SIZE);

    private List<XenVirtualMachine> vmList = new CopyOnWriteArrayList<XenVirtualMachine>();

    private XenServerPool serverPool;

    private Connection connection;

    private Timer perfTimer;

    private com.xensource.xenapi.Host host;

    private String hostName;

    private int numCPU = -1;

    private float aggregatedCPULoad = -1;

    private long cpuFrequencyMhz = -1;

    private RemoteExec.SshAuthInfo authInfo;

    public static class XenAPIParams {
        private final String xenAPIPort;

        private final String xenAPIUser;

        private final String xenAPIPassword;

        public XenAPIParams(final String xenAPIPort, final String xenAPIUser, final String xenAPIPassword) {
            this.xenAPIPort = xenAPIPort;
            this.xenAPIUser = xenAPIUser;
            this.xenAPIPassword = xenAPIPassword;
        }

        public String getXenAPIPort() {
            return this.xenAPIPort;
        }

        public String getXenAPIUser() {
            return this.xenAPIUser;
        }

        public String getXenAPIPassword() {
            return this.xenAPIPassword;
        }

    }

    public static XenHost newHost(final XenServerPool resourcePool, final ObjectName objectName, final String hostName,
        final XenAPIParams xenAPIParams, final RemoteExec.SshAuthInfo authInfo) {

        try {
            String xenAPIURL = "http://" + hostName + ":" + xenAPIParams.getXenAPIPort();
            Connection connection = new Connection(xenAPIURL, xenAPIParams.getXenAPIUser(), xenAPIParams.getXenAPIPassword());
            Connection connection2 = new Connection(xenAPIURL, xenAPIParams.getXenAPIUser(), xenAPIParams.getXenAPIPassword());
            XenHost xenHost = new XenHost(resourcePool, connection, connection2, objectName, hostName, authInfo);
            return xenHost;
        } catch (Exception ex) {
            XenHost.logger.error("Failed to establish Xen-API connection with host " + hostName + " with login "
                + xenAPIParams.getXenAPIUser(), ex);
            return null;
        }

    }

    private XenHost(final XenServerPool serverPool, final Connection connection, final Connection connection2,
        final ObjectName objectName, final String hostName, final RemoteExec.SshAuthInfo authInfo) {
        super(objectName);
        this.serverPool = serverPool;
        this.connection = connection;
        this.hostName = hostName;
        this.authInfo = authInfo;

        try {
            Set<com.xensource.xenapi.Host> hosts = com.xensource.xenapi.Host.getAll(connection);
            this.host = hosts.iterator().next();
            this.populate();
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + hostName, ex);
        }

        // executorService.execute(new Runnable() {
        // public void run() {
        // try {
        // Set<String> eventClasses = new HashSet<String>();
        // Event.register(connection2, eventClasses);
        // while (true) {
        // Set<Event.Record> events = Event.next(connection2);
        // for (Event.Record ev : events) {
        // if (ev.clazz.equals("VM")) {
        // XenVirtualMachine xenVM = null;
        // for (XenVirtualMachine vm : vmList)
        // if (vm.getUuid().equals(ev.objUuid)) {
        // xenVM = vm;
        // }
        // if (xenVM != null) {
        // if (ev.operation == Types.EventOperation.MOD)
        // xenVM.onVMStateChanged();
        // else if (ev.operation == Types.EventOperation.ADD) {
        //
        // } else if (ev.operation == Types.EventOperation.DEL) {
        //
        // }
        // }
        // }
        // }
        // }
        // } catch (Exception ex) {
        // logger.debug("Host " + XenHost.this.hostName, ex);
        // }
        // }
        // });

    }

    RemoteExec.SshAuthInfo getSshAuthInfo() {
        return this.authInfo;
    }

    Connection getXenAPIConnection() {
        return this.connection;
    }

    private void addVM(final XenVirtualMachine vm) {
        this.vmList.add(vm);
    }

    private void removeVM(final XenVirtualMachine vm) {
        this.vmList.remove(vm);
    }

    public ServerPoolMXBean getServerPool() {
        return this.serverPool;
    }

    public void postMigrateVM(final XenVirtualMachine vm, final XenHost newHost) {
        this.removeVM(vm);
        newHost.addVM(vm);
    }

    public void onVMDestroy(final XenVirtualMachine vm) {
        this.removeVM(vm);
        this.emitNotification(NotificationTypes.VM_DEL, "Destroyed", vm.getObjectName());
    }

    private Set<PerfMetric> currentMonitoredMetrics;

    private long currentMonitoringPeriod;

    public void configurePerfMonitor(final Set<PerfMetric> metricsOfInterest, long periodMillis) {
        this.stopPerfMonitor();
        if (periodMillis < 1000) {
            periodMillis = 1000;
        }
        if (metricsOfInterest.contains(PerfMetric.VM_CPU_LOAD)) {
            this.currentMonitoringPeriod = periodMillis;
            this.currentMonitoredMetrics = metricsOfInterest;
            this.startPerfMonitor();
        }
    }

    String getIPAddress(String macAddress) {
        macAddress = macAddress.toUpperCase();

        XenHost.logger.debug("Determining IP address from MAC address " + macAddress + " ...");

        RemoteExec.Result result = null;
        String command = "getIPfromMAC.sh " + macAddress;

        XenHost.logger.debug("Launching command: " + command);

        try {
            result = RemoteExec.commandAsRoot(this.hostName, this.authInfo, command);
        } catch (RemoteExec.SshException ex) {
            XenHost.logger.debug("SSH failure", ex);
            return null;
        }
        if (result.exitCode != 0) {
            XenHost.logger.error("Cannot get IP from MAC " + result.output);
            return null;
        }

        String ipAddress = result.output;
        if (ipAddress != null && ipAddress.length() > 0 && ipAddress.endsWith("\n")) {
            ipAddress = ipAddress.substring(0, ipAddress.length() - 1);
        }
        XenHost.logger.debug("Mac-to-IP " + macAddress + " -> " + ipAddress);
        return ipAddress;
    }

    private synchronized void startPerfMonitor() {
        if (this.perfTimer == null) {
            this.perfTimer = new Timer();
            this.perfTimer.schedule(new PerfTimerTask(), 0, this.currentMonitoringPeriod);
            XenHost.logger.info("Host " + this.hostName + ": started periodic performance reporting task");
        }
    }

    private synchronized void stopPerfMonitor() {
        if (this.perfTimer != null) {
            this.perfTimer.cancel();
            XenHost.logger.info("Host " + this.hostName + ": stopped periodic performance reporting task");
        }
        this.perfTimer = null;
    }

    @Override
    public ObjectName getObjectName() {
        return this.objectName;
    }

    HashMap<String, String> hypervisorInfo;

    // TODO
    public Map<String, String> getHypervisorInfo() {
        if (this.hypervisorInfo == null) {
            this.hypervisorInfo = new HashMap<String, String>();
            this.hypervisorInfo.put("vendor", "");
            this.hypervisorInfo.put("name", "Xen");
            this.hypervisorInfo.put("version", "3.2");
        }
        return this.hypervisorInfo;
    }

    HashMap<String, String> cpuInfo;

    public Map<String, String> getCPUInfo() {
        if (this.cpuInfo == null) {
            this.cpuInfo = new HashMap<String, String>();
            synchronized (this.connection) {
                try {
                    HostCpu[] cpus = this.host.getHostCPUs(this.connection).toArray(new HostCpu[0]);
                    this.numCPU = cpus.length;
                    HostCpu.Record rec = cpus[0].getRecord(this.connection);
                    this.cpuInfo.put("model", rec.modelname);
                    this.cpuFrequencyMhz = rec.speed;
                    this.cpuInfo.put("speed", Long.toString(rec.speed));
                    this.cpuInfo.put("vendor", rec.vendor);
                    // cpuInfo.put("features",rec.features);
                    // cpuInfo.put("flags",rec.flags);
                } catch (Exception ex) {
                    XenHost.logger.debug("Host " + this.hostName, ex);
                }
            }
        }
        return this.cpuInfo;
    }

    public long getCPUFrequencyMhz() {
        if (this.cpuFrequencyMhz == -1) {
            this.getCPUInfo();
        }
        return this.cpuFrequencyMhz;

    }

    public int getNumCPU() {
        if (this.numCPU == -1) {
            this.getCPUInfo();
        }
        return this.numCPU;
    }

    public String getHostName() {
        return this.hostName;
    }

    public VirtualMachineImageStoreMXBean getVMImageStore() {
        return this.serverPool.getVMImageStore();
    }

    public long getFreeMemoryMB() {
        RemoteExec.Result result = null;
        String command = "xm info | grep ^free_memory | awk '{print $3}'";
        try {
            result = RemoteExec.commandAsRoot(this.hostName, this.authInfo, command);
        } catch (RemoteExec.SshException ex) {
            XenHost.logger.error("SSH exception: ", ex);
        }
        if (result.exitCode != 0) {
            XenHost.logger.error("getFreeMemoryMB: SSH " + command + " failed");
            return 0;
        }
        if (result.output.endsWith("\n")) {
            result.output = result.output.substring(0, result.output.length() - 1);
        }
        long freeMem = Long.parseLong(result.output);

        for (VirtualMachineMXBean vm : this.vmList) {
            XenVirtualMachine xvm = (XenVirtualMachine) vm;
            if (xvm.getDomID() == 0) {
                freeMem += xvm.getMemorySizeMB() - (196 + 350); // XXX
            }
        }

        return freeMem;
    }

    private void populate() {
        try {
            synchronized (this.connection) {
                Set<VM> vms = VM.getAll(this.connection);
                for (VM vm : vms) {
                    ObjectName name;
                    if (vm.getDomid(this.connection) == 0) {
                        // for Domain 0, we use the host UUID
                        name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/"
                            + vm.getNameLabel(this.connection), this.host.getUuid(this.connection));
                    } else {
                        name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/"
                            + vm.getNameLabel(this.connection), vm.getUuid(this.connection));
                    }

                    XenVirtualMachine xenVM = new XenVirtualMachine(name, this, this.connection, vm, null);
                    this.addVM(xenVM);
                    AgentCommon.getMBeanServer().registerMBean(xenVM, name);
                    XenHost.logger.info("Added XenVirtualMachineMBean " + name);
                }
            }
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + this.hostName, ex);
        }
    }

    public List<VirtualMachineMXBean> getResidentVMs() {
        List<VirtualMachineMXBean> result = new ArrayList<VirtualMachineMXBean>(this.vmList);
        return result;
    }

    public VirtualMachineMXBean createVM(final VMConfigSpec vmSpecIn, final boolean sync) throws InvalidVMConfigException,
        VMMException {
        final VMConfigSpec vmSpec = new VMConfigSpec(vmSpecIn);
        XenHost.logger.info("Creating new VM [name=" + vmSpec.getName() + ",memorySizeMB=" + vmSpec.getMemorySizeMB()
            + ",diskSize=" + vmSpec.getDiskSizeMB() + ",numVCPU=" + vmSpec.getNumVCPU() + "]");

        final XenVirtualMachineImage template = (XenVirtualMachineImage) this.serverPool.getVMImageStore().lookUpByUUID(
            vmSpec.getVmImageUUID());
        if (template == null) {
            throw new InvalidVMConfigException("Invalid VM Image UUID");
        }
        // XXX we make the vm label unique by appending the current time
        final String vmName = vmSpec.getName() + "-" + System.currentTimeMillis();

        if (!sync) {
            XenHost.executorService.execute(new Runnable() {
                public void run() {
                    try {
                        XenHost.this.createVM2(vmSpec, vmName, template);
                    } catch (Exception ex) {
                    }
                }
            });
            return null;
        } else {
            return this.createVM2(vmSpec, vmName, template);
        }
    }

    private VirtualMachineMXBean createVM2(final VMConfigSpec vmSpec, final String vmName, final XenVirtualMachineImage template)
        throws InvalidVMConfigException, VMMException {

        RemoteExec.Result result;
        try {
            result = XenSSH.createVM(this, vmName, vmSpec, template.getFileName());
        } catch (RemoteExec.SshException ex) {
            XenHost.logger.error("SSH failed: ", ex);
            throw new VMMException("VM creation failed: SSH connection failure", ex);
        }
        if (result.exitCode != 0) {
            XenHost.logger.error("Command createVM failed: " + result.error);
            // TODO parse error message
            throw new VMMException("VM creation failed: " + result.error);
        }

        VM newVM = null;
        String vmUUID = "";
        synchronized (this.connection) {
            try {
                newVM = VM.getByNameLabel(this.connection, vmName).iterator().next();
                vmUUID = newVM.getUuid(this.connection);
            } catch (Exception ex) {
                XenHost.logger.error("Failed to get VM uuid: ", ex);
            }
        }
        ObjectName name = null;
        XenVirtualMachine xenVM = null;
        try {
            name = MBeanObjectNamer.makeVirtualMachineName(this.serverPool.getPath() + "/" + vmName, vmUUID);
            HashMap<String, String> map = new HashMap<String, String>();
            map.put("imageID", vmSpec.getVmImageUUID());
            xenVM = new XenVirtualMachine(name, this, this.connection, newVM, map);
            this.addVM(xenVM);

            AgentCommon.getMBeanServer().registerMBean(xenVM, name);
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + this.hostName, ex);
            throw new VMMException(ex);
        }

        this.emitNotification(NotificationTypes.VM_ADD, "Created", name);

        return xenVM;
    }

    public VirtualMachineMXBean createVM(final String name, final long memorySizeMB, final int numVCPU, final int diskSizeMB,
        final String vmImageUUID, final boolean sync) throws InsufficientResourcesException, InvalidVMConfigException,
        VMMException {
        final VMConfigSpec vmSpec2 = new VMConfigSpec(name, memorySizeMB, numVCPU, diskSizeMB, vmImageUUID);

        return this.createVM(vmSpec2, sync);
    }

    public long getTotalMemoryMB() {
        synchronized (this.connection) {
            try {
                return this.host.getMetrics(this.connection).getMemoryTotal(this.connection) / (1024 * 1024);
            } catch (Exception ex) {
                XenHost.logger.debug("Host " + this.hostName, ex);
            }
        }
        return 0;
    }

    public float[] getLoadPerCPU() {
        float[] result = new float[0];
        synchronized (this.connection) {
            try {
                Set<HostCpu> cpus = this.host.getHostCPUs(this.connection);
                result = new float[cpus.size()];
                int i = 0;
                for (HostCpu cpu : cpus) {
                    result[i++] = (float) (cpu.getUtilisation(this.connection) * 100);
                }
            } catch (Exception ex) {
                XenHost.logger.debug("Host " + this.hostName, ex);
            }
        }
        return result;
    }

    public float getCPULoad() {
        return this.aggregatedCPULoad;
    }

    @Override
    public MBeanNotificationInfo[] getNotificationInfo() {
        return new MBeanNotificationInfo[] {new MBeanNotificationInfo(new String[] {NotificationTypes.VM_STATE_CHANGE,
            NotificationTypes.VM_ADD, NotificationTypes.VM_DEL, NotificationTypes.ERROR, NotificationTypes.LOG},
            Notification.class.getName(), "Host event")};
    }

    public Map<String, Float> getVMCPULoads() {
        HashMap<String, Float> cpuLoads = new HashMap<String, Float>();

        RemoteExec.Result result = null;
        String command = "xentop -b -i 2 -d 1";
        try {
            result = RemoteExec.commandAsRoot(XenHost.this.hostName, XenHost.this.authInfo, command);
        } catch (RemoteExec.SshException ex) {
            return cpuLoads;
        }

        try {
            BufferedReader reader = new BufferedReader(new StringReader(result.output));
            String s;
            int matched = 0;
            while (true) {
                s = reader.readLine();
                if (s == null) {
                    break;
                }
                if (s.indexOf("NAME") != -1) {
                    matched++;
                    if (matched == 2) {
                        break;
                    }
                }
            }
            // skip header line
            // reader.readLine();
            float acc = 0;
            while ((s = reader.readLine()) != null) {
                StringTokenizer st = new StringTokenizer(s);
                String vmLabel = st.nextToken();
                st.nextToken(); // state
                st.nextToken(); // CPU (sec)
                String cpuLoadString = st.nextToken();
                st.nextToken(); // MEM(k)
                st.nextToken(); // MEM (%)
                String t = st.nextToken(); // MAXMEM (k)
                if (t.equals("no")) {
                    st.nextToken(); // limit
                }
                st.nextToken(); // MAXMEM (%)
                String vcpusString = st.nextToken();
                float cpuLoad = Float.parseFloat(cpuLoadString) / 100;
                cpuLoad /= this.getNumCPU(); // Integer.parseInt(vcpusString);
                cpuLoads.put(vmLabel, cpuLoad);
            }
        } catch (Exception ex) {
            XenHost.logger.debug("Host " + this.hostName, ex);
        }
        return cpuLoads;
    }

    private long lastCPULoadSampleTime = 0;

    private class PerfTimerTask extends TimerTask {
        @Override
        public void run() {
            HashMap<String, Float> cpuLoads = new HashMap<String, Float>();
            RemoteExec.Result result = null;
            // two iterations of xentop, we read the second one...
            String command = "xentop -b -i 2 -d 1";
            try {
                result = RemoteExec.commandAsRoot(XenHost.this.hostName, XenHost.this.authInfo, command);
            } catch (RemoteExec.SshException ex) {
                XenHost.logger.error("Failed to invoke xentop", ex);
                return;
            }
            long now = System.currentTimeMillis();
            try {
                BufferedReader reader = new BufferedReader(new StringReader(result.output));
                String s;
                int matched = 0;
                while (true) {
                    s = reader.readLine();
                    if (s == null) {
                        break;
                    }
                    if (s.indexOf("NAME") != -1) {
                        matched++;
                        if (matched == 2) {
                            break;
                        }
                    }
                }
                // skip header line
                // reader.readLine();
                float acc = 0;
                boolean sampleCPU = (now - XenHost.this.lastCPULoadSampleTime) > 20 * 1000;
                while ((s = reader.readLine()) != null) {
                    StringTokenizer st = new StringTokenizer(s);
                    String vmLabel = st.nextToken();
                    st.nextToken(); // state
                    st.nextToken(); // CPU (sec)
                    String cpuLoadString = st.nextToken();
                    st.nextToken(); // MEM(k)
                    st.nextToken(); // MEM (%)
                    String t = st.nextToken(); // MAXMEM (k)
                    if (t.equals("no")) {
                        st.nextToken(); // limit
                    }
                    st.nextToken(); // MAXMEM (%)
                    String vcpusString = st.nextToken();
                    float cpuLoad = Float.parseFloat(cpuLoadString) / 100;
                    cpuLoad /= XenHost.this.getNumCPU();

                    cpuLoads.put(vmLabel, cpuLoad);

                    if (sampleCPU) {
                        for (VirtualMachineMXBean vm : XenHost.this.vmList) {
                            XenVirtualMachine xvm = (XenVirtualMachine) vm;
                            if (xvm.getNameLabel().equals(vmLabel)) {
                                xvm.updateCPUConsumption(now, cpuLoad);
                            }
                        }
                    }

                    acc += cpuLoad;
                }
                XenHost.this.aggregatedCPULoad = acc;
                if (sampleCPU) {
                    XenHost.this.lastCPULoadSampleTime = now;
                }
            } catch (Exception ex) {
                XenHost.logger.debug("Host " + XenHost.this.hostName, ex);
            }
            XenHost.this.emitNotification(NotificationTypes.PERF_REPORT, "CPU load", cpuLoads);
        }
    }
}
